/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence.btreeimpl.btreestorage;

import java.io.*;
import java.text.*;
import java.util.*;

import org.netbeans.mdr.persistence.*;

/** Files stored in the FileCache must start with this header; it contains
* information used by the logging system.  The client of the cache
* should assume that the first 64 bytes of the file are reserved (64
* to allow for growth). All files stored in the cache are committed
* together, and should always have identical file headers.
*/
public class FileHeader {

    /** A random number generated when this set of files is created */
    long fileId;

    private static final int TIMESTAMP_OFFSET = 8;

    /** A timestamp associated with the last comitted transaction
    * for this set of files */
    long timeStamp;

    /** The size of the header */
    public static final int HEADER_SIZE = 64;

    /** create a new file header
    */
    FileHeader() {
        fileId = new Random(new Object().hashCode()).nextLong();
        updateTime();
    }

    /** Write the file header to a file.
    * @param file the file to which the header is written
    * @exception StorageException I/O error writing to the file
    */
    void write(RandomAccessFile file) throws StorageException {
        try {
            byte buffer[] = new byte[HEADER_SIZE];
            write(buffer);
            file.seek(0);
            file.write(buffer);
        }
        catch (IOException ex) {
            throw new StorageIOException(ex);
        }
    }

    /** Write the file header to a byte array
    * @param buffer the byte array to write the header to.
    */
    void write(byte buffer[]) {
        int offset = 0;
        offset = Converter.writeLong(buffer, offset, fileId);
        offset = Converter.writeLong(buffer, offset, timeStamp);
        for (int i = offset; i < HEADER_SIZE; i++) {
            buffer[i] = 0;
        }
    }

    /** Read the header from an existing file
    * @param file the file from which the header is read
    * @exception StorageException I/O error reading the file
    */
    FileHeader(RandomAccessFile file) throws StorageException {
        try {
            file.seek(0);
            fileId = file.readLong();
            timeStamp = file.readLong();
        }
        catch (IOException ex) {
            throw new StorageIOException(ex);
        }
    }

    /** update the time stamp in the header
    */
    void updateTime() {
        timeStamp = System.currentTimeMillis();
    }

    /** update the timestamp in the cache
    * @param page the page to update
    */
    static void updateTime(CachedPage page, long newTimeStamp) {
        Converter.writeLong(page.contents, TIMESTAMP_OFFSET, newTimeStamp);
    }
        
    /** Create new files with a given header
    * @param names the names of the files
    * @param size if greater than 0, the size to make the files
    * @param replace if true, replace existing files.  if false, throw
    * an exception if any of the files exists
    * @exception StorageException I/O error creating the files
    * or "replace" was false, and a file already exists
    */
    public void addFiles(
                    String names[], int size, boolean replace) 
                                        throws StorageException {

        if (!replace) {
            for (int i = 0; i < names.length; i++) {
                File file = new File(names[i]);
                if (file.exists()) {
                    throw new StorageBadRequestException(
                        MessageFormat.format("File {0} already exists",
                            new Object[] {names[i]} ));
                }
            }
        }

        try {
            for (int i = 0; i < names.length; i++) {
                RandomAccessFile file = new RandomAccessFile(names[i], "rw");
                write(file);
                if (size > 0)
                    file.setLength(size);
                else
                    file.setLength(file.getFilePointer());
                file.close();
            }
        }
        catch (IOException ex) {
            throw new StorageIOException(ex);
        }
    }

    /** Create a set of files with a common header
    * @param names the names of the files
    * @param size if greater than 0, the size to make the files
    * @param replace if true, replace existing files.  if false, throw
    * an exception if any of the files exists
    * @return the header used to create the files
    * @exception StorageException I/O error creating the files
    * or "replace" was false, and a file already exists
    */
    public static FileHeader createFiles(
                    String names[], int size, boolean replace) 
                                        throws StorageException {

        FileHeader header = new FileHeader();
	header.addFiles(names, size, replace);
        return header;
    }

    /** 
    * test for equality with another FileHeader
    * @param equals compare for equality with this
    */
    public boolean equals(Object o) {
        if (!(o instanceof FileHeader))
            return false;

        FileHeader header = (FileHeader)o;
        return (header.fileId == this.fileId) &&
                (header.timeStamp == this.timeStamp);
    }

}

