/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence.btreeimpl.btreestorage;

import java.io.*;
import java.text.*;
import java.util.*;

import org.netbeans.mdr.persistence.*;
/**
* A NormalBtreeExtent is the first (and possibly only) extent in a record.
*
*   <p>
*   Disk format:
*   <ol>
*   <li>
*   bytes 0-7               generic BtreeExtent header
*   <li>
*   bytes 8-9               key length (KL)
*   <li>
*   bytes 10-13             extent data length (DL)
*   <li>
*   bytes 14-17             record data length
*   <li>
*   bytes 18-17+KL          key
*   <li>
*   bytes 18+KL-17+KL+DL    data
*   </ol>
*   DL is data length for the extent plus any continuation extents.
*   If there are continuation extents, all of the bytes in this extent are full,
*   and the extent is of maximum size.
*   <p>
*   Note that the key must fit within the first chunk (the first 128 bytes).
*   That is, keys of longer than MAX_KEY_LENGTH bytes are not allowed.
*   <p>
*   extent data length will be unequal to record data length if and only
*   if there are continuation records.
*/
class NormalBtreeExtent extends ActiveBtreeExtent {

    /** size of fixed part of header */
    static final int NORMAL_FIXED_LENGTH = 18;

    /** maximum legal key length */
    static final int MAX_KEY_LENGTH = BtreeDataFile.BTREE_CHUNK_SIZE - NORMAL_FIXED_LENGTH;

    /** the record key */
    byte key[];

    /** the number of data bytes in the entire record */
    int totalLength;

    /** initialize a new NormalBtreeExtent
    * @param file the BtreeDataFile this extent will belong to
    * @param chunkNum where this extent begins
    * @param numChunks the size of the extent
    */
    NormalBtreeExtent(BtreeDataFile file, int chunkNum, short numChunks) {
        super(file, chunkNum, numChunks);
    }

    /** Convert a deleted extent to an active one.  The deleted extent has
    * already been removed from its chain
    * @param del the extent to convert
    * @param length data length
    */
    NormalBtreeExtent(DeletedBtreeExtent del, byte k[], int length) {
        super(del);
        key = k;
        dataStart = NORMAL_FIXED_LENGTH + key.length;
        setMyDataLength(length);
    }

    /** create a new NormalBtreeExtent in memory
    * @param file the BtreeDataFile this extent will belong to
    * @param chunkNum where this extent begins
    * @param numChunks the size of the extent
    * @param dLen how much data this extent will contain
    * @param totLen hom much data the entire record will contain
    * @param k the record's key
    */
    NormalBtreeExtent(
        BtreeDataFile file, int chunkNum, short numChunks, int totLen, byte k[]) 
                        throws StorageException {

        super(file, chunkNum, numChunks);
        if (k.length > MAX_KEY_LENGTH) {
            throw new StorageBadRequestException(
                MessageFormat.format("Invalid key length: {0}",
                    new Object[] {new Integer(k.length)} ));
        }
        key = k;
        setMyDataLength(totLen);
        totalLength = totLen;
        dataStart = NORMAL_FIXED_LENGTH + key.length;
        this.headerIsDirty = true;
    }

    /** read the type-specific parts of the extent header from the
    * buffer.
    * @param buffer the buffer to read from
    * @param offset the offset to being reading at
    */
    void readHeaderFromPage(byte buffer[], IntHolder offset) {
        short keyLen = Converter.readShort(buffer, offset);
        key = new byte[keyLen];
        dataLength =  Converter.readInt(buffer, offset);
        totalLength =  Converter.readInt(buffer, offset);
        System.arraycopy(buffer, offset.getValue(), key, 0, keyLen);
        dataStart = NORMAL_FIXED_LENGTH + key.length;
    }

    /** write the type-specific part of the header to the file cache
    * @param page the page to write to
    * @param offset the offset to begin an
    */
    protected void writeHeaderToPage(CachedPage page, int offset) {
        offset = Converter.writeShort(page.contents, offset, (short)key.length);
        offset = Converter.writeInt(page.contents, offset, dataLength);
        offset = Converter.writeInt(page.contents, offset, totalLength);
        System.arraycopy(key, 0, page.contents, offset, key.length);
    }

    /** get the amount of data contained in this extent
    * @return amount of data
    */
    int getMyDataLength() {
        return dataLength;
    }

    /** get the amount of data contained in the entire record
    * @return amount of data
    */
    int getTotalDataLength() {
        return totalLength;
    }

    /** get how much data this extent could contain
    * @return maximum amount of data which would fit
    */
    int getAvailableDataLength() {
        return getAvailableDataLength(chunks, key.length);
    }

    /** Get the magic number for this type of extent
    * @return the magic number
    */
    short getMagic() {
        return NORMAL_MAGIC;
    }

    /** return the number of chunks an extent with this much data would
    * occupy.  If an extent of maximum size wouldn't hold this much,
    * return the maximum extent size.
    * @param dataLength amount of data to hold.
    * @return size of extent in chunks
    */
    static int getNumChunks(int keyLength, int dataLength) {
        int size = (keyLength + dataLength + NORMAL_FIXED_LENGTH - 1) / 
                                        BtreeDataFile.BTREE_CHUNK_SIZE + 1;
        return Math.min(size, BtreeDataFile.MAX_CHUNKS_IN_EXTENT);
    }

    /** return the number of data bytes which can fit in a normal 
    * extent with the
    * given key length
    * @param numChunks size of extent in chunks
    * @param keyLength length of key
    * @return available data bytes
    */
    static int getAvailableDataLength(int numChunks, int keyLength) {
        return (BtreeDataFile.BTREE_CHUNK_SIZE * numChunks) 
            - NORMAL_FIXED_LENGTH - keyLength;
    }

    /** set the amount of data contained in this extent
    * @param length amount of data
    */
    int setMyDataLength(int length) {
        int oldDataLength = dataLength;
        dataLength = Math.min(length, getAvailableDataLength());
        if (dataLength != oldDataLength) {
            headerIsDirty = true;
        }
        if (totalLength != length) {
            totalLength = length;
            headerIsDirty = true;
        }
        return dataLength;
    }

    /** is the extent already full of data
    * @return true if the extent has no room for more data
    */
    boolean isMaximum() {
        return key.length + dataLength + NORMAL_FIXED_LENGTH
                                    == BtreeDataFile.MAX_BYTES_IN_EXTENT;
    }

    /** return type of extent
    * @return IS_NORMAL
    */
    byte getType() {
        return IS_NORMAL;
    }

    /** return name of type of extent
    * @return NORMAL_NAME
    */
    String getTypeName() {
        return NORMAL_NAME;
    }

    /** dump extent as text (for debugging)
    * @param level bitmask of what to dump. See the superclass for the
    * meaning of level.
    * @param strm where to dump it to
    */
    void dump(int level, PrintWriter strm) 
                    throws StorageException{

        super.dump(level, strm);
	if ((level & DUMP_BASIC) != 0) {
	    strm.println("" + totalLength + " bytes in record");
	    strm.println("" + key.length + " bytes in key");
	}

        if ((level & DUMP_KEY) != 0) {
            dumpBytesAsHex(new ByteArrayInputStream(key), strm, "\t");
        }
    }
}
