/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence.btreeimpl.btreestorage;

import java.util.*;
import org.netbeans.mdr.persistence.MOFID;

public class TransactionCache {

    // max. number of cached operations
    private static final int GLOBAL_TRESHOLD = 768;
    private static final int LOCAL_TRESHOLD = 512;
    
    // operations codes
    public static final byte OP_INSERT = 0;
    public static final byte OP_DELETE = 1;
    public static final byte OP_REPLACE = 2;
    
    // number of cached operations
    private static int cacheSize = 0;
    private int localCacheSize = 0;
    private boolean tresholdReached = false;
    
    // true if transaction cache stores some commited operations
    private boolean dataCommited = false;
    
    // list of lits containing commited operations (each list corresponds to one commit)
    private LinkedList commitedOperations = new LinkedList ();
    // cached operations that have not been commited yet
    private LinkedList operations = new LinkedList ();    
            
    public TransactionCache() {
    }

    /**
     * @return true if transaction cache treshold has been reached
     */
    public boolean tresholdReached () {
        return tresholdReached;
    }
    
    /**
     * @return true if cache stores some commited operations
     */
    public boolean containsCommitedData () {
        return dataCommited;
    }
    
    /**
     * Resets cache (to store no operations).
     */
    public void clear () {
        operations = new LinkedList ();                
        commitedOperations.clear ();
        synchronized (TransactionCache.class) {
            cacheSize -= localCacheSize;
            localCacheSize = 0;
            tresholdReached = false;
        }
        dataCommited = false;
    }
    
    private void incrementCacheSize() {
        synchronized (TransactionCache.class) {
            localCacheSize++;
            cacheSize++;
     
            tresholdReached |= cacheSize >= GLOBAL_TRESHOLD || localCacheSize >= LOCAL_TRESHOLD;
        }
    }
    
    /** adds one insert operation */
    public void addInserted (MOFID id, byte [] value) {
        if (!tresholdReached) {
            operations.addLast (new Record (OP_INSERT, id, value));
            incrementCacheSize();
        }
    }
    
    /** adds one delete operation */
    public void addDeleted (MOFID id) {
        if (!tresholdReached) {
            operations.addLast ((new Record (OP_DELETE, id, null)));
            incrementCacheSize();
        }
    }
    
    /** adds one replace operation */
    public void addReplaced (MOFID id, byte [] value) {
        if (!tresholdReached) {
            operations.addLast (new Record (OP_REPLACE, id, value));
            incrementCacheSize();
        }
    }
    
    /**
     * Commits operations temporarly stored in @link #operations.
     */
    public void commit () {
        if (!operations.isEmpty())
            commitedOperations.addLast (operations);
                
        operations = new LinkedList ();
        dataCommited = true;
    }

    /**
     * Returns cache iterator.
     */
    public CacheIterator iterator () {
        return new CacheIterator ();
    }
    
    // Record *******************************************************************
        
    public static class Record {
        byte op; // operation code
        MOFID id; // key
        byte [] value; // serialized value of a Streamable object
        
        public Record (byte opId, MOFID id, byte [] value) {
            this.op = opId;
            this.id = id;
            this.value = value;
        }
    }
    
    // CacheIterator ************************************************************
    
    public class CacheIterator {
        
        private Iterator primar;
        private Iterator secondar = null;
        private Object nextItem = null;
        
        CacheIterator () {
            primar = commitedOperations.iterator ();
            if (primar.hasNext ()) {
                secondar = ((List)primar.next ()).iterator ();
                fetchNext();
            }
        }
        
        private void fetchNext () {            
            if (secondar.hasNext ()) {
                nextItem = secondar.next ();
            } else {
                if (primar.hasNext ()) {
                    secondar = ((List)primar.next()).iterator();
                    nextItem = secondar.next ();
                } else {
                    nextItem = null;
                }
            }
        }
        
        public boolean hasNext () {
            return nextItem != null;
        }
        
        public Record next () {
            Object temp = nextItem;
            fetchNext ();
            return (Record) temp;
        }
                    
    }

}