/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence.memoryimpl;

import java.util.*;
import java.io.*;

import org.netbeans.mdr.persistence.*;
import org.netbeans.mdr.util.*;

public class PrimaryIndexImpl extends SinglevaluedIndexImpl {

    /* Streams used to serialize data */
    private ByteArrayOutputStream baoStrm = new ByteArrayOutputStream ();
    private DataOutputStream daoStrm = new DataOutputStream (baoStrm);    
    
    public PrimaryIndexImpl(StorageImpl storage) {
        super(StorageImpl.PRIMARY_INDEX_NAME, storage, Storage.EntryType.MOFID, Storage.EntryType.STREAMABLE);        
    }
    
    /** Removes all values assosiated in the index with specified key.
     * @return true if this index changed as a result of this call
     * @param key
     * @throws StorageException
     */
    public synchronized boolean remove(Object key) throws StorageException {
        Object original = table.remove(key);
        if (original != null) {
            Object valueLog = transLog.isLogged(key) ? null : createValueLog((Streamable) original);
            transLog.logRemove(key, valueLog);
            return true;
        } else
            return false;
    }
    
    /** Associates the specified value with the specified key in this index.
     * @return true if there was an item in this index that was associated with the key
     * prior to this call
     * @param key
     * @param value
     * @throws StorageException
     */
    public synchronized boolean put(Object key,Object value) throws StorageException {
        Object original = table.put(key, value);
        if (original == null) {
            transLog.logAdd(key);
        } else {
            Object valueLog = transLog.isLogged(key) ? null : createValueLog((Streamable) original);
            transLog.logReplace(key, valueLog);
        }
        return original != null;
    }
    
    /** Replaces the original value associated with the specified key in this index
     * with new value. If no value was associated with this key prior to this call
     * StorageBadRequestException is thrown.
     * @param key
     * @param value
     * @throws StorageException
     * @throws StorageBadRequestException if the index has no entry with the given
     *           key 
     */
    public synchronized void replace(Object key,Object value) throws StorageException, StorageBadRequestException {
        Object original = table.put (key, value);
        if (original == null) {
            table.remove(key);
            throw new StorageBadRequestException ("Cannot replace item that does not exist in the index.");
        }
        Object valueLog = transLog.isLogged(key) ? null : createValueLog((Streamable) original);
        transLog.logReplace(key, valueLog);
    }
    
    public synchronized void willChange (Object key) throws StorageException {
        if (!transLog.isLogged(key)) {
            Object value = table.get (key);
            transLog.logValue(key, createValueLog ((Streamable) value));
        }
        transLog.setDirty(key);
    }
    
    public void changed (Object key) {        
        // do nothing
    }
    
    private PrimaryValueLog createValueLog (Streamable value) throws StorageException {
        baoStrm.reset();
        try {
            ((Streamable) value).write(daoStrm);
        } catch (RuntimeException e) {
            // ignore
        }
        return new PrimaryValueLog (baoStrm.toByteArray(), value);
    }

    /* -------------------------------------------------------------------- */
    /* -- Implementation of org.netbeans.mdr.persistence.Streamable ------- */
    /* -------------------------------------------------------------------- */

    /** This method will be used to move changed object from storage cache
     * to the persistent part of storage. It writes the object`s state
     * (set of attributes) in the stream as an array of bytes, for example
     * in textual representation.
     * @param outputStream OutputStream that holds value of a Streamable object
     */
    public void write(java.io.OutputStream out) throws StorageException {
        try {
            IOUtils.writeInt(out, table.size());
            for (Iterator it = table.entrySet().iterator(); it.hasNext();) {
                Map.Entry entry = (Map.Entry) it.next();
                storage.writeMOFID(out, (MOFID) entry.getKey());
                Streamable value = (Streamable) entry.getValue();
                IOUtils.writeString(out, value.getClass().getName());
                value.write(out);
            }
        } catch (java.io.IOException e) {
            throw new StorageIOException(e);
        }
    }
    /** Restore state of the Storable object from the stream.
     * @param inputStream InputStream that represents an internal representation of fields of a Streamable object
     * in which it was written by {@link write } method
     */
    public void read(java.io.InputStream is) throws StorageException {
        try {
            int size = IOUtils.readInt(is);
            table = new HashMap(size * 4 / 3);
            for (int i = 0; i < size; i++) {
                MOFID key = storage.readMOFID(is);
                Streamable value = (Streamable) Class.forName(IOUtils.readString(is)).newInstance();
                if (value instanceof StorageClient) {
                    ((StorageClient) value).setStorage(storage);
                }
                value.read(is);
                table.put(key, value);
            }
        } catch (java.io.IOException e) {
            throw new StorageIOException(e);
        } catch (Exception e) {
            throw (StorageException) Logger.getDefault().annotate(new StoragePersistentDataException(), e);
        }
    }

    // inner class ..............................................................

    private class PrimaryValueLog implements TransactionLog.ValueLog {

        private byte [] bytes;
        private Streamable obj;

        PrimaryValueLog (byte [] bytes, Streamable obj) {
            this.bytes = bytes;
            this.obj = obj;
        }

        public Object resolveOriginalValue ()  throws StorageException {
            ByteArrayInputStream baiStrm = new ByteArrayInputStream (bytes);
            DataInputStream daiStrm = new DataInputStream (baiStrm);
            if (obj instanceof StorageClient) {
                ((StorageClient) obj).setStorage(storage);
            }
            obj.read(daiStrm);
            return obj;
        }
    }
}
