/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.persistence.memoryimpl;

import java.util.*;
import org.netbeans.mdr.persistence.*;

public class TransactionLog {

    // operations codes
    private static final byte OP_NONE = 0;
    private static final byte OP_ADD = 1;
    private static final byte OP_REMOVE = 2;
    private static final byte OP_REPLACE = 3;

    private HashMap log = new HashMap ();
    private SinglevaluedIndex index;
    private boolean rollingBack = false;
    
    // ..........................................................................    
    
    public TransactionLog (SinglevaluedIndex index) {
        this.index = index;
    }
    
    // ..........................................................................
    
    /**
     * Clears the transaction log.
     */
    public void clear () {
        log.clear ();
    }

    /**
     * Logs add operation.
     */
    public void logAdd (Object key) {
        if (rollingBack)
            return;
        Record rec = (Record) log.get(key);
        if (rec == null) {
            rec = new Record (OP_REMOVE, null);
            log.put (key, rec);
        } else {
            rec.opCode = OP_REPLACE;
        }
    }

    /**
     * Logs remove operation.
     */
    public void logRemove (Object key, Object value) {
        if (rollingBack)
            return;
        Record rec = (Record) log.get(key);
        if (rec == null) {
            rec = new Record (OP_ADD, value);
            log.put (key, rec);
        } else {
            log.remove (key);
        }
    }

    /**
     * Logs replace operation.
     */
    public void logReplace (Object key, Object oldValue) {
        if (rollingBack)
            return;
        Record rec = (Record) log.get(key);
        if (rec == null) {
            rec = new Record (OP_REPLACE, oldValue);
            log.put (key, rec);
        }
    }        

    public void logValue (Object key, Object value) {
        if (rollingBack)
            return;
        log.put (key, new Record (OP_NONE, value));
    }
    
    public boolean isLogged (Object key) {
        return log.get (key) != null;
    }
    
    public void setDirty (Object key) {
        if (rollingBack)
            return;
        Record rec = (Record) log.get(key);
        if (rec.opCode == OP_NONE)
            rec.opCode = OP_REPLACE;
    }        
    
    /**
     * Rolls back all changes performed over the index during the recorded transaction.
     */
    public void rollBack () throws StorageException {
        rollingBack = true;
        Iterator iter = log.keySet().iterator();
        while(iter.hasNext()) {
            Object key = iter.next();
            Record rec = (Record) log.get(key);
            if (rec.opCode == OP_NONE)
                continue;
            Object value;
            if (rec.value instanceof ValueLog) {
                value = ((ValueLog) rec.value).resolveOriginalValue ();
            } else {
                value = rec.value;
            }
            switch (rec.opCode) {
                case OP_ADD: 
                    index.add (key, value);
                break;
                case OP_REMOVE:
                    index.remove (key);
                break;
                case OP_REPLACE:
                    index.replace (key, value);
                break;
            } // switch                                    
        }
        rollingBack = false;
    }

    // ValueLog .................................................................
    
    public interface ValueLog {
        
        public Object resolveOriginalValue () throws StorageException;
        
    } // ValueLog
    
    // Record ...................................................................

    private static class Record {
        
        byte opCode; // operation code
        Object value;

        public Record (byte opCode, Object value) {
            this.opCode = opCode;
            this.value = value;
        }
        
    } // Record

}
