/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.storagemodel;

import org.netbeans.mdr.NBMDRepositoryImpl;
import org.netbeans.mdr.handlers.EnumResolver;
import org.netbeans.mdr.handlers.StructImpl;
import org.netbeans.mdr.persistence.MOFID;
import org.netbeans.mdr.persistence.StorageException;
import org.netbeans.mdr.util.*;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.io.IOException;
import java.io.StringReader;
import java.net.URL;
import java.util.*;

/** Special XmiReader used for reading MOF model
 * when the repository is empty (during the bootsequence).
 *
 * @author Petr Hrebejk, Pavel Buzek, Martin Matula
 * @version 0.2
 */
public class BootReader extends Object {
    /** package prefix for JMI interfaces */
    private static final String JMI_PACKAGE_PREFIX = "javax.jmi.model.";

    /** MDRStorage which has to be booted */
    private final MdrStorage mdrStorage;
    /** URL of XMI document to be read */
    private final URL docURL;
    /** object descriptors by fully qualified name */
    private final Hashtable odByName = new Hashtable(100);
    /** object descriptors by XMI.ID */
    private final Hashtable odByXmiId = new Hashtable(100);
    /** TypeAlias XMI.ID -> XMI.ID of type it is pointing to */ 
    private final Hashtable aliases = new Hashtable(10);
    /** data types that are going to be excluded - instances of PrimitiveType which are not contained in PrimitiveTypes package */
    private final List excludeTypes = new ArrayList(3);
    /** object MofIds by fully qualified name */
    private final Hashtable objectsByName = new Hashtable(100);
    /** class proxy MofIds by fully qualified name of metaobject */
    private final Hashtable proxyIdByName = new Hashtable(11);
    
    /**  class proxy objects by MofIds */
    private final Hashtable proxyById = new Hashtable();
    /**  List of descriptor objects : class - superclass */
    private final ArrayList superclassByClass = new ArrayList();
    /**  class proxy objects by XmiIds */
    private final Hashtable proxyIdByXmiId = new Hashtable();
    /**  forward proxies references */
    private final HashSet proxyReferences = new HashSet();
    /**  association ends by other ends (FQNs) */
    private final HashMap otherEnds = new HashMap();

    /** parsed XMI document */
    private Document document = null;
    /** reference to Contains association */
    private StorableAssociation containsAssociation = null;

    private static class DummyER implements EntityResolver {
        private static EntityResolver instance = new DummyER();

        public static EntityResolver getInstance() {
            return instance;
        }

        public InputSource resolveEntity(String publicID, String systemID) {
            Logger.getDefault().log("resolving reference: " + publicID + ", " + systemID);
            return new InputSource(new StringReader(""));
        }
    }   

    /** Creates new BootReader
     * @param storage reference to instance of <CODE>MdrStorage</CODE> class
     * @param docURL URL of XMI document to read (the document should contain metamodel of MOF)
     */
    public BootReader(MdrStorage storage, URL docURL) {
        mdrStorage = storage;
        this.docURL = docURL;
    }

    /** Parses the XMI document.
     * @param docURL URL of the XMI document to be parsed
     * @throws IOException
     * @throws SAXException
     * @throws ParserConfigurationException
     * @return parsed XMI document
     */    
    protected Document parse(URL docURL) throws IOException, SAXException, ParserConfigurationException {
        Document result = null;
        DocumentBuilderFactory bfact = DocumentBuilderFactory.newInstance();
	bfact.setValidating(false);
        DocumentBuilder docBuilder = bfact.newDocumentBuilder();
	docBuilder.setEntityResolver(DummyER.getInstance());
        InputSource is = new InputSource(docURL.toString());
        result = docBuilder.parse(is);
        return result;
    }

    /** Reads the XMI document that was passed to the costructor
     * into the MdrStorage which was passed to the constructor.
     * @return outermost containers in the imported model
     * @throws IOException 
     * @throws SAXException  
     */
    public Collection read() throws IOException, SAXException {
        try {
            document = parse( docURL );
            Node xmiContent = document.getElementsByTagName(XmiConstants.XMI_CONTENT).item(0);
            createProxies(xmiContent);
            Collection result = createInstances(xmiContent);
            rebuildMetas();
            initSuperClasses();
            return result;
        } catch (ParserConfigurationException e) {
            Logger.getDefault().log("Unable to parse document. "+e.getMessage() );
            return null;
        }
    }
    
    private void initSuperClasses() {
        Object sc;
        Class iface[] = new Class[1];
        for (Iterator it = proxyById.values().iterator(); it.hasNext();) {
            sc = it.next();
            if (sc instanceof StorableClass) {
                try {
                    iface[0] = null;
                    ((StorableClass) sc).initClassSuperclass(iface);
                    iface[0] = null;
                    ((StorableClass) sc).initInstanceSuperclass(iface);
                } catch (Exception e) {
                    throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
                }
            }
        }
    }

    /** Goes through the whole XMI document and creates all needed proxies
     * (for all packages, classes and associations in the document).
     * @param xmiContent XMI.content node of the XMI document
     */    
    private void createProxies(Node xmiContent) {
        NodeList childNodes = xmiContent.getChildNodes();

        for (int i = 0; i < childNodes.getLength(); i++) {
            Node packageNode = childNodes.item(i);
            if (XmiUtils.isTextNode(packageNode)) continue;
            if (XmiUtils.getXmiAttrValueAsString(packageNode, MOFConstants.MODEL_MODEL_ELEMENT_NAME).equals(MOFConstants.MODEL)) {
                String logicalMOFName = ""; // NOI18N
                createProxiesInPackage(null, new MOFID (bootSequenceNumber(logicalMOFName),logicalMOFName), packageNode);
            }
        }
    }
    
    /** Goes through the whole XMI document and creates all instances contained in the document.
     * @param xmiContent XMI.content node of the XMI document
     * @return outermost containers in the model
     */    
    private Collection createInstances(Node xmiContent) {
        XmiUtils.XmiNodeIterator classes = new XmiUtils.XmiNodeIterator(xmiContent);
        Collection result = createInstances(classes, null, "", null);
        resolveSuperClasses();
        resolveReferences();
        setValues();
        return result;
    }

    /** Sets metaobjects for all read objects to reference the correct model element instances.
     */
    private void rebuildMetas() {
        try {
            StorablePackage sp = mdrStorage.getContextOutermostPackage(NBMDRepositoryImpl.BOOT_MOF);
            sp.replaceValues(objectsByName);

            StorableClass sc;
            StorableObject so;
            StorableAssociation sa;
            
            // replace class proxy metas
            for (Iterator classes = sp.getAllClasses().iterator(); classes.hasNext();) {
                sc = (StorableClass) classes.next();
                sc.replaceValues(objectsByName);
                // replace instance metas
                for (Iterator instances = sc.allObjects(false).iterator(); instances.hasNext();) {
                    so = (StorableObject) instances.next();
                    so.replaceValues(objectsByName);
                }
            }

            for (Iterator associations = sp.getAllAssociations().iterator(); associations.hasNext();) {
                sa = (StorableAssociation) associations.next();
                sa.replaceValues(objectsByName);
            }
            
            mdrStorage.rebuildMetas(NBMDRepositoryImpl.BOOT_MOF, objectsByName);
            mdrStorage.setProperty(MdrStorage.MOF_TOP_PACKAGE, sp.getMofId());
        } catch (Exception e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }
    
    /**
     * @param immediatePackage
     * @param fullPackageName
     * @param packageNode
     * @return  */    
    private StorablePackage createProxiesInPackage(StorablePackage immediatePackage, MOFID fullPackageName, Node packageNode) {
        // Find all non abstract classes and associations
        String stringifiedFullPackageName = fullPackageName.getStorageID ();
        String packagePrefix = stringifiedFullPackageName.length() > 0 ? stringifiedFullPackageName + "." : "";
        String elementName = XmiUtils.getXmiAttrValueAsString(packageNode, MdrStorage.MODEL_MODEL_ELEMENT_NAME);
        String logicalMOFName = packagePrefix + elementName;
        MOFID packageName = new MOFID (bootSequenceNumber(logicalMOFName), logicalMOFName);

        StorablePackage currentPackage = createPackage(immediatePackage, packageName, elementName);

        Node contents = XmiUtils.getChildNode(packageNode, MdrStorage.MODEL_NAMESPACE_CONTENTS);
        XmiUtils.XmiNodeIterator ni = new XmiUtils.XmiNodeIterator(contents);
        
        MOFID name = null;
        String nodeName = null;
        List attrDescs = null;
        List clAttrDescs = null;
        Node classContents = null;
        XmiUtils.XmiNodeIterator features = null;
        String storableXmiId = null;
        StorableClass sc = null;

        for (; ni.hasNext();) {
            Node nonAbstract = ni.next();
            elementName = XmiUtils.getXmiAttrValueAsString(nonAbstract, MdrStorage.MODEL_MODEL_ELEMENT_NAME);
            logicalMOFName = packageName.getStorageID() + "." + elementName;
            name = new MOFID (bootSequenceNumber(logicalMOFName), logicalMOFName);
            nodeName = XmiUtils.resolveFullName(nonAbstract);

            if (nodeName.equals(MdrStorage.MODEL_CLASS)) {
                //prepare attribute descriptor for createClass - StorableClass construcor
                attrDescs = new ArrayList();
                clAttrDescs = new ArrayList();
                boolean classDerived = false;
                boolean instanceDerived = false;
                boolean isSingleton = "true".equalsIgnoreCase(XmiUtils.getXmiAttrValueAsString(nonAbstract, MdrStorage.MODEL_CLASS_IS_SINGLETON));
                boolean isAbstract = "true".equalsIgnoreCase(XmiUtils.getXmiAttrValueAsString(nonAbstract, MdrStorage.MODEL_GENERALIZABLE_ELEMENT_IS_ABSTRACT));

                classContents = XmiUtils.getChildNode(nonAbstract, MdrStorage.MODEL_NAMESPACE_CONTENTS);
                if (classContents != null) { 
                    features = new XmiUtils.XmiNodeIterator(classContents); //collect all attributes

                    if (features != null) {
                        Node featureNode = null;
                        String attName = null;
                        boolean isDerived = false;
                        for (; features.hasNext();) {
                            featureNode = features.next();
                            if (XmiUtils.resolveFullName(featureNode).equals(MdrStorage.MODEL_ATTRIBUTE)) {
                                attName = XmiUtils.getXmiAttrValueAsString(featureNode, MdrStorage.MODEL_MODEL_ELEMENT_NAME);
                                isDerived = "true".equalsIgnoreCase(XmiUtils.getXmiAttrValueAsString(featureNode, MdrStorage.MODEL_ATTRIBUTE_IS_DERIVED));
                                boolean classifier = MOFConstants.SCOPE_CLASSIFIER.equals(XmiUtils.getXmiAttrValueAsString(featureNode, MOFConstants.MODEL_FEATURE_SCOPE));
                                if (!isDerived && attName != null) {
                                    Node multiplicity = XmiUtils.getChildNode(featureNode, MdrStorage.MODEL_STRUCTURAL_FEATURE_MULTIPLICITY);
                                    Node fields = XmiUtils.getChildNode(multiplicity, MdrStorage.MODEL_MULTIPLICITY_TYPE);
                                    int maxSize;
                                    String upper;
                                    if (fields == null) {
                                        // this is for the case when structure fields are represented using xmi.field tags in the XMI
                                        XmiUtils.XmiNodeIterator fieldVals = new XmiUtils.XmiNodeIterator(multiplicity, XmiConstants.XMI_FIELD);
                                        fieldVals.next(); // skip lower
                                        upper = fieldVals.next().getFirstChild().getNodeValue();
                                    } else {
                                        // this is for the case when structure fields are represented the same way as MOF attributes in the XMI
                                        upper = XmiUtils.getXmiAttrValueAsString(fields, MdrStorage.MODEL_MULTIPLICITY_TYPE_UPPER);
                                    }
                                    try {
                                        maxSize = Integer.parseInt(upper);
                                    } catch (NumberFormatException e) {
                                        throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
                                    }
                                    logicalMOFName = name.getStorageID() + "." + attName;
                                    StorableClass.AttributeDescriptor desc = new StorableClass.AttributeDescriptor(mdrStorage, new MOFID (bootSequenceNumber(logicalMOFName), logicalMOFName), attName, Object.class, 0, maxSize, false, true, true, null);
                                    if (classifier) {
                                        clAttrDescs.add(desc);
                                    } else {
                                        attrDescs.add(desc);
                                    }
                                }
                            } else if (XmiUtils.resolveFullName(featureNode).equals(MdrStorage.MODEL_OPERATION)) {
                                isDerived = true;
                            } else {
                                isDerived = false;
                            }
                            
                            if (isDerived) {
                                if ("classifier_level".equalsIgnoreCase(XmiUtils.getXmiAttrValueAsString(featureNode, MdrStorage.MODEL_FEATURE_SCOPE))) {
                                    classDerived = true;
                                } else {
                                    instanceDerived = true;
                                }
                            }
                        }
                    }
                }
                sc = createClass(currentPackage, name, elementName, attrDescs, clAttrDescs, classDerived, instanceDerived, isSingleton, isAbstract);
                storableXmiId = XmiUtils.getAttributeValueAsString(nonAbstract, XmiConstants.XMI_ID);
                if ((sc != null) && (storableXmiId != null)) {
                    proxyIdByXmiId.put(storableXmiId, sc.getMofId()); //prepare for later use - CLASSES are enough
                } else {
                    Logger.getDefault().log( "Unable to set proxy by XmiId: "+nonAbstract.getNodeName() );
                }

            } else if (nodeName.equals(MdrStorage.MODEL_ASSOCIATION)) {
                // find the two ends
                Node asocContents = XmiUtils.getChildNode(nonAbstract, MdrStorage.MODEL_NAMESPACE_CONTENTS);
                XmiUtils.XmiNodeIterator asocEnds = new XmiUtils.XmiNodeIterator(asocContents, MdrStorage.MODEL_ASSOCIATION_END);
                // the first end
                Node asocEndA = asocEnds.next ();
                String endAName = XmiUtils.getXmiAttrValueAsString(asocEndA, MdrStorage.MODEL_MODEL_ELEMENT_NAME);
                // read multiplicity
                Node multiplicityA = XmiUtils.getChildNode(asocEndA, MdrStorage.MODEL_ASSOCIATION_END_MULTIPLICITY);
                Node fieldsA = XmiUtils.getChildNode(multiplicityA, MdrStorage.MODEL_MULTIPLICITY_TYPE);
                String lowerA;
                String upperA;
                String orderedA;
                String uniqueA;
                if (fieldsA == null) {
                    // this is for the case when structure fields are represented using xmi.field tags in the XMI
                    XmiUtils.XmiNodeIterator fields = new XmiUtils.XmiNodeIterator(multiplicityA, XmiConstants.XMI_FIELD);
                    lowerA = fields.next().getFirstChild().getNodeValue();
                    upperA = fields.next().getFirstChild().getNodeValue();
                    orderedA = fields.next().getFirstChild().getNodeValue();
                    uniqueA = fields.next().getFirstChild().getNodeValue();
                } else {
                    // this is for the case when structure fields are represented the same way as MOF attributes in the XMI
                    lowerA = XmiUtils.getXmiAttrValueAsString(fieldsA, MdrStorage.MODEL_MULTIPLICITY_TYPE_LOWER);
                    upperA = XmiUtils.getXmiAttrValueAsString(fieldsA, MdrStorage.MODEL_MULTIPLICITY_TYPE_UPPER);
                    orderedA = XmiUtils.getXmiAttrValueAsString(fieldsA, MdrStorage.MODEL_MULTIPLICITY_TYPE_IS_ORDERED);
                    uniqueA = XmiUtils.getXmiAttrValueAsString(fieldsA, MdrStorage.MODEL_MULTIPLICITY_TYPE_IS_UNIQUE);
                }
                boolean aggrA = !XmiUtils.getXmiAttrValueAsString(asocEndA, MdrStorage.MODEL_ASSOCIATION_END_AGGREGATION).equals("none");

                // the second end
                Node asocEndB = asocEnds.next();
                String endBName = XmiUtils.getXmiAttrValueAsString (asocEndB, MdrStorage.MODEL_MODEL_ELEMENT_NAME);
                // read multiplicity
                Node multiplicityB = XmiUtils.getChildNode(asocEndB, MdrStorage.MODEL_ASSOCIATION_END_MULTIPLICITY);
                Node fieldsB = XmiUtils.getChildNode(multiplicityB, MdrStorage.MODEL_MULTIPLICITY_TYPE);
                String lowerB;
                String upperB;
                String orderedB;
                String uniqueB;
                if (fieldsB == null) {
                    // this is for the case when structure fields are represented using xmi.field tags in the XMI
                    XmiUtils.XmiNodeIterator fields = new XmiUtils.XmiNodeIterator(multiplicityB, XmiConstants.XMI_FIELD);
                    lowerB = fields.next().getFirstChild().getNodeValue();
                    upperB = fields.next().getFirstChild().getNodeValue();
                    orderedB = fields.next().getFirstChild().getNodeValue();
                    uniqueB = fields.next().getFirstChild().getNodeValue();
                } else {
                    // this is for the case when structure fields are represented the same way as MOF attributes in the XMI
                    lowerB = XmiUtils.getXmiAttrValueAsString(fieldsB, MdrStorage.MODEL_MULTIPLICITY_TYPE_LOWER);
                    upperB = XmiUtils.getXmiAttrValueAsString(fieldsB, MdrStorage.MODEL_MULTIPLICITY_TYPE_UPPER);
                    orderedB = XmiUtils.getXmiAttrValueAsString(fieldsB, MdrStorage.MODEL_MULTIPLICITY_TYPE_IS_ORDERED);
                    uniqueB = XmiUtils.getXmiAttrValueAsString(fieldsB, MdrStorage.MODEL_MULTIPLICITY_TYPE_IS_UNIQUE);
                }
                boolean aggrB = !XmiUtils.getXmiAttrValueAsString(asocEndB, MdrStorage.MODEL_ASSOCIATION_END_AGGREGATION).equals("none");

                StorableAssociation storableAssociation;
                try {
                    storableAssociation = createAssociation(currentPackage, name, elementName, endAName, endBName, Integer.parseInt(lowerA), Integer.parseInt(upperA), Integer.parseInt(lowerB), Integer.parseInt(upperB), orderedA.equals("true"), orderedB.equals("true"), uniqueA.equals("true"), uniqueB.equals("true"), aggrA, aggrB);
                } catch (NumberFormatException e) {
                    throw new DebugException("Wrong format of multiplicity");
                }
                String endALogicalName = name.getStorageID() + "." + endAName;
                String endBLogicalName = name.getStorageID() + "." + endBName;
                long asn = bootSequenceNumber (endALogicalName);
                long bsn = bootSequenceNumber (endBLogicalName);
                otherEnds.put(new MOFID (asn, endALogicalName), new MOFID (bsn, endBLogicalName));
                otherEnds.put(new MOFID (bsn, endBLogicalName), new MOFID (asn, endALogicalName));

                if (name.getStorageID().equals(MdrStorage.MODEL_CONTAINS)) {
                    containsAssociation = storableAssociation;
                }
            } else if (nodeName.equals(MdrStorage.MODEL_PACKAGE)) {
                createProxiesInPackage(currentPackage, packageName, nonAbstract);
            } else if (nodeName.equals(MdrStorage.MODEL_PRIMITIVE_TYPE)) {
                if (!name.getStorageID().startsWith(MdrStorage.DATATYPES + ".")) {
                    String xmiId = XmiUtils.getAttributeValueAsString(nonAbstract, XmiConstants.XMI_ID);
                    excludeTypes.add(xmiId);
                }
            } else if (nodeName.equals(MdrStorage.MODEL_ALIAS_TYPE)) {
                String xmiId = XmiUtils.getAttributeValueAsString(nonAbstract, XmiConstants.XMI_ID);
                aliases.put(xmiId, XmiUtils.getXmiRefValue(nonAbstract, MdrStorage.MODEL_TYPED_ELEMENT_TYPE).get(0));
            }
        }
        return currentPackage;
    }
    
/**
 * @param items
 * @param containerMofId
 * @param namePrefix
 * @param parent
 * @return  */    
    private Collection createInstances(XmiUtils.XmiNodeIterator items, MOFID containerMofId, String namePrefix, ObjectDescriptor parent) {
        ArrayList instances = new ArrayList();
        for (;items.hasNext();) {
            Node classNode = items.next();

            if (XmiUtils.isTextNode(classNode)) continue;

            // collect data needed for object creation
            String logicalMOFName = namePrefix + XmiUtils.getXmiAttrValueAsString(classNode, MdrStorage.MODEL_MODEL_ELEMENT_NAME);
            MOFID name = new MOFID (bootSequenceNumber(logicalMOFName), logicalMOFName);
            
            String xmiId = XmiUtils.getAttributeValueAsString(classNode, XmiConstants.XMI_ID);
            logicalMOFName = XmiUtils.resolveFullName (classNode);
            MOFID mofClassName = new MOFID (bootSequenceNumber(logicalMOFName), logicalMOFName);
            MOFID mofPackageName = getPackageName (mofClassName);
            MOFID classProxy = (MOFID) proxyIdByName.get(mofClassName);
            MOFID packageProxy = (MOFID) proxyIdByName.get(mofPackageName);
            String dataType = null;

            if (xmiId != null && excludeTypes.contains(resolveTypeId(xmiId))) continue;
            
            if (mofClassName.getStorageID().equals(MdrStorage.MODEL_ATTRIBUTE) || mofClassName.getStorageID().equals(MdrStorage.MODEL_STRUCTURE_FIELD)) {
                dataType = resolveTypeId((String) XmiUtils.getXmiRefValue(classNode, MdrStorage.MODEL_TYPED_ELEMENT_TYPE).get(0));
                if (excludeTypes.contains(dataType)) continue;
            }
            
            // create new object
            StorableObject instance = createObject(mofClassName, packageProxy, classProxy);
            instances.add(instance);

            // register object to helper hashtables
            objectsByName.put(name, instance.getMofId());
            ObjectDescriptor od = new ObjectDescriptor(classNode, instance, name);
            if (odByName.put(name, od) != null) {
                Logger.getDefault().log("duplicate for: " + name);
            }
            if (xmiId != null) {
                odByXmiId.put(xmiId, od);
            }

            // if the object is contained in other object update contains reference
            if (containerMofId != null) {
                try {
                    containsAssociation.addLink(containerMofId, instance.getMofId());
                } catch (StorageException e) {
                    throw new DebugException("Storage exception: " + e);
                }
            }
            
            // in special cases deeper investigation is needed
            if (mofClassName.getStorageID().equals(MdrStorage.MODEL_ATTRIBUTE)) {
                if (XmiUtils.getXmiAttrValueAsString(classNode, MdrStorage.MODEL_ATTRIBUTE_IS_DERIVED).equals("false")) {
                    String upper = XmiUtils.getXmiAttrValueAsString(XmiUtils.getChildNode(XmiUtils.getChildNode(classNode, MdrStorage.MODEL_STRUCTURAL_FEATURE_MULTIPLICITY), MdrStorage.MODEL_MULTIPLICITY_TYPE), MdrStorage.MODEL_MULTIPLICITY_TYPE_UPPER);
                    if (upper == null) {
                        XmiUtils.XmiNodeIterator fields = new XmiUtils.XmiNodeIterator(XmiUtils.getChildNode(classNode, MdrStorage.MODEL_STRUCTURAL_FEATURE_MULTIPLICITY), XmiConstants.XMI_FIELD);
                        fields.next();
                        upper = fields.next().getFirstChild().getNodeValue();
                    }
                    parent.addFeature(new Feature(name, Feature.ATTRIBUTE, dataType, null, !upper.equals("1")));
                }
            } else if (mofClassName.getStorageID().equals(MdrStorage.MODEL_STRUCTURE_FIELD)) {
                parent.addFeature(new Feature(name, Feature.FIELD, dataType, null, false));
            } else if (mofClassName.getStorageID().equals(MdrStorage.MODEL_REFERENCE)) {
                String referencedEnd = (String) XmiUtils.getXmiRefValue(classNode, MdrStorage.MODEL_REFERENCE_REFERENCED_END).get(0);
                parent.addFeature(new Feature(name, Feature.REFERENCE, null, referencedEnd, true));
//                MOFID proxyId = (MOFID) proxyIdByName.get(parent.getName ());
                proxyReferences.add(parent); // -- add forward reference - ObjectDescriptor
            } else if (mofClassName.getStorageID().equals(MdrStorage.MODEL_CLASS)) {
                List superTypes = XmiUtils.getXmiRefValue(classNode, MdrStorage.MODEL_GENERALIZABLE_ELEMENT_SUPERTYPES);
                for (Iterator st = superTypes.iterator(); st.hasNext();) {
                    String superTypeXmiId = (String) st.next();
                    od.addSupertype(superTypeXmiId);
                    MOFID superProxyId = (MOFID) proxyIdByXmiId.get(superTypeXmiId);
                    if (superProxyId != null) { //is superTypeXmiId proxyClass
                        superclassByClass.add(new SuperClassDescriptor((MOFID) proxyIdByXmiId.get(xmiId), superProxyId)); //add desciptor
                    } else {
                        throw new DebugException("Unable to get MofId for proxy woth XmiId : " + superTypeXmiId);
                    }
                }
            }

            // read components
            Node classContents = XmiUtils.getChildNode(classNode, MdrStorage.MODEL_NAMESPACE_CONTENTS);
            if (classContents != null) {
                XmiUtils.XmiNodeIterator subnodes = new XmiUtils.XmiNodeIterator(classContents);
                createInstances(subnodes, instance.getMofId(), name.getStorageID () + ".", od);
            }
        }
        return instances;
    }

    private void setValues() {
        ObjectDescriptor od;
        for (Iterator it = odByName.values().iterator(); it.hasNext();) {
            od = (ObjectDescriptor) it.next();
            readFeatureValues(od, (ObjectDescriptor) odByName.get(od.getStorable().getMetaObjectId()));
            /*
            try {
                Logger.getDefault().log("- name: " + od.getStorable().getAttribute("name"));
                if (od.getStorable().getAttribute("name") == null) {
                    Logger.getDefault().log("null:" + XmiUtils.getXmiAttrValueAsString(od.getNode(), MOFConstants.MODEL_MODEL_ELEMENT_NAME));
                }
            } catch (Exception e) {
            }
             */
        }
    }
    
/**
 * @param object
 * @param od  */    
    private void readFeatureValues(ObjectDescriptor object, ObjectDescriptor od) {
        Feature feature;
        for (Iterator it = od.getFeatures().iterator(); it.hasNext();) {
            feature = (Feature) it.next();
            try {
                switch (feature.getType()) {
                    case Feature.ATTRIBUTE:
                        Object attrValue = getXmiAttrValue(object.getNode(), feature);
                        String featureName = getObjectName (feature.getName());
                        object.getStorable().setAttribute(featureName, attrValue);
                        break;
                    case Feature.REFERENCE:
                        if (!feature.getName().getStorageID().equals(MdrStorage.MODEL_NAMESPACE_CONTENTS)) {
                            List value = XmiUtils.getXmiRefValue(object.getNode(), feature.getName().getStorageID());
                            for (Iterator refs = value.iterator(); refs.hasNext();) {
//                                try {
                                String ref = (String) refs.next();
                                object.getStorable().addReference(getObjectName(feature.getName()), ((ObjectDescriptor) odByXmiId.get(ref)).getStorable().getMofId());
//                                } catch (StorageBadRequestException e) {
//                                }
                            }
                            break;
                        }
                }
            } catch (StorageException e) {
                throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
            }
        }
        
        for (Iterator it = od.getSupertypes().iterator(); it.hasNext();) {
            readFeatureValues(object, (ObjectDescriptor) odByXmiId.get(it.next()));
        }
    }
    
/**
 * @param node
 * @param feature
 * @return  */    
    private Object getXmiAttrValue(Node node, Feature feature) {
        ObjectDescriptor dataType = feature.getDataType();
        MOFID metaType = dataType.getStorable().getMetaObjectId();
        Object value = null;
        
        if (metaType.getStorageID().equals(MdrStorage.MODEL_STRUCTURE_TYPE)) {
            Map fields = new HashMap();
            Feature field;
            
            Node attrNode = XmiUtils.getChildNode(node, feature.getName().getStorageID());
            Node structNode = XmiUtils.getChildNode(attrNode, dataType.getName().getStorageID());
            
            if (structNode == null) {
                // this is for the case when values for structure fields are stored in xmi.field tags
                XmiUtils.XmiNodeIterator fieldNodes = new XmiUtils.XmiNodeIterator(attrNode, XmiConstants.XMI_FIELD);
                for (Iterator it = dataType.getFeatures().iterator(); it.hasNext();) {
                    Node fieldNode = fieldNodes.next();
                    field = (Feature) it.next();
                    fields.put(getObjectName(field.getName()), decodeStringValue(fieldNode.getFirstChild().getNodeValue(), field.getDataType().getStorable().getMetaObjectId(), field.getDataType()));
                }
            } else {
                // this is for the case when values for structure fields are stored in a same way as values for attributes
                for (Iterator it = dataType.getFeatures().iterator(); it.hasNext();) {
                    field = (Feature) it.next();
                    fields.put(getObjectName(field.getName()), getXmiAttrValue(structNode, field));
                }
            }
            
            try {
                value = StructImpl.newInstance(Class.forName(JMI_PACKAGE_PREFIX + getObjectName(dataType.getName())), new ArrayList(fields.keySet()), fields, parseDots(dataType.getName()));
            } catch (ClassNotFoundException e) {
                throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
            }
        } else {
            if (feature.isMultivalued()) {
                List vals = XmiUtils.getXmiMultiValueAsString(node, feature.getName().getStorageID());
                List x = new ArrayList(vals.size());
                for (Iterator it = vals.iterator(); it.hasNext(); ) {
                    x.add(decodeStringValue((String)it.next(), metaType, dataType));
                }
                value = x;
            } else {
                value = decodeStringValue(XmiUtils.getXmiAttrValueAsString(node, feature.getName().getStorageID()), metaType, dataType);
//                    Logger.getDefault().log("multivalued: " + feature.getName() + ", " + value);
            }
        }
        
        return value;
    }
    
    private Object decodeStringValue(String stringValue, MOFID metaType, ObjectDescriptor dataType) {
        Object value = null;
        
        if (metaType.getStorageID().equals(MdrStorage.MODEL_ENUMERATION_TYPE)) {
            value = EnumResolver.resolveEnum(JMI_PACKAGE_PREFIX + getObjectName (dataType.getName()), stringValue);
        } else if (metaType.getStorageID().equals(MdrStorage.MODEL_PRIMITIVE_TYPE)) {
            String typeName = dataType.getName().getStorageID();
            if (typeName.equals(MdrStorage.DATATYPES_STRING)) {
                value = stringValue;
            } else if (typeName.equals(MdrStorage.DATATYPES_BOOLEAN)) {
                value = Boolean.valueOf(stringValue);
            } else if (typeName.equals(MdrStorage.DATATYPES_INTEGER)) {
                value = new Integer(stringValue);
            } else if (typeName.equals(MdrStorage.DATATYPES_DOUBLE)) {
                value = new Double(stringValue);
            } else if (typeName.equals(MdrStorage.DATATYPES_FLOAT)) {
                value = new Float(stringValue);
            } else if (typeName.equals(MdrStorage.DATATYPES_LONG)) {
                value = new Long(stringValue);
            } else {
                Logger.getDefault().log("unrecognized type: " + typeName);
            }
        } else {
            Logger.getDefault().log("unrecognized metatype: " + metaType);
        }
        
        return value;
    }
    
/**
 * @param xmiId
 * @return  */    
    private String resolveTypeId(String xmiId) {
        String result = xmiId;
        String temp;
        
        while ((temp = (String) aliases.get(result)) != null) {
            result = temp;
        }
        
        return result;
    }

/**
 * @param immediatePackage
 * @param metaObjectId
 * @return  */    
    private StorablePackage createPackage(StorablePackage immediatePackage, MOFID metaObjectId, String metaObjectName) {
        try {
            StorablePackage sp = new StorablePackage(mdrStorage, immediatePackage == null ? null : immediatePackage.getMofId(), metaObjectId, NBMDRepositoryImpl.BOOT_MOF, new HashMap());
            MdrStorage storage = mdrStorage;
            storage.addBootObject(sp.getMofId());
            proxyIdByName.put(metaObjectId, sp.getMofId());
            if (immediatePackage != null) {
                immediatePackage.addPackage(metaObjectName, sp.getMofId());
            }
            return sp;
        } catch ( StorageException e ) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }

/**
 * @param immediatePackage
 * @param metaMofId
 * @return  */    
    private StorableClass createClass(StorablePackage immediatePackage, MOFID metaMofId, 
        String metaObjectName, List attrDescs, List clAttrDescs, boolean classDerived, boolean instanceDerived, 
        boolean isSingleton, boolean isAbstract) {
        
        try {
            StorableClass sc = new StorableClass(mdrStorage, immediatePackage.getMofId(), 
                metaMofId, attrDescs, clAttrDescs, new HashMap(), classDerived, instanceDerived, 
                isSingleton, isAbstract
            );
            MdrStorage storage = mdrStorage;
            storage.addBootClass(sc.getMofId());
            proxyIdByName.put(metaMofId, sc.getMofId());

            //Inserted 040601 by MaS - Start
            proxyById.put( sc.getMofId(), sc );
            //Inserted 040601 by MaS - End

            immediatePackage.addClass(metaObjectName, sc.getMofId());
            return sc;
        } catch (StorageException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }

/**
 * @param immediatePackage
 * @param metaMofId
 * @param endA
 * @param endB
 * @param minA
 * @param minB
 * @param orderedA
 * @param orderedB
 * @param uniqueA
 * @param uniqueB
 * @param aggrA
 * @param aggrB
 * @return  */    
    private StorableAssociation createAssociation(StorablePackage immediatePackage,MOFID metaMofId,String metaObjectName, String endA,String endB,int minA, int maxA, int minB, int maxB,boolean orderedA,boolean orderedB,boolean uniqueA,boolean uniqueB,boolean aggrA, boolean aggrB) {
        try {
            String endALogicalName = metaMofId.getStorageID () + "." + endA;
            String endBLogicalName = metaMofId.getStorageID() + "." + endB;
            
            StorableAssociation sa = new StorableAssociation(
                mdrStorage, immediatePackage.getMofId(), metaMofId, endA, new MOFID (bootSequenceNumber(endALogicalName), endALogicalName), 
                endB, new MOFID(bootSequenceNumber(endBLogicalName), endBLogicalName), Object.class, Object.class, minA, maxA, minB, maxB, 
                orderedA, orderedB, uniqueA, uniqueB, aggrA, aggrB, false, false
            );
            MdrStorage storage = mdrStorage;
            storage.addBootAssociation(sa.getMofId());
            proxyIdByName.put(metaMofId, sa.getMofId());

            //Inserted 040601 by MaS - Start
            proxyById.put( sa.getMofId(), sa );
            //Inserted 040601 by MaS - End


            immediatePackage.addAssociation(metaObjectName, sa.getMofId());
            return sa;
        } catch (StorageException e) {
            throw new DebugException("Storage exception: " + e);
        }
    }

/**
 * @param metaObject
 * @param immediatePackage
 * @param classProxy
 * @return  */    
    private StorableObject createObject(MOFID metaObject, MOFID immediatePackage, MOFID classProxy) {
        try {
            StorableObject so = new StorableObject(mdrStorage, immediatePackage, metaObject, classProxy);
            MdrStorage storage = mdrStorage;
            storage.addBootObject(so.getMofId());
            return so;
        } catch (StorageException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }
    
/**
 * @param mofid
 * @return  */    
    private List parseDots(MOFID mofid) {
        List value = new ArrayList();
        if (!(mofid instanceof MOFID))
            throw new IllegalArgumentException ();
        String attr = mofid.getStorageID();
        int pos;
        while ((pos = attr.indexOf('.')) > -1) {
            if (pos > 0) {
                value.add(attr.substring(0, pos));
            }
            attr = attr.substring(pos + 1);
        }
        value.add(attr);
        return value;
    }

//Inserted 040601 by MaS - Start
/**  Resolves and sets forward references for StorableClass */
    private void resolveReferences() {
        ObjectDescriptor od = null;
        Feature feature = null;
        Object sc = null;
        for (Iterator it = proxyReferences.iterator(); it.hasNext(); ) {
            od = (ObjectDescriptor) it.next();
            sc = proxyById.get(proxyIdByName.get(od.getName()));
            if (sc instanceof StorableClass) {
                for (Iterator features = od.getFeatures().iterator(); features.hasNext(); ) {
                    feature = (Feature) features.next();
                    if (feature.getType() == feature.REFERENCE) {
                        MOFID endName = (MOFID) otherEnds.get(feature.getAssocEndName());
//                        Logger.getDefault().log("registering reference: " + feature.getName() + " for end: " + endName);
                        ((StorableClass)sc).addReferenceDescriptor(feature.getName(), getObjectName(feature.getName()), (MOFID) proxyIdByName.get(feature.getAssocName()), getObjectName(endName));
                    }
                }
            }
        }
    }

/**  Resolves and sets superClasses for StorableClass */
    private void resolveSuperClasses() {
        MOFID classId = null;
        ArrayList superclasses = new ArrayList();
        ArrayList resolved = new ArrayList();
        SuperClassDescriptor scd = null;
        Object sc = null;
        for (Iterator it = superclassByClass.iterator(); it.hasNext(); ) {
            superclasses.clear();
            scd = (SuperClassDescriptor)it.next();
            classId = scd.getClassId();
            sc = proxyById.get( classId );
            if ((!(resolved.contains( classId ))) && (sc instanceof StorableClass) ) {
                resolved.add( classId );
                superclasses.addAll( getAllSuperClasses( classId ) );
                StorableClass storable = (StorableClass) sc;
                for (Iterator foundSuperClasses = superclasses.iterator(); foundSuperClasses.hasNext(); ) {
                    storable.addSuperclass( (MOFID)foundSuperClasses.next() );
                }
            }
        }
    }

/**  Returns all super classes - recursion
 ** @param classId
 */
    private List getAllSuperClasses(MOFID classId) {
        ArrayList result = new ArrayList();
        MOFID superClassId = null;
        List superClassIds = new ArrayList();
        superClassIds.addAll( getSuperClassByClass(classId) );
        
        for (Iterator it = superClassIds.iterator(); it.hasNext(); ) {
            superClassId = (MOFID)it.next();
            if ( (superClassId != null) && (!result.contains( superClassId ) ) ) {
                result.add( superClassId );
                result.addAll( getAllSuperClasses( superClassId ) );
            }
        }
        return result;
    }

/**  Returns super classes for classId form SuperClassDescriptors list
 ** @param classId
 */
    private List getSuperClassByClass(MOFID classId) {
        ArrayList result = new ArrayList();
        SuperClassDescriptor scd = null;
        if (classId != null) {
            for (Iterator it = superclassByClass.iterator(); it.hasNext(); ) {
                scd = (SuperClassDescriptor)it.next();
                if (classId.equals( scd.getClassId() ) ) {
                    result.add( scd.getSuperClassId() );
                }
            }
        }
        return result;
    }
    
    
    private MOFID getPackageName (MOFID bootMOFID) {
        String storageId = bootMOFID.getStorageID ();
        String logicalMOFName = storageId.substring(0, storageId.lastIndexOf('.'));
        return new MOFID (bootSequenceNumber (logicalMOFName), logicalMOFName);
    }
        
    private static String getObjectName (MOFID bootMOFID) {
        String storageId = bootMOFID.getStorageID();
        return storageId.substring(storageId.lastIndexOf('.') + 1);
    }
    
    private long bootSequenceNumber (String logicalMOFName) {
        return logicalMOFName.hashCode ();
    }

/**  SuperClasDescriptor
 * used when collecting superclasses for proxies
 */    
    private class SuperClassDescriptor {
        private MOFID classId = null;
        private MOFID superClassId = null;
        public SuperClassDescriptor(MOFID classId, MOFID superClassId) {
            this.classId = classId;
            this.superClassId = superClassId;
        }
        public MOFID getClassId() {
            return classId;
        }
        public MOFID getSuperClassId() {
            return superClassId;
        }
    }
//Inserted 040601 by MaS - End

    private class ObjectDescriptor {
        private final Node node;
        private final StorableObject storable;
        private final MOFID name;
        private MOFID parentName;
        private List features = null;
        private List supertypes = null;
        
        public ObjectDescriptor(Node node, StorableObject storable, MOFID name) {
            this.node = node;
            this.storable = storable;
            this.name = name;
        }
        
/**
 * @return  */        
        public MOFID getName() {
            return name;
        }
        
/**
 * @return  */        
        public MOFID getParentName() {
            if (parentName == null) {
                String sname = name.getStorageID();
                String logicalMOFName = sname.substring(0, sname.lastIndexOf('.'));
                parentName = new MOFID (bootSequenceNumber(logicalMOFName), logicalMOFName);
            }    
            return parentName;
        }
        
/**
 * @return  */        
        public Node getNode() {
            return node;
        }
        
/**
 * @return  */        
        public StorableObject getStorable() {
            return storable;
        }
        
/**
 * @return  */        
        public List getSupertypes() {
            if (supertypes == null) {
                supertypes = new ArrayList();
            }
            
            return supertypes;
        }
        
/**
 * @return  */        
        public List getFeatures() {
            if (features == null) {
                features = new ArrayList();
            }
            return features;
        }
        
/**
 * @param supertypeXmiId  */        
        public void addSupertype(String supertypeXmiId) {
            getSupertypes().add(supertypeXmiId);
        }
        
/**
 * @param feature  */        
        public void addFeature(Feature feature) {
            getFeatures().add(feature);
        }
    }

    private class Feature {
        public static final int ATTRIBUTE = 0;
        public static final int REFERENCE = 1;
        public static final int FIELD = 2;

        private final MOFID fullName;
        private final int type;
        private final String dataTypeXmiId;
        private final String assocEndXmiId;
        private final boolean multivalued;

        private ObjectDescriptor dataType = null;
        private MOFID assocName = null;
        private MOFID assocEndName = null;

        private void resolveNames() {
            ObjectDescriptor od = (ObjectDescriptor) odByXmiId.get(assocEndXmiId);
            assocEndName = od.getName();
            assocName = od.getParentName();
        }

        public Feature(MOFID fullName, int type, String dataTypeXmiId, String assocEndXmiId, boolean multivalued) {
            this.fullName = fullName;
            this.type = type;
            this.dataTypeXmiId = dataTypeXmiId;
            this.assocEndXmiId = assocEndXmiId;
            this.multivalued = multivalued;
        }

/**
 * @return  */        
        public MOFID getName() {
            return fullName;
        }

/**
 * @return  */        
        
        
        public MOFID getAssocName() {
            if (assocName == null) {
                resolveNames();
            }
            return assocName;
        }
/*
**
* @return  */        
        public MOFID getAssocEndName() {
            if (assocEndName == null) {
                resolveNames();
            }
            return assocEndName;
        }

/**
 * @return  */        
        public ObjectDescriptor getDataType() {
            if (dataType == null) {
                dataType = (ObjectDescriptor) odByXmiId.get(dataTypeXmiId);
            }
            return dataType;
        }
        
/**
 * @return  */        
        public boolean isMultivalued() {
            return multivalued;
        }

/**
 * @return  */        
        public int getType() {
            return type;
        }
    }

}
