/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.storagemodel;

import java.util.*;

import org.netbeans.mdr.util.*;
import org.netbeans.mdr.handlers.BaseObjectHandler;

/**
 *
 * @author Petr Hrebejk, Martin Matula
 */

public class DatatypeDescriptor extends Object {
    private final int memberIDs[];
    private final int memberTypeIDs[];
    private final int ifcIndex;
    private final int typeNameIndex;
    
    private final transient MdrStorage storage;
    
    // name of the interface that the datatype maps to
    private transient String ifcName = null;
    // list of datatype members (enum. literals/structure fields)
    private transient List members = null;
    // list of datatypes of members (applicable for structure fields only)
    private transient List memberTypes = null;
    // fully qualified name of datatype
    // (stored only for structs - FQNs for enums are hardcoded in their impls)
    private transient List typeName = null;
    // storage id
    private transient String storageId = null;

    // used for constructing structure descriptor
    public DatatypeDescriptor(MdrStorage storage, List typeName, List members, List memberTypes, String ifcName, String storageId) {
        this.storage = storage;
        this.ifcName = ifcName;
        this.storageId = storageId;
        this.ifcIndex = storage.storageValues(this.storageId).store(ifcName);
        this.members = Collections.unmodifiableList(members);
        this.memberIDs = new int[members.size()];
        for (int i = 0; i < memberIDs.length; i++) {
            memberIDs[i] = storage.storageValues(this.storageId).store(members.get(i));
        }
        if (typeName == null) {
            this.typeNameIndex = 0;
        } else {
            this.typeName = Collections.unmodifiableList(typeName);
            this.typeNameIndex = storage.storageValues (this.storageId).store(typeName);
        }
        if (memberTypes == null) {
            this.memberTypeIDs = null;
        } else {
            this.memberTypes = Collections.unmodifiableList(memberTypes);
            memberTypeIDs = new int[memberTypes.size()];
            for (int i = 0; i < memberTypeIDs.length; i++) {
                memberTypeIDs[i] = storage.storageValues(this.storageId).store(((Class) memberTypes.get(i)).getName());
            }
        }
    }
    
    // used for creating enumeration descriptor
    public DatatypeDescriptor(MdrStorage storage, List members, String ifcName, String storageId) {
        this(storage, null, members, null, ifcName, storageId);
    }
    
    private DatatypeDescriptor(MdrStorage storage, int typeNameIndex, int[] memberIDs, int[] memberTypeIDs, int ifcIndex, String storageId) {
        this.storage = storage;
        this.typeNameIndex = typeNameIndex;
        this.memberIDs = memberIDs;
        this.memberTypeIDs = memberTypeIDs;
        this.ifcIndex = ifcIndex;
        this.storageId = storageId;
    }
    
    public synchronized List getMembers() {
        if (members == null) {
            members = new ArrayList(memberIDs.length);
            for (int i = 0; i < memberIDs.length; i++) {
                members.add(storage.storageValues(this.storageId).resolve(memberIDs[i]));
            }
            members = Collections.unmodifiableList(members);
        }
        return members;
    }

    public synchronized List getMemberTypes() {
        if (memberTypeIDs == null) return null;
        if (memberTypes == null) {
            memberTypes = new ArrayList(memberTypeIDs.length);
            for (int i = 0; i < memberTypeIDs.length; i++) {
                try {
                    memberTypes.add(BaseObjectHandler.resolveInterface((String) storage.storageValues(this.storageId).resolve(memberTypeIDs[i])));
                } catch (ClassNotFoundException e) {
                    throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
                }
            }
            memberTypes = Collections.unmodifiableList(memberTypes);
        }
        return memberTypes;
    }

    public synchronized List getTypeName() {
        if (typeName == null) {
            typeName = (List) storage.storageValues(this.storageId).resolve(typeNameIndex);
        }
        return typeName;
    }

    public synchronized String getIfcName() {
        if (ifcName == null) {
            ifcName = (String) storage.storageValues(this.storageId).resolve(ifcIndex);
        }
        
        return ifcName;
    }
    
    static DatatypeDescriptor readResolve(java.io.InputStream inputStream, StorableBaseObject storable) throws java.io.IOException {
        int typeNameIndex = IOUtils.readInt(inputStream);
        int[] memberIDs = new int[IOUtils.readInt(inputStream)];
        for (int i = 0; i < memberIDs.length; i++) {
            memberIDs[i] = IOUtils.readInt(inputStream);
        }
        int size = IOUtils.readInt(inputStream);
        int[] memberTypeIDs = null;
        if (size > 0) {
            memberTypeIDs = new int[size - 1];
            for (int i = 0; i < size - 1; i++) {
                memberTypeIDs[i] = IOUtils.readInt(inputStream);
            }
        }
        
        return new DatatypeDescriptor(storable.getMdrStorage(), typeNameIndex, memberIDs, memberTypeIDs, IOUtils.readInt(inputStream), MdrStorage.getStorageIdFromMofId(storable.getMofId()));
    }
    
    void write(java.io.OutputStream outputStream) throws java.io.IOException {
        IOUtils.writeInt(outputStream, typeNameIndex);
        IOUtils.writeInt(outputStream, memberIDs.length);
        for (int i = 0; i < memberIDs.length; i++) {
            IOUtils.writeInt(outputStream, memberIDs[i]);
        }
        if (memberTypeIDs == null) {
            IOUtils.writeInt(outputStream, 0);
        } else {
            IOUtils.writeInt(outputStream, memberTypeIDs.length + 1);
            for (int i = 0; i < memberTypeIDs.length; i++) {
                IOUtils.writeInt(outputStream, memberTypeIDs[i]);
            }
        }
        IOUtils.writeInt(outputStream, ifcIndex);
    }
}
