/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.storagemodel;

import java.util.*;

import org.netbeans.mdr.persistence.*;
import org.netbeans.mdr.util.DebugException;
import org.netbeans.mdr.util.Logger;

/**
 * Instances of this class are immutable, live collections representing
 * results of index access.
 *
 * @author Martin Matula
 */
public class IndexImmutSet implements Collection {
    
    /* --------------------------------------------------------------------- */
    /* -- Attributes (protected) ------------------------------------------- */
    /* --------------------------------------------------------------------- */
    
    protected final MdrStorage storage;
    protected final MultivaluedIndex index;
    protected final Object indexKey;
    
    /* --------------------------------------------------------------------- */
    /* -- Constructor (protected) ------------------------------------------ */
    /* --------------------------------------------------------------------- */
    
    /**
     * @param storage
     * @param index  the index where to look up the members of the current collection
     * @param indexKey the key under which to look up the members of the current collection
     */
    protected IndexImmutSet(MdrStorage storage, MultivaluedIndex index, Object indexKey) {
        this.storage = storage;
        this.index = index;
        this.indexKey = indexKey;
    }

    /* --------------------------------------------------------------------- */
    /* -- Methods for index access ----------------------------------------- */
    /* --------------------------------------------------------------------- */
    
    protected Collection getItems() {
        try {
            return index.getItems(indexKey);
        } catch (StorageException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }
    
    protected Collection getObjects() {
        try {
            return storage.getObjectsFromIndex(index, indexKey);
        } catch (StorageException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }
    
    /* --------------------------------------------------------------------- */
    /* -- implements java.util.Collection ---------------------------------- */
    /* --------------------------------------------------------------------- */
    
    public int size() {
        return getItems().size();
    }
    
    public boolean contains(Object obj) {
        return getItems().contains(obj);
    }
    
    public Iterator iterator() {
        return new IndexImmutIterator(getObjects().iterator());
    }
    
    public boolean isEmpty() {
        return getItems().isEmpty();
    }
    
    public boolean containsAll(Collection collection) {
        return getItems().containsAll(collection);
    }
    
    public Object[] toArray(Object[] obj) {
        return getObjects().toArray(obj);
    }
    
    public Object[] toArray() {
        return getObjects().toArray();
    }
    
   /**
    * operation not supported
    */
    public boolean remove(Object o) {
        throw new UnsupportedOperationException();
    }        
    
   /**
    * operation not supported
    */
    public boolean add(Object obj) {
        throw new UnsupportedOperationException();
    }
    
   /**
    * operation not supported
    */
    public boolean removeAll(Collection collection) {
        throw new UnsupportedOperationException();
    }
    
   /**
    * operation not supported
    */
    public boolean addAll(Collection collection) {
        throw new UnsupportedOperationException();
    }
    
   /**
    * operation not supported
    */
    public boolean retainAll(Collection collection) {
        throw new UnsupportedOperationException();
    }
    
   /**
    * operation not supported
    */
    public void clear() {
        throw new UnsupportedOperationException();
    }
    
    /* --------------------------------------------------------------------- */
    /* -- IndexImmutIterator (inner class) --------------------------------- */
    /* --------------------------------------------------------------------- */
    
    /**
     * Wrapper for the index access iterator diabling modifying accesses.
     */
    protected class IndexImmutIterator implements Iterator {
        protected Iterator innerIterator;
        
        protected IndexImmutIterator(Iterator innerIterator) {
            this.innerIterator = innerIterator;
        }
        
        public boolean hasNext() {
            return innerIterator.hasNext();
        }
        
        public Object next() {
            return innerIterator.next();
        }
        
       /**
        * operation not supported
        */
        public void remove() {
            throw new UnsupportedOperationException();
        }
    }
}
