/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.storagemodel;

import java.util.*;

import org.netbeans.mdr.persistence.*;
import org.netbeans.mdr.util.*;

/** Root class for all storable MOF repository objects.
 *
 * @author Martin Matula, Pavel Buzek
 * @version 0.2
 */
public abstract class StorableBaseObject implements Streamable, StorageClient {

    /** MOFID of this object */
    private org.netbeans.mdr.persistence.MOFID id;
    /** MOFID of this object's metaobject */
    protected org.netbeans.mdr.persistence.MOFID meta; // serialization of this value is handled by subclasses
    /** MOFID of this object's immediate package */
    protected org.netbeans.mdr.persistence.MOFID immediatePackage; //(de)serialization of this value handled by subclasses
    /** MOFID of this object's outermost package */
    private transient org.netbeans.mdr.persistence.MOFID outermostPackage = null; // value inited lazy on @link #getOutermostPackage method first call
    
    private transient StorablePackage immediatePackageObj = null;
    private transient StorablePackage outermostPackageObj = null;
    
    /** Storage reference */
    private transient Storage storage;
    
    /** parent MdrStorage */
    private transient MdrStorage mdrStorage;
    
    // semaphore to avoid calls to "objectStateChanged" before the object is added to the primary index
    protected boolean initFinished;
    
    // slots for storing handler's private information
    private Object slot1 = null;
    private Object slot2 = null;
    private Object slot3 = null;
    private Object slot4 = null;
    private Map propertiesSlot = null;
    // number of used slots (to optimize storage)
    private int slotsCount = 0;

    /** Creates new StorableBaseObject.
     * This default constructor is to be called only when deserializing
     * object from the storage. It is immediatelly followed by call to <code>read</code> method.
     */
    public StorableBaseObject() {
        super();
        initFinished = true;
    }

    StorableBaseObject (MdrStorage mdrStorage, org.netbeans.mdr.persistence.MOFID immediatePackage, org.netbeans.mdr.persistence.MOFID meta) throws StorageException {
        this(mdrStorage, immediatePackage, meta, null);
    }

    StorableBaseObject (MdrStorage mdrStorage, org.netbeans.mdr.persistence.MOFID immediatePackage, org.netbeans.mdr.persistence.MOFID meta, String storageId) throws StorageException {
        initFinished = false;
        if (storageId == null) {
            this.id = mdrStorage.generateMOFID (immediatePackage);
        }
        else {
            this.id = mdrStorage.generateMOFID (storageId);
        }
        this.meta = meta;
        this.immediatePackage = immediatePackage;
        this.mdrStorage = mdrStorage;
        //mdrStorage.addObject(this);
        if (immediatePackage != null) {
            org.netbeans.mdr.persistence.MOFID parentOutermost = mdrStorage.getObject(immediatePackage).getOutermostPackageId();
            if (parentOutermost == null) {
                this.outermostPackage = immediatePackage;
            } else {
                this.outermostPackage = parentOutermost;
            }
        } else {
            this.outermostPackage = this.id;
        }
    }
    
    /** Returns metaobject of this object.
     * @throws StorageException problem in storage
     * @return metaobject
     */    
    public StorableObject getMetaObject() throws StorageException {
        return (StorableObject) getMdrStorage().getObject(meta);
    }
    
    /** Returns immediate package of this object.
     * @throws StorageException problem in storage
     * @return immediate package
     */    
    public StorablePackage getImmediatePackage() throws StorageException {
        if (immediatePackageObj == null) {
            immediatePackageObj = (StorablePackage) getMdrStorage().getObject(immediatePackage);
        }
        return immediatePackageObj;
    }
    
    /** Returns MOFID of immediate package of this object.
     * @return MOFID of immediate package
     */    
    public org.netbeans.mdr.persistence.MOFID getImmediatePackageId() {
        return immediatePackage;
    }
    
    /** Returns MOFID of metaobject of this object.
     * @return MOFID of metaobject
     */    
    public org.netbeans.mdr.persistence.MOFID getMetaObjectId() {
        return meta;
    }
    
    /** Returns MOFID of outermost package of this object.
     * @return MOFID of outermost package
     */    
    public org.netbeans.mdr.persistence.MOFID getOutermostPackageId() {
        if (outermostPackage == null) {
            // value has to be inited
            if (immediatePackage == null) {
                outermostPackage = getMofId(); // the object is proxy of an outermost package
            } else {
                StorableBaseObject temp, pkg;
                try {
                    pkg = mdrStorage.getObject(immediatePackage);
                    do {
                        temp = pkg.getImmediatePackage();
                        if (temp != null) pkg = temp;
                        else break;
                    } while (temp != null);
                    outermostPackage = pkg.getMofId();
                } catch (StorageException e) {
                    throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
                }
            } // if
        } // if
        return outermostPackage;
    }

    /** Returns outermost package of this object.
     * @throws StorageException problem in storage
     * @return outermost package
     */    
    public StorablePackage getOutermostPackage() throws StorageException {        
        if (outermostPackageObj == null) {
            outermostPackageObj = (StorablePackage) getMdrStorage().getObject(getOutermostPackageId());
        }
        return outermostPackageObj;
    }

    /** Returns MOFID of this object.
     * @return MOFID
     */    
    public org.netbeans.mdr.persistence.MOFID getMofId() {
        return id;
    }

    /** Returns string representation of this object.
     * @return string representation of this object
     */    
    public String toString () {
        return getClass().getName() + "("+id.toString()+")";
    }

    /** Compares this object with another object.
     * @param o object to compare
     * @return <code>true</code> if the objects represent the same repository object */    
    public boolean equals(Object o) {
        return (o instanceof StorableBaseObject) && (((StorableBaseObject)o).getMofId().equals(this.getMofId()));
    }
    
    /** Returns object's hashcode
     * @return hashcode
     */
    public int hashCode() {
        return id.hashCode();
    }
    
    protected void deleteRecursive() throws StorageException {
        getMdrStorage().removeObject(this);
    }

    /** Writes this object to an output stream.
     * @param outputStream outputstream to be written to
     */    
    public void write (java.io.OutputStream outputStream) {
//        if (id == null || meta == null || storageID == null || context == null) throw new NullPointerException();
        try {
            IOUtils.writeMOFID (outputStream, id,this.getMdrStorage(), this.id);
            outputStream.write(slotsCount);
            switch (slotsCount) {
                case 5: IOUtils.write(outputStream, this.propertiesSlot, this);
                case 4: IOUtils.write(outputStream, slot4, this);
                case 3: IOUtils.write(outputStream, slot3, this);
                case 2: IOUtils.write(outputStream, slot2, this);
                case 1: IOUtils.write(outputStream, slot1, this);
            }
        } catch (java.io.IOException e) {
            throw (DebugException) Logger.getDefault().annotate(new RuntimeException(), e);
        }
    }
    
    /** Reads this object from the passed input stream.
     * @param inputStream inputstream to be read from
     */    
    public void read (java.io.InputStream inputStream) {
        try {
            id = IOUtils.readMOFID (inputStream, storage);
            slotsCount = inputStream.read();
            switch (slotsCount) {
                case 5: this.propertiesSlot = (Map) IOUtils.read (inputStream, this, null);
                case 4: slot4 = IOUtils.read(inputStream, this, null);
                case 3: slot3 = IOUtils.read(inputStream, this, null);
                case 2: slot2 = IOUtils.read(inputStream, this, null);
                case 1: slot1 = IOUtils.read(inputStream, this, null);
            }
            outermostPackage = null; // not inited
        } catch (java.io.IOException e) {
            throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
        }
    }
    
    public void setStorage(Storage storage) {
        this.storage = storage;
        mdrStorage = MdrStorage.getInstance(storage);
    }
    
    /** Returns reference to parent MdrStorage.
     * @return parent MdrStorage 
     */    
    public MdrStorage getMdrStorage() {
        return mdrStorage;
    }
    
    /** Adds a user specified property into the map of handler properties
     *  @param Object key, the key must be storable by IOUtils.write method
     *  @param Object value, the value must be storable by IOUtils.erite value
     *  @exception IllegalArgumentException is thrown when key or value is null
     *  or is not serializable by IOUtils.write method.
     */
    public void putProperty (Object key, Object value) {
        // commented out - we cannot afford to do such an expensive checks
        if (key == null || value == null /* || !IOUtils.checkObjectValidity (key) || !IOUtils.checkObjectValidity (value)*/)
            throw new IllegalArgumentException ();
        this.objectWillChange();
        if (this.propertiesSlot == null) {
            this.propertiesSlot = new HashMap ();
            slotsCount = 5;
        }
        boolean failed = true;
        this.mdrStorage.getRepository().beginTrans(true);
        try {
            this.propertiesSlot.put (key, value);
            this.objectChanged ();
            failed = false;
        } finally {
            this.mdrStorage.getRepository().endTrans(failed);
        }
    }
    
    /** Removes a user specified property from the map of handler properties
     *  @param Object key
     *  @return associated value or null, if no value is associated with given key
     */
    public Object removeProperty (Object key) {
        if (this.propertiesSlot == null)
            return null;
        boolean failed = true;
        this.mdrStorage.getRepository().beginTrans(true);
        try {
            this.objectWillChange();
            Object result = this.propertiesSlot.remove (key);
            if (result != null) {
                this.objectChanged ();
            }
            if (this.propertiesSlot.size () == 0)
                this.propertiesSlot = null;
            failed = false;
            return result;
        } finally {
            this.mdrStorage.getRepository().endTrans(failed);
        }
    }
    
    /** Returns the value which corresponds to key in the map of handler properties
     *  @param Object key
     *  @return associated value or null, if no value is associated with given key
     */
    public Object getProperty (Object key) {
        if (this.propertiesSlot == null)
            return null;
        else
            return this.propertiesSlot.get (key);
    }
    
    public Object setSlot1(Object value) {
        Object result = slot1;
        objectWillChange();
        slot1 = value;
        if (slotsCount < 1) slotsCount = 1;
        objectChanged();
        return result;
    }
    
    public Object getSlot1() {
        return slot1;
    }
    
    public Object setSlot2(Object value) {
        Object result = slot2;
        objectWillChange();
        slot2 = value;
        if (slotsCount < 2) slotsCount = 2;
        objectChanged();
        return result;
    }
    
    public Object getSlot2() {
        return slot2;
    }
    
    public Object setSlot3(Object value) {
        Object result = slot3;
        objectWillChange();
        slot3 = value;
        if (slotsCount < 3) slotsCount = 3;
        objectChanged();
        return result;
    }
    
    public Object getSlot3() {
        return slot3;
    }
    
    public Object setSlot4(Object value) {
        Object result = slot4;
        objectWillChange();
        slot4 = value;
        if (slotsCount < 4) slotsCount = 4;
        objectChanged();
        return result;
    }
    
    public Object getSlot4() {
        return slot4;
    }
    
    /** Replaces MOFIDs using the passed map. (This method is used when rebuilding metaobjects.)
     * @param table Map of old MOFIDs to new MOFIDs.
     */    
    protected void replaceValues(Map table) {
        meta = (org.netbeans.mdr.persistence.MOFID) table.get(meta);
    }

    /** Signals to the storage that this object will be changed.
     */
    public void objectWillChange() {
        if (initFinished)        
            try {
                getMdrStorage().objectStateWillChange(id);
            } catch (StorageException e) {
                throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
            }
    }
    
    /** Signals to the storage that this object has been changed.
     */
    public void objectChanged() {
        if (initFinished)        
            try {
                getMdrStorage().objectStateChanged(id);
            } catch (StorageException e) {
                throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
            }
    }

}
