/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.mdr.storagemodel.transientimpl;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Properties;
import java.util.Stack;
import org.netbeans.mdr.persistence.*;
import org.netbeans.mdr.storagemodel.MdrStorage;
import org.netbeans.mdr.storagemodel.TransientStorableObject;

/**
 *
 * @author Tomas Zezula
 */
public class TransientStorage implements Storage {
    
    public static final String STORAGE_ID = "GC";  // NOI18N
    
    private MdrStorage mdrStorage;
    private long sequenceNumber;
    private String name;
    private Stack indexTxLog;
    private Stack attrTxLog;
    
    private HashMap indexes;
    
    /** Creates a new instance of TransientStorage */
    public TransientStorage (String name) {
        this.name = name;
        this.sequenceNumber = 1;
        this.indexes = new HashMap ();
        this.indexTxLog = new Stack ();
        this.attrTxLog = new Stack ();
    }
    
    public String getName() {
        return this.name;
    }
    
    public String getStorageId () {
        return STORAGE_ID;
    }
    
    public synchronized long getSerialNumber () {
        return this.sequenceNumber++;
    }
    
    /** This method is not used by transient storage,
     *  the data are not externalized due to its transient behaviour.
     */
    public MOFID readMOFID (java.io.InputStream inputStream) throws StorageException {
        throw new UnsupportedOperationException ();
    }
    
    /** This method is not used by transient storage,
     *  the data are not externalized due to its transient behaviour.
     */
    public void writeMOFID (java.io.OutputStream outputStream, MOFID modId) throws StorageException {
        throw new UnsupportedOperationException ();
    }
    
    public MOFID resolveMOFID (String sMofId) {
        if (sMofId == null)
            return null;
        if (!sMofId.startsWith (STORAGE_ID))
            return null;
        try {
            long serialNumber = Long.parseLong (sMofId.substring(STORAGE_ID.length()+1),16);
            return new MOFID (serialNumber, STORAGE_ID);
        }catch (NumberFormatException nfe) {
            return null;
        }
    }

    /** Returns the primary index in this Storage. There is exactly one primary
     * index in every storage, primary index has valueType STREAMABLE.
     */
    public synchronized SinglevaluedIndex getPrimaryIndex() throws StorageException {
        SinglevaluedIndex primaryIndex = (SinglevaluedIndex) this.indexes.get (TransientObjectResolverIndex.NAME);
        if ( primaryIndex == null) {
            primaryIndex = new TransientObjectResolverIndex ();
            this.indexes.put (TransientObjectResolverIndex.NAME, primaryIndex);
            this.indexTxLog.push ( new CompensatingTransaction.CreateIndexCTx (name, primaryIndex));
        }
        return primaryIndex;
    }
    
    /* Returns true if the Storage of the given name exists */
    public boolean exists() throws StorageException {
        return true;
    }
    
    /** Delete any persistent resources associated with the Storage.
     * @return true if and only if the Storage is succesfully deleted, false otherwise
     */
    public synchronized boolean delete() throws StorageException {
        for (Iterator it = this.indexes.keySet ().iterator (); it.hasNext(); ) {
            this.dropIndex ((String) it.next());
        }
        return true;
    }
    
    /* Create a new Storage instance.  For btree, this creates a new btree repository.  
     * For an RDBMS implementation, this might create a new database, or new tables and
     * indexes within an existing database.  For an implementation which uses a JMI
     * service, it might create a top-level package containing all MDR pacakges.
     * After the repository is created, it is opened.
     * If the repository already exists, and "replace" is false, an exception is
     * thrown. If replace is true, the existing repository is deleted and a new one created.
     * // PENDING: should return primary index or nothing ?
     */
    public void create (boolean replace, ObjectResolver resolver) throws StorageException {
        this.mdrStorage = (MdrStorage) resolver;
    }
    
    /* Open an existing repository.  If createOnNoExist is true, and the Storage
     * doesn't exist, create it and then open it.
     * // PENDING: should return primary index or nothing ?
     */
    public void open(boolean createOnNoExist, ObjectResolver resolver) throws StorageException {
        throw new UnsupportedOperationException ();
    }
    
    /* Close the Storage */
    public void close() throws StorageException {
        this.commitChanges ();
        this.delete ();
    }

    /** Create index that holds exactly one value for each key.
     * @return created index
     * @param name name of the index
     * @param keyType type of keys in the index
     * @param valueType type of values in the index (any type except STREAMABLE)
     */
    public synchronized SinglevaluedIndex createSinglevaluedIndex(String name, EntryType keyType, EntryType valueType) throws StorageException {
        if (this.indexes.get (name) != null)
            throw new StorageBadRequestException ("Index already exists.");
        TransientSinglevaluedIndex tsi =  new TransientSinglevaluedIndex (this.mdrStorage, name, keyType, valueType);
        this.indexes.put (name, tsi);
        this.indexTxLog.push ( new CompensatingTransaction.CreateIndexCTx (name, tsi));
        return tsi;
    }
    /** Create index that holds sorted set of values for each key.
     * @return created index
     * @param name name of the index
     * @param keyType type of keys in the index
     * @param valueType type of values in the index (any type except STREAMABLE)
     * @param unique true if values associated with one key do not contain duplicates
     */
    public synchronized MultivaluedOrderedIndex createMultivaluedOrderedIndex(String name, EntryType keyType, EntryType valueType, boolean unique) throws StorageException {
        if (this.indexes.get (name) != null)
            throw new StorageBadRequestException ("Index already exists.");
        MultivaluedOrderedIndex tmoi = new TransientMultivaluedOrderedIndex (this.mdrStorage, name, keyType, valueType, unique);
        this.indexes.put (name, tmoi);
        this.indexTxLog.push ( new CompensatingTransaction.CreateIndexCTx (name, tmoi));
        return tmoi;
    }
    /** Create index that hold a set of values for each key. Elements in one Multivalued are
     * not sorted. Set does not contain duplicate values.
     * @return created index
     * @param name name of the index
     * @param keyType type of keys in the index
     * @param valueType type of values in the index (any type except STREAMABLE)
     * @param unique true if values associated with one key do not contain duplicates
     */
    public synchronized MultivaluedIndex createMultivaluedIndex(String name, EntryType keyType, EntryType valueType, boolean unique) throws StorageException {
        if (this.indexes.get (name) != null)
            throw new StorageBadRequestException ("Index already exists.");
        MultivaluedIndex tmi = new TransientMultivaluedIndex (this.mdrStorage, name, keyType, valueType, unique);
        this.indexes.put (name, tmi);
        this.indexTxLog.push ( new CompensatingTransaction.CreateIndexCTx (name, tmi));
        return tmi;
    }

    /** Retrieve index by name.
     * @param name name of the index
     * @return index of the specified name
     */
    public synchronized Index getIndex(String name) throws StorageException {
        return (Index) this.indexes.get (name);
    }
    /** Retrieve index by name.
     * @param name name of the index
     * @return index of the specified name and type
     */
    public synchronized SinglevaluedIndex getSinglevaluedIndex(String name) throws StorageException {
        return (SinglevaluedIndex) this.getIndex (name);
    }
    /** Retrieve index by name.
     * @param name name of the index
     * @return index of the specified name and type
     */
    public synchronized MultivaluedIndex getMultivaluedIndex(String name) throws StorageException {
        return (MultivaluedIndex) this.getIndex (name);
    }
    /** Retrieve index by name.
     * @param name name of the index
     * @return index of the specified name and type
     */
    public synchronized MultivaluedOrderedIndex getMultivaluedOrderedIndex(String name) throws StorageException {
        return (MultivaluedOrderedIndex) this.getIndex (name);
    }
    /** Delete index.
     * @param name name of the index
     */
    public synchronized void dropIndex(String name) throws StorageException {
        Object index = this.indexes.remove (name);
        this.indexTxLog.push ( new CompensatingTransaction.DropIndexCTx (name, index));
    }
    
    public void objectStateWillChange (Object key) throws StorageException {
        // do nothing
    }
    
    /** Notify the Storage that state of the object associated with this key
     * was changed.  This must be called after the change is made to ensure 
     * that the changed state is comitted correctly.
     * @param key key of the object that was changed and must be saved
     */
    public void objectStateChanged (Object key) throws StorageException {
        this.attrTxLog.push (key);
    }

    /** Save all objects changed since this method was last call.
     * This operation implements transactions on the storage.
     * It must either whole complete or whole fail.
     */
    public void commitChanges() throws StorageException {
        this.indexTxLog.clear ();
        Iterator it = null;
        synchronized (this) {
            it = ((HashMap)this.indexes.clone ()).values ().iterator ();
        }
        while (it.hasNext ()) {
            ((TransactionalIndex)it.next()).commit ();
        }
        while (!this.attrTxLog.empty ()) {
            MOFID id = (MOFID) this.attrTxLog.pop ();
            SinglevaluedIndex pi = this.getPrimaryIndex ();
            TransientStorableObject tso = (TransientStorableObject) pi.getIfExists(id);
            if (tso != null)
                tso.commit ();
        }
    }

    /** Discard all changes since commitChanges() method was last call.
     * This operation implements transactions on the storage.
     * It must either whole complete or whole fail.
	 * Note that, after this method completes, the persistent MDR and
	 * any modified objects in memory are inconsistent, so it should
	 * be followed shortly by program exit.
     */
    public void rollBackChanges () throws StorageException {
        while (!this.indexTxLog.empty ()) {
            CompensatingTransaction ctx = (CompensatingTransaction) this.indexTxLog.pop ();
            ctx.perform (this.indexes);
        }
        Iterator it = null;
        synchronized (this) {
            it = ((HashMap)this.indexes.clone ()).values ().iterator ();
        }
        while (it.hasNext ()) {
            ((TransactionalIndex)it.next ()).rollBack ();
        }
        while (!this.attrTxLog.empty ()) {
            MOFID id = (MOFID) this.attrTxLog.pop ();
            SinglevaluedIndex pi = this.getPrimaryIndex ();
            TransientStorableObject tso = (TransientStorableObject) pi.getIfExists(id);
            if (tso != null)
                tso.rollBack ();
        }
    }
    
    /**
     * Performs operations needed on exit.
     */
    public void shutDown() throws StorageException {
    }
    
    
    
}
