/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.util;

import java.io.*;
import java.lang.reflect.Array;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.*;
import java.lang.reflect.Field;

import javax.jmi.reflect.*;
import javax.jmi.model.*;

import org.netbeans.mdr.NBMDRepositoryImpl;
import org.netbeans.mdr.handlers.gen.TagSupport;

import org.netbeans.mdr.storagemodel.*;
import org.netbeans.mdr.persistence.MOFID;
import org.netbeans.mdr.persistence.Storage;
import org.netbeans.mdr.persistence.StorageException;
import org.netbeans.mdr.util.DebugException;
import org.netbeans.mdr.handlers.*;

/**
 *
 * @author Martin Matula
 * @version
 */
public class IOUtils extends Object {
    
    public static final int T_NULL = 0;
    public static final int T_STRING = 1;
    public static final int T_BOOLEAN = 2;
    public static final int T_MAP = 3;
    public static final int T_INTEGER = 4;
    public static final int T_COLLECTION = 5;
    public static final int T_STRUCT = 6;
    public static final int T_ENUM = 7;
    //    public static final int T_SERIALIZABLE = 8;
    public static final int T_STORABLE = 9;
    public static final int T_MOF_REFERENCE = 10;
    public static final int T_LIST_IMMUTABLE = 11;
    public static final int T_LIST_MUTABLE = 12;
    public static final int T_LIST_U_IMMUTABLE = 13;
    public static final int T_LIST_U_MUTABLE = 14;
    public static final int T_COLL_U_MUTABLE = 15;
    public static final int T_COLL_MUTABLE = 16;
    public static final int T_CLASS = 17;
    public static final int T_FLOAT = 18;
    public static final int T_DOUBLE = 19;
    public static final int T_OBJECT = 20;
    public static final int T_LONG = 21;
    public static final int T_SHORT = 22;
    public static final int T_MOFID = 23;
    public static final int T_ARRAY = 24;
    public static final int T_BYTE = 25;
    
    /** Creates new IOUtils */
    public IOUtils() {
    }
    
    public static void write(OutputStream outputStream, Object object) throws IOException {
        write(outputStream, object, null);
    }
    
    /**
     * Writes an integer to the output stream. It tries to optimize for space a little:
     * small values are stored just as one byte. Values <= 0xffff are stored as
     * 3 bytes, little-endian and other values as 5 bytes, little-endian
     */
    public static void writeInt(OutputStream outputStream, int val) throws IOException {
        if ((val & 0x7F) == val) {
            outputStream.write((byte)val);
        } else {
            byte [] arr = new byte [5];
            arr[1] = (byte)val;
            arr[2] = (byte)(val >>> 8);
            if ((val >>> 16) == 0) {
                arr[0] = (byte)(T_SHORT | 0x80);
                outputStream.write(arr, 0, 3);
            } else {
                arr[3] = (byte)(val >>> 16);
                arr[4] = (byte)(val >>> 24);
                arr[0] = (byte)(T_INTEGER | 0x80);
                outputStream.write(arr, 0, 5);
            }
        }
    }
    
    public static void writeByte(OutputStream outputStream, byte val) throws IOException {
        outputStream.write(val);
    }
    
    public static void writeLong(OutputStream outputStream, long val) throws IOException {
        if ((val & 0x7F) == val) {
            outputStream.write((byte)val);
        } else if ((val >>> 32) == 0) {
            outputStream.write(T_INTEGER | 0x80);
            outputStream.write((byte)val);
            outputStream.write((byte)(val >>> 8));
            outputStream.write((byte)(val >>> 16));
            outputStream.write((byte)(val >>> 24));
        } else {
            outputStream.write(T_LONG | 0x80);
            outputStream.write((byte)val);
            outputStream.write((byte)(val >>> 8));
            outputStream.write((byte)(val >>> 16));
            outputStream.write((byte)(val >>> 24));
            outputStream.write((byte)(val >>> 32));
            outputStream.write((byte)(val >>> 40));
            outputStream.write((byte)(val >>> 48));
            outputStream.write((byte)(val >>> 56));
        }
    }
    
    public static void writeBoolean(OutputStream outputStream, boolean val) throws IOException {
        outputStream.write(val ? 1 : 0);
    }
    
    public static void writeString(OutputStream outputStream, String val) throws IOException {
        if (val == null) {
            writeInt(outputStream, 0);
            return;
        }
        int len = val.length();
        char [] arr = new char[len];
        val.getChars(0, len, arr, 0);
        writeInt(outputStream, len + 1);
        for (int i = 0; i < len; i++) {
            char ch = arr[i];
            if (ch <= 0x7f) {
                outputStream.write((byte) ch);
            } else if (ch <= 0x7ff) {
                outputStream.write((byte) (0xc0 | (ch >> 6)));
                outputStream.write((byte) (0x80 | (ch & 0x3f)));
            } else {
                outputStream.write((byte) (0xe0 | (ch >> 12)));
                outputStream.write((byte) (0x80 | ((ch >> 6) & 0x3f)));
                outputStream.write((byte) (0x80 | (ch & 0x3f)));
            }
        }
    }

    public static void writeMOFID (OutputStream outputStream, MOFID mofId, MdrStorage mdrStorage, MOFID storableID) throws IOException {
        Storage storage = mdrStorage.getStorageByMofId(storableID);
        writeMOFID (outputStream, mofId, storage);
    }
    
    public static void writeMOFID (OutputStream outputStream, MOFID mofId, Storage storage) throws IOException {
        try {
            storage.writeMOFID (outputStream, mofId);
        }catch (StorageException se) {
            throw (IOException) Logger.getDefault().annotate(new IOException(se.getMessage()), se);
        }
    }
    
    public static void write(OutputStream outputStream, Object object, StorableBaseObject storable) throws IOException {
        if (object == null) {
            outputStream.write(T_NULL);
            
        } else if (object instanceof String) {
            outputStream.write(T_STRING);
            writeString(outputStream, (String)object);
        } else if (object instanceof Integer) {
            outputStream.write(T_INTEGER);
            writeInt(outputStream, ((Integer)object).intValue());
        } else if (object instanceof Boolean) {
            outputStream.write(T_BOOLEAN);
            writeBoolean(outputStream, ((Boolean) object).booleanValue());
        } else if (object instanceof Float) {
            outputStream.write(T_FLOAT);
            writeInt(outputStream, Float.floatToIntBits(((Float) object).floatValue()));
        } else if (object instanceof Double) {
            outputStream.write(T_DOUBLE);
            writeLong(outputStream, Double.doubleToLongBits(((Double) object).doubleValue()));
        } else if (object instanceof Long) {
            outputStream.write(T_LONG);
            writeLong(outputStream, ((Long) object).longValue());
        } else if (object instanceof Byte) {
            outputStream.write(T_BYTE);
            writeByte(outputStream, ((Byte)object).byteValue());
        } else if (object instanceof Map) {
            outputStream.write(T_MAP);
            Map temp = (Map) object;
            writeInt(outputStream, temp.size());
            Map.Entry key;
            for (Iterator it = temp.entrySet().iterator(); it.hasNext();) {
                key = (Map.Entry) it.next();
                write(outputStream, key.getKey(), storable);
                write(outputStream, key.getValue(), storable);
            }
        } else if (object.getClass().isArray()) {
            outputStream.write(T_ARRAY);
            int length = Array.getLength(object);
            writeInt(outputStream, length);
            if (object instanceof int[]) {
                outputStream.write(T_INTEGER);
                int[] a = (int[]) object;
                for (int i = 0; i < length; i++) {
                    writeInt(outputStream, a[i]);
                }
            } else if (object instanceof byte[]) {
                outputStream.write(T_BYTE);
                byte[] a = (byte[]) object;
                for (int i = 0; i < length; i++) {
                    writeByte(outputStream, a[i]);
                }
            }
            
        } else if (object instanceof AttrCollection) {
            if (object instanceof AttrUList) {
                outputStream.write(T_LIST_U_MUTABLE);
            } else if (object instanceof AttrList) {
                outputStream.write(T_LIST_MUTABLE);
            } else if (object instanceof AttrSet) {
                outputStream.write(T_COLL_U_MUTABLE);
            } else {
                outputStream.write(T_COLL_MUTABLE);
            }
            ((AttrCollection) object).write(outputStream);
            
        } else if (object instanceof AttrImmutList) {
            if (storable == null) Logger.getDefault().notify(Logger.INFORMATIONAL, new DebugException());
            if (object instanceof AttrImmutUList) {
                outputStream.write(T_LIST_U_IMMUTABLE);
            } else {
                outputStream.write(T_LIST_IMMUTABLE);
            }
            ((AttrImmutList) object).write(outputStream, storable);
            
        } else if (object instanceof Collection) {
            outputStream.write(T_COLLECTION);
            Collection col = (Collection) object;
            writeInt(outputStream, col.size());
            for (Iterator it = col.iterator(); it.hasNext();) {
                write(outputStream, it.next(), storable);
            }
            
        } else if (object instanceof RefStruct) {
            outputStream.write(T_STRUCT);
            RefStruct struct = (RefStruct) object;
            MdrStorage storage = storable.getMdrStorage();
            if (storage == null) Logger.getDefault().notify(Logger.INFORMATIONAL, new DebugException());
            writeInt(outputStream, storage.values(storable.getMofId()).indexOf(struct.refTypeName()));
            List fields = struct.refFieldNames();
            writeInt(outputStream, fields.size());
            for (Iterator it = fields.iterator(); it.hasNext();) {
                String field = (String) it.next();
                writeInt(outputStream, storage.values(storable.getMofId()).indexOf(field));
                write(outputStream, struct.refGetValue(field), storable);
            }
            
        } else if (object instanceof RefEnum) {
            MdrStorage storage = storable.getMdrStorage();
            if (storage == null) Logger.getDefault().notify(Logger.INFORMATIONAL, new DebugException());
            outputStream.write(T_ENUM);
            writeInt(outputStream, storage.values(storable.getMofId()).indexOf(object.toString()));
            
            // [PENDING] should be removed from here
        } else if (object instanceof StorableClass.AttributeDescriptor) {
            throw new DebugException("AttributeDescriptor should serialized elswhere.");
        } else if (object instanceof StorableClass.ReferenceDescriptor) {
            outputStream.write(T_MOF_REFERENCE);
            writeMOFID (outputStream, ((StorableClass.ReferenceDescriptor) object).getMofId(),storable.getMdrStorage(), storable.getMofId());
            writeMOFID (outputStream, ((StorableClass.ReferenceDescriptor) object).getAssociationId(),storable.getMdrStorage(), storable.getMofId());
            writeString(outputStream, ((StorableClass.ReferenceDescriptor) object).getEndName());
            
        } else if (object instanceof RefObject) {
            outputStream.write(T_MOFID);
            writeMOFID (outputStream, ((BaseObjectHandler) object)._getDelegate().getMofId(),storable.getMdrStorage(), storable.getMofId());
            
        } else if (object instanceof Class) {
            MdrStorage storage = storable.getMdrStorage();
            if (storage == null) Logger.getDefault().notify(Logger.INFORMATIONAL, new DebugException());
            outputStream.write(T_CLASS);
            writeInt(outputStream, storage.values(storable.getMofId()).indexOf(((Class) object).getName()));
/*
        } else if (object instanceof Serializable) {
            outputStream.write(T_SERIALIZABLE);
            new ObjectOutputStream(outputStream).writeObject(object);
 */
        } else if (object instanceof MOFID) {
            outputStream.write (T_MOFID);
            writeMOFID (outputStream, (MOFID)object, storable.getMdrStorage(), storable.getMofId());
        } else if (object instanceof ValueWrapper) {
            write(outputStream, ((ValueWrapper) object).getValue(), storable);
        } else if (object instanceof Storable) {
            outputStream.write(T_STORABLE);
            write(outputStream, object.getClass(), storable);
            ((Storable) object).write(outputStream, storable);
        } else {
            throw new IOException("Unsupported type of object (object must be serializable): " + object.getClass().getName());
        }
    }
    
    public static Object read(InputStream inputStream) throws IOException {
        return read(inputStream, null, null);
    }
    
    public static Object read(InputStream inputStream, StorableBaseObject storable) throws IOException {
        return read(inputStream, storable, null);
    }
    
    public static boolean readBoolean(InputStream is) throws IOException {
        return (is.read() == 1);
    }
    
    public static byte readByte(InputStream is) throws IOException {
        return (byte)is.read();
    }
    
    public static int readInt(InputStream inputStream) throws IOException {
        int type = inputStream.read();
        if (type <= 0x7f)
            return type;
        switch (type) {
            case T_SHORT | 0x80: {
                int ch1, ch2;
                ch1 = inputStream.read();
                ch2 = inputStream.read();
                return (ch2 << 8) | ch1;
            }
            case T_INTEGER | 0x80: {
                int ch1, ch2, ch3, ch4;
                ch1 = inputStream.read();
                ch2 = inputStream.read();
                ch3 = inputStream.read();
                ch4 = inputStream.read();
                return (ch4 << 24) | (ch3 << 16) | (ch2 << 8) | ch1;
            }
            default:
                throw new IOException("Unknown int format: " + type);
        }
    }
    
    public static long readLong(InputStream inputStream) throws IOException {
        int t = inputStream.read();
        if (t <= 0x7f)
            return t;
        switch (t) {
            case T_INTEGER | 0x80: {
                int ch1, ch2, ch3, ch4;
                ch1 = inputStream.read();
                ch2 = inputStream.read();
                ch3 = inputStream.read();
                ch4 = inputStream.read();
                return (ch4 << 24) | (ch3 << 16) | (ch2 << 8) | ch1;
            }
            case T_LONG | 0x80: {
                int ch1, ch2, ch3, ch4;
                long v_low,v_high;
                
                ch1 = inputStream.read();
                ch2 = inputStream.read();
                ch3 = inputStream.read();
                ch4 = inputStream.read();
                v_low = (ch4 << 24) | (ch3 << 16) | (ch2 << 8) | ch1;
                ch1 = inputStream.read();
                ch2 = inputStream.read();
                ch3 = inputStream.read();
                ch4 = inputStream.read();
                v_high = ((ch4 << 24) | (ch3 << 16) | (ch2 << 8) | ch1);
                return (v_high << 32) | (v_low & 0xFFFFFFFFL);
            }
            default:
                throw new IOException("Unknown int format: " + t);
        }
    }
    
    public static String readString(InputStream inputStream) throws IOException {
        int length = readInt(inputStream);
        if (length == 0)
            return null;
        else if (--length == 0)
            return "";
        StringBuffer sb = new StringBuffer(length);
        do {
            int b = inputStream.read() & 0xff;
            if (b >= 0xe0) {
                b = (b & 0x0f) << 12;
                b |= (inputStream.read() & 0x3f) << 6;
                b |= inputStream.read() & 0x3f;
            } else if (b >= 0xc0) {
                b = (b & 0x1f) << 6;
                b |= inputStream.read() & 0x3f;
            }
            sb.append((char)b);
        } while (--length > 0);
        return sb.toString().intern();
    }

    public static MOFID readMOFID (InputStream inputStream, MdrStorage mdrStorage, MOFID storableID) throws IOException {
        Storage storage = mdrStorage.getStorageByMofId(storableID);
        return readMOFID (inputStream, storage);
    }
    
    public static MOFID readMOFID (InputStream inputStream, Storage storage) throws IOException {
        try {
            return storage.readMOFID (inputStream);
        } catch (StorageException se) {
            throw (IOException) Logger.getDefault().annotate(new IOException(se.getMessage()), se);
        }
    }
    
    public static Object read(InputStream inputStream, StorableBaseObject storable, String className) throws IOException {
        int type = inputStream.read();
        return read(inputStream, storable, className, type);
    }
    
    public static Object read(InputStream inputStream, StorableBaseObject storable, String className, int type) throws IOException {
        switch (type) {
            case T_NULL:
                return null;
            case T_STRING:
                return readString(inputStream);
            case T_INTEGER:
                return new Integer(readInt(inputStream));
            case T_LONG:
                return new Long(readLong(inputStream));
            case T_FLOAT:
                return new Float(Float.intBitsToFloat(readInt(inputStream)));
            case T_DOUBLE:
                return new Double(Double.longBitsToDouble(readLong(inputStream)));
            case T_BYTE:
                return new Byte(readByte(inputStream));
            case T_BOOLEAN: {
                return readBoolean(inputStream) ? Boolean.TRUE : Boolean.FALSE;
            } case T_ARRAY: {
                int length = readInt(inputStream);
                int elementType = inputStream.read();
                switch (elementType) {
                    case T_INTEGER: 
                        int[] result = new int[length];
                        for (int i = 0; i < length; i++) {
                            result[i] = readInt(inputStream);
                        }
                        return result;
                    case T_BYTE:
                        byte[] res = new byte[length];
                        for (int i = 0; i < length; i++) {
                            res[i] = readByte(inputStream);
                        }
                        return res;
                    default:
                        throw new DebugException();
                }
            } case T_MAP: {
                int size = readInt(inputStream);
                Map result = new HashMap(size);
                for (int i = 0; i < size; i++) {
                    result.put(read(inputStream, storable), read(inputStream, storable));
                }
                return result;
            } case T_LIST_IMMUTABLE:
            case T_LIST_U_IMMUTABLE: {
                if (storable == null) Logger.getDefault().notify(Logger.INFORMATIONAL, new DebugException());
                AttrImmutList result;
                if (type == T_LIST_IMMUTABLE) {
                    result = new AttrImmutList();
                } else {
                    result = new AttrImmutUList();
                }
                result.read(inputStream, storable);
                return result;
            } case T_LIST_MUTABLE:
            case T_LIST_U_MUTABLE:
            case T_COLL_MUTABLE:
            case T_COLL_U_MUTABLE: {
                AttrCollection result;
                if (type == T_LIST_MUTABLE) {
                    result = new AttrList();
                } else if (type == T_LIST_U_MUTABLE) {
                    result = new AttrUList();
                } else if (type == T_COLL_MUTABLE) {
                    result = new AttrCollection();
                } else {
                    result = new AttrSet();
                }
                result.read(inputStream, (StorableFeatured) storable);
                return result;
            } case T_COLLECTION: {
                int size = readInt(inputStream);
                java.util.List result = new java.util.ArrayList(size);
                for (int i = 0; i < size; i++) {
                    result.add(read(inputStream, storable));
                }
                return result;
            } case T_STRUCT: {
                if (storable == null) Logger.getDefault().notify(Logger.INFORMATIONAL, new DebugException());
                MdrStorage storage = storable.getMdrStorage();
                List qualifiedName = (List) storage.values(storable.getMofId()).resolve(readInt(inputStream));
                int size = readInt(inputStream);
                List fields = new ArrayList(size);
                Map values = new HashMap(size, 1);
                for (int i = 0; i < size; i++) {
                    Object field = storage.values(storable.getMofId()).resolve(readInt(inputStream));
                    fields.add(field);
                    values.put(field, read(inputStream, storable));
                }
                
                Class clazz;
                
                if (className == null) {
                    StorableObject struct = resolveByFQN(storable, qualifiedName);
                    className = TagSupport.getDataTypeName(struct);
                }
                
                try {
                    clazz = BaseObjectHandler.resolveInterface(className);
                } catch (ClassNotFoundException e) {
                    throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
                }
                
                return StructImpl.newInstance(clazz, fields, values, qualifiedName);
            } case T_ENUM: {
                if (storable == null) Logger.getDefault().notify(Logger.INFORMATIONAL, new DebugException());
                return EnumResolver.resolveEnum(className, (String) storable.getMdrStorage().values(storable.getMofId()).resolve(readInt(inputStream)));
            } case T_MOF_REFERENCE: {
                if (storable == null) Logger.getDefault().notify(Logger.INFORMATIONAL, new DebugException());
                return ((StorableClass) storable).new ReferenceDescriptor(readMOFID(inputStream, storable.getMdrStorage(), storable.getMofId()), readMOFID(inputStream, storable.getMdrStorage(), storable.getMofId()), readString(inputStream));
            } case T_CLASS: {
                if (storable == null) Logger.getDefault().notify(Logger.INFORMATIONAL, new DebugException());
                try {
                    return BaseObjectHandler.resolveInterface((String) storable.getMdrStorage().values(storable.getMofId()).resolve(readInt(inputStream)));
                } catch (ClassNotFoundException e) {
                    throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
                }
            } case T_MOFID: {
                return readMOFID (inputStream, storable.getMdrStorage (), storable.getMofId());
            } case T_STORABLE: {
                Class cls = (Class) read(inputStream, storable);
                try {
                    Method method = cls.getMethod("read", new Class[] {InputStream.class, StorableBaseObject.class});
                    return method.invoke(null, new Object[] {inputStream, storable});
                } catch (NoSuchMethodException e) {
                    throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
                } catch (SecurityException e) {
                    throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
                } catch (IllegalAccessException e) {
                    throw (DebugException) Logger.getDefault().annotate(new DebugException(), e);
                } catch (InvocationTargetException e) {
                    if (e.getTargetException() instanceof IOException) {
                        throw (IOException) e.getTargetException();
                    }
                    throw (DebugException) Logger.getDefault().annotate(new DebugException(), e.getTargetException());
                }
            } default: {
                throw new IOException("Object type not recognized: " + type);
            }
        }
    }
    
    private static StorableObject resolveByFQN(StorableBaseObject obj, List qualifiedName) {
        try {
            RefPackage pkg = (RefPackage) obj.getMdrStorage().getRepository().getHandler(obj.getMetaObject().getOutermostPackage());
            RefObject result = resolveByFQN(pkg , qualifiedName);
            return (StorableObject) (result == null ? null : ((BaseObjectHandler) result)._getDelegate());
        } catch (Exception e) {
            Logger.getDefault().notify(e);
            return null;
        }
    }
    
    private static RefObject resolveByFQN(RefPackage pkg, List qualifiedName) {
        qualifiedName = new ArrayList(qualifiedName);
        String name = (String) qualifiedName.remove(0);
        MofPackage outer = findPackage(pkg, name);
        if (outer == null) {
            for (Iterator it = pkg.refAllPackages().iterator(); it.hasNext() && outer == null;) {
                outer = findPackage((RefPackage) it.next(), name);
            }
        }
        if (outer == null) return null;
        try {
            return outer.resolveQualifiedName(qualifiedName);
        } catch (NameNotResolvedException e) {
            return null;
        }
    }
    
    private static MofPackage findPackage(RefPackage pkg, String name) {
        for (Iterator it = ((ModelPackage) pkg).getMofPackage().refAllOfType().iterator(); it.hasNext();) {
            MofPackage result = (MofPackage) it.next();
            if (name.equals(result.getName()) && result.refImmediateComposite() == null) {
                return result;
            }
        }
        return null;
    }
    
    /** Implementors must also declare: <br>
     * <pre>public static Object read(InputStream inputStream, StorableBaseObject storable) throws IOException</pre>
     */
    public interface Storable {
        public void write(OutputStream outputStream, StorableBaseObject storable) throws IOException;
    }

    /* commented out - we cannot affort to make such an expensive checks
    public static boolean checkObjectValidity(Object obj) {
        if (obj == null
        || obj instanceof String
        || obj instanceof Integer
        || obj instanceof Boolean
        || obj instanceof Float
        || obj instanceof Double
        || obj instanceof Long
        || obj instanceof RefObject
        || obj instanceof AttrCollection
        || obj instanceof AttrImmutList
        || obj instanceof RefStruct
        || obj instanceof RefEnum
        || obj instanceof StorableClass.ReferenceDescriptor
        || obj instanceof Class)
            return true;
        else if (obj instanceof Map) {
            Iterator keyIterator = ((Map)obj).keySet().iterator();
            Iterator valueIterator = ((Map)obj).values().iterator();
            while (keyIterator.hasNext()) {
                if (!checkObjectValidity(keyIterator.next())
                || !checkObjectValidity(valueIterator.next()))
                    return false;
            }
            return true;
        }
        else if (obj instanceof Collection) {
            for (Iterator it = ((Collection)obj).iterator(); it.hasNext();) {
                if (!checkObjectValidity(it.next()))
                    return false;
            }
            return true;
        }
        else return false;
    }
     */
}
