/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.util;

import java.lang.ref.Reference;
import java.lang.ref.WeakReference;
import java.security.AllPermission;
import java.security.PermissionCollection;
import java.security.Permissions;
import java.security.ProtectionDomain;
import java.util.HashMap;
import java.util.Map;
import java.util.WeakHashMap;

/**
 * Static utilities for the loading of implementation classes.
 *
 * @author  Martin Matula
 * @version 
 */
public abstract class ImplClass {
    
    /* Suffix for the names of implementation classes. */
    protected static final String classNameSuffix = "$Impl";
    private static final ProtectionDomain pd;

    private static final Map loaderToCache = new WeakHashMap(3);
    private static final Object pendingGenerationMarker = new Object();

    static {
        PermissionCollection pc = new Permissions();
        pc.add(new AllPermission());
        pd = new ProtectionDomain(null, pc);
    }
    
    public static String getClassNameSuffix() {
        return classNameSuffix;
    }

    /**
     * Checks if <code>loader</code> may load the interface class <code>ifc</code>.
     *
     * @throws IllegalArgumentException if <code>loader</code> cannot load
     *    <code>ifc</code> or if <code>ifc</code> is not an interface
     */
    protected static void check(ClassLoader loader, Class ifc) throws IllegalArgumentException {
        Class interfaceClass = null;

        try {
            interfaceClass = Class.forName(ifc.getName(), false, loader);
        } catch (ClassNotFoundException e) {
        }
        if (interfaceClass != ifc) {
            throw new IllegalArgumentException(ifc + " is not visible from class loader");
        }

        if (!interfaceClass.isInterface()) {
            throw new IllegalArgumentException(interfaceClass.getName() + " is not an interface");
        }
    }
    
    /**
     * Returns the name of the implementation class for interface <code>ifc</code>.
     */
    protected static String getName(Class ifc) {
        return ifc.getName() + classNameSuffix;
    }
    
    /**
     * Returns and, if necessary, creates the cache for <code>loader</code>.
     */
    protected static Map getLoaderCache(ClassLoader loader) {
	Map cache;
	synchronized (loaderToCache) {
	    cache = (Map) loaderToCache.get(loader);
	    if (cache == null) {
		cache = new HashMap(3);
		loaderToCache.put(loader, cache);
	    }
	}
        
        return cache;
    }
    
    /**
     * Returns the class <code>className</code> from the given cache. If
     * the class is not yet contained in the cache, the method returns
     * <code>null</code> indicating that the caller shall create the class
     * and put it into the cache. In this case a mark is set to prevent other
     * threads from creating the class also. If a second thread enters the
     * method before the class is created and put into the cache, it blocks,
     * until class creation is performed. As a consequence the caller
     * <em>must</em> try to create the class and then call
     * {@link releaseCache(Map, Class, String) releaseCache} either with the
     * newly created class or <code>null</code> to remove the mark set.
     *
     * @param cache the cache from where to retrieve the class
     * @param ifc (not needed)
     * @param className name of class implementing <code>ifc</code>
     * @return the class requested or <code>null</code> if the class still
     *     has to be put into the cache
     */
    protected static Class getFromCache(Map cache, Class ifc, String className) {
        Class implClass = null;
        
	synchronized (cache) {
	    do {
		Object value = cache.get(className);
		if (value instanceof Reference) {
		    implClass = (Class) ((Reference) value).get();
		}
		if (implClass != null) {
		    return implClass;
		} else if (value == pendingGenerationMarker) {
		    try {
			cache.wait();
		    } catch (InterruptedException e) {
		    }
		    continue;
		} else {
		    cache.put(className, pendingGenerationMarker);
		    break;
		}
	    } while (true);
	}
        
        return null;
    }

    /** Puts <code>implClass</code> into <code>cache</code> using key
     *  <code>className</code> and notifies all waiters on <code>cache</code>.
     *  If <code>implClass == null</code> the entry for key <code>className</code>
     *  is removed from the cache.
     */
    protected static void releaseCache(Map cache, Class implClass, String className) {
        synchronized (cache) {
            if (implClass != null) {
                cache.put(className, new WeakReference(implClass));
            } else {
                cache.remove(className);
            }
            cache.notifyAll();
        }
    }

//    protected static Class defineClass(ClassLoader loader, String proxyName, byte[] b) throws IllegalArgumentException {
//        try {
//            Method method = ClassLoader.class.getDeclaredMethod("defineClass", new Class[] {String.class, byte[].class, Integer.TYPE, Integer.TYPE, ProtectionDomain.class});
//            method.setAccessible(true);
//            return (Class) method.invoke(loader, new Object[] {proxyName, b, new Integer(0), new Integer(b.length), pd});
//        } catch (ClassFormatError e) {
//            throw new IllegalArgumentException();
//        } catch (Exception e) {
//            throw Logger.getDefault().annotate(new DebugException(), e);
//        }
//    }
}
