/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.mdr.util;

import java.util.HashMap;
import java.util.HashSet;
import org.netbeans.api.mdr.events.TransactionEvent;
import org.netbeans.mdr.NBMDRepositoryImpl;
import org.netbeans.mdr.storagemodel.MdrStorage;
import org.netbeans.mdr.persistence.StorageException;

/** This class implements mutex that controls storage transactions as well as 
 * MDR events firing.
 *
 * @author  Martin Matula
 * @version 0.1
 */
public abstract class TransactionMutex {

    // storage
    private final MdrStorage storage;
    
    // events notifier
    private final EventNotifier notifier;

    // the repository (used as event source for transaction events)
    private final NBMDRepositoryImpl repository;
    
    protected TransactionMutex(Object storage, Object notifier, Object repository) {
        this.storage = (MdrStorage) storage;
        this.notifier = (EventNotifier) notifier;
        this.repository = (NBMDRepositoryImpl) repository;
    }
    
    public abstract boolean willFail();
    
    public abstract boolean pendingChanges();
 
    public abstract void enter(boolean writeAccess);
    public abstract boolean leave(boolean fail);
    
    public final void leave() {
        leave(false);
    }
    
    protected final void start() {
        TransactionEvent event = new TransactionEvent(
            repository,
            TransactionEvent.EVENT_TRANSACTION_START
        );
        notifier.REPOSITORY.firePlannedChange(storage, event);
    }
    
    protected final void end(boolean fail) {
        try {
            TransactionEvent event = new TransactionEvent(
                repository,
                TransactionEvent.EVENT_TRANSACTION_END
            );
            notifier.REPOSITORY.firePlannedChange(storage, event);
            if (fail) {
                notifier.fireCancelled();
                storage.rollback();
//                            Logger.getDefault().notify(Logger.INFORMATIONAL, new DebugException("rollback"));
            } else {
                notifier.fireChanged();
                storage.commit();
                //Logger.getDefault().log("commited");
//                            Logger.getDefault().notify(Logger.INFORMATIONAL, new DebugException("commit"));
            }
        } catch (StorageException e) {
            try { 
                Logger.getDefault().notify(Logger.INFORMATIONAL, e);
                storage.rollback();
            } catch (StorageException fatal) {
                throw new DebugException("Fatal I/O error: " + fatal.toString());
            }
        }
    }
}
