/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.mdr.test;

import java.io.*;
import java.util.*;
import java.net.*;
import java.text.SimpleDateFormat;

import junit.extensions.*;

import org.netbeans.api.mdr.*;
import org.openide.util.Lookup;
import org.netbeans.junit.*;

import org.netbeans.mdr.util.*;
import org.netbeans.mdr.NBMDRepositoryImpl;
import org.netbeans.lib.jmi.xmi.*;
import org.netbeans.lib.jmi.mapping.*;

import org.xml.sax.*;

import javax.jmi.reflect.*;
import javax.jmi.model.*;

/**
 * Test case measuring performance of various MDR operations.
 * To run the test case, run ant with the property
 * "xtest.testtype=perf". By default, test results can be found in
 * xtest's log files. In addition, when specifying the
 * property "xtest.userdata.propertyfile=perf.properties", test
 * results are recorded in our local performace tests database.
 * The performace data stored in this database can be viewed on
 * http://beetle.czech.sun.com:8080/PerformanceChartsViewer/PerformanceChartsViewer
 */
public class MDRPerformanceTests extends NbTestCase {
        
    private static final URL MODEL_XMI_URL = MDRPerformanceTests.class.getResource("data/java-model.xml");
    private static final String MOFMODEL = "MOF";
    private static final String METAMODEL = "MOF4PerformaceTest";
    private static final String MODEL = "PerformanceTest";
    private static final String PACKAGE = "Physical";
    
    private static final short MAX_INST = 100;
    private static final short MAX_ASSOC = 20;
    private static final short MAX_ALL_OF_CLASS = 10;
    private static final short MAX_QUERY_ASSOC = 20;
    private static final short MAX_INST_TO_DEL = 100;
    private static final short MAX_INST_TO_MODIFY = 10;
    private static final int MAX_ITERATIONS = 20;
    private static final int MAX_XMI_ITERATIONS = 5;
    private static final int MAX_DEL_ITERATIONS = 1;
    private static float correction = 0;

    private static MDRepository repository = null;
    private static RefPackage pkg = null;
    private static RandomDataSupport random = null;
    private static boolean enabled = false;

    private static RefObject findPackage(ModelPackage extent, String name) {
        for (Iterator it = extent.getMofPackage().refAllOfClass().iterator(); it.hasNext();) {
            ModelElement temp = (ModelElement) it.next();
            if (temp.getName().equals(name)) {
                return temp;
            }
        }
        return null;
    }        

    private void initSupport() {
        if (random == null) {
            if (pkg == null)
                fail("Extent does not exist.");
            random = new RandomDataSupport(pkg);
        }
    } 
    
    public MDRPerformanceTests(String testName) {
        super (testName);
    }
    
    public static void main (String[] args) {
        junit.textui.TestRunner.run (suite ());
    }
    
    public static NbTest suite () {
        NbTestSuite suite = new NbTestSuite ();
        suite.addTestSuite (MDRPerformanceTests.class);
        
        NbTestSetup setup = new NbTestSetup (suite) {
            public void setUp () {
            }
            public void tearDown () {
            }
        };        
        return setup;        
    }
    
    protected void setUp() {
        String workDir = System.getProperty("work.dir");        
        
        // properties will be set only if running outside of IDE
        if (System.getProperty("org.openide.version") == null) {
            System.setProperty("org.netbeans.mdr.persistence.Dir", workDir + File.separator + "test_repository_1");
            System.setProperty("org.netbeans.mdr.storagemodel.StorageFactoryClassName",
            "org.netbeans.mdr.persistence.btreeimpl.btreestorage.BtreeFactory");
            System.setProperty("org.openide.util.Lookup", "org.netbeans.mdr.test.MDRTestLookup");
            System.setProperty("org.netbeans.mdr.SaveStorageOnExit", "true");
        }
        
        if (repository == null) {
            MDRManager mdrManager = MDRManager.getDefault();
            if (mdrManager == null)
                fail("Unable to lookup MDRManager");
            repository = mdrManager.getDefaultRepository();
            if (repository == null)
                fail ("Repository not found.");
        }
        
        String timestamp = null;
        File timestampFile = new File(workDir + File.separator + "timestamp");
        if (timestampFile.exists()) {
            try {
                timestamp = new DataInputStream(new FileInputStream(timestampFile)).readUTF();    
            } catch (Exception e) {}
        }
        else {
            timestamp = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss").format(new Date());
            try {
                DataOutputStream dos = new DataOutputStream(new FileOutputStream(timestampFile));
                dos.writeUTF(timestamp);
                dos.close();
            } catch (Exception e) {}
        }

        if (!enabled && "true".equals(System.getProperty("netbeans.performance.enabled"))) {
            enabled = true;
            System.setProperty("netbeans.performance.serverPassword","");
            System.setProperty("netbeans.performance.serverUsername","perfserver");
            System.setProperty("netbeans.performance.serverURL","jdbc:mysql://beetle.czech.sun.com:3306/performance");
            System.setProperty("netbeans.performance.testSuite","MDRPerformanceTests");
            System.setProperty("netbeans.performance.totalTestCases","12");
            System.setProperty("org.openide.version",timestamp);
            System.setProperty("netbeans.build.branch","");
        }

    }

    /** Tests performance of the first MDR startup.
     *  As a result 1) a total time to boot MOF within a newly created repository
     *  is recorded together with 2) a total time to instantitate then the MOF
     *  within the repository.
     */    
    public void testMDRFirstStart () {
        long start = System.currentTimeMillis();
        repository.getExtent(MOFMODEL);
        long result = System.currentTimeMillis() - start;

        PerformanceServerConnection conn = null;
        if (enabled) {
            conn = new PerformanceServerConnection();
            conn.logTestCase("MDR - booting MOF (first start)", result);
        }
        PrintStream log = getLog();
        log.println("MDR first startup performace results:");
        log.println("------------------------------------");
        log.println("Booting MOF: " + (float)result/(float)1000 + " s");
        
        ModelPackage mmExtent = null;
        try {
            start = System.currentTimeMillis();
            mmExtent = (ModelPackage) repository.createExtent(METAMODEL);
            result = System.currentTimeMillis() - start;
        } catch (CreationFailedException e) {
            fail ("Package instantiation failed: " + METAMODEL);
        }
        
        if (enabled)
            conn.logTestCase("MDR - instantiating MOF (first start)", result);
        log.println("Creating MOF instance: " + (float)result/(float)1000 + " s");
    }
    
    /** Tests performance of the repetitional MDR startup.
     *  As a result 1) a total time to obtain a reference to MOF within an already
     *  existing repository is recorded together with 2) a total time to instantitate
     *  then the MOF within the repository.
     */    
    public void testMDRStart () {
        long start = System.currentTimeMillis();
        repository.getExtent(MOFMODEL);
        long result = System.currentTimeMillis() - start;

        PerformanceServerConnection conn = null;
        if (enabled) {
            conn = new PerformanceServerConnection();
            conn.logTestCase("MDR - booting MOF", result);
        }
        PrintStream log = getLog();
        log.println("MDR startup performace results:");
        log.println("------------------------------------");
        log.println("Booting MOF: " + (float)result/(float)1000 + " s");
        
        ModelPackage mmExtent = null;
        try {
            start = System.currentTimeMillis();
            mmExtent = (ModelPackage) repository.createExtent(METAMODEL);
            result = System.currentTimeMillis() - start;
        } catch (CreationFailedException e) {
            fail ("Package instantiation failed: " + METAMODEL);
        }
        
        if (enabled)
            conn.logTestCase("MDR - instantiating MOF", result);
        log.println("Creating MOF instance: " + (float)result/(float)1000 + " s");
        
        XMIReader reader = XMIReader.getDefault();
        if (reader == null)
            fail("Unable to lookup XMIReader");

        try {
            repository.beginTrans (true);
            reader.read (MODEL_XMI_URL.toString(), new RefPackage[] {mmExtent});
        } catch (Exception e) {
            fail (e.getMessage ());
        } finally {
            repository.endTrans ();
        }

        RefObject mmPackage = findPackage(mmExtent, PACKAGE);
        if (mmPackage == null)
            fail("Unable to find package: " + PACKAGE);
        
        try {
            pkg = repository.createExtent(MODEL, mmPackage);
        } catch (CreationFailedException e) {
            fail ("Package instantiation failed: " + MODEL);
        }
    }

    /** Tests performance of object creation.
     *  Objects are created randomly within a Java metamodel's extent.
     *  As a result 1) an average time to create an object instance is recoreded
     *  both excuding and including time necessary to start and finish an
     *  appropriate transaction. Separately, 2) a time to create the first instance
     *  of a class is recorded. This time reflects an overhead of generating handlers
     *  in MDR. 
     */    
    public void testObjectCreation () {
        initSupport();
        long[] results = new long[MAX_ITERATIONS];
        long[] extResults = new long[MAX_ITERATIONS];
        int[] nums = new int[MAX_ITERATIONS];
        System.out.println("Object creation performance test started (it takes some time to complete)...");
        for (int i = 0; i < MAX_ITERATIONS; i++) {
            System.gc();
            long seed = System.currentTimeMillis();
            random.generateObjects (seed, MAX_INST, i == 0, true);
            System.gc();
            long warmUpStart = System.currentTimeMillis();
            random.generateObjects (seed, MAX_INST, i == 0, true);
            long warmUp = System.currentTimeMillis() - warmUpStart;
            System.gc();
            long extStart = System.currentTimeMillis();
            try {
                repository.beginTrans (true);            
                long start = System.currentTimeMillis();
                random.generateObjects (seed, MAX_INST, i == 0, false);
                results[i] = System.currentTimeMillis() - start - warmUp;
            } catch (Exception e) {
                fail (e.getMessage ());
            }
            finally {
                repository.endTrans ();
            }
            extResults[i] = System.currentTimeMillis() - extStart - warmUp;
            nums[i] = random.getInstancesCounter();
        }
        System.out.println("Done.");

        int numberOfInst = 0;
        long totalTime = 0;
        long totalExtTime = 0;
        int validInst = 0;
        long validTime = 0;
        long validExtTime = 0;
        for (int i = 0; i < MAX_ITERATIONS; i++) {
            numberOfInst += nums[i];
            totalTime += results[i];
            totalExtTime += extResults[i];
            if (i >= 10) {
                validTime += results[i];
                validExtTime += extResults[i];
                validInst += nums[i];
                if (enabled && nums[i] > 0) {
                    PerformanceServerConnection conn = new PerformanceServerConnection();
                    conn.logTestCase("MDR - 100 objects creation time", results[i] * 100 / nums[i]);
                    conn.logTestCase("MDR - 100 objects creation time (including transactions)", extResults[i] * 100 / nums[i]);
                }
            }
        }
        
        PrintStream log = getLog();
        log.println("Random object creation test results:");
        log.println("------------------------------------");
        log.println("Number of created instances: " + numberOfInst);
        log.println("Total time (without transactions): " + (float)totalTime/(float)1000 + " s");
        log.println("Total time: " + (float)totalExtTime/(float)1000 + " s");
        log.println("First object creation average time (without transactions): " + (float)results[0] / (float)nums[0] + " ms");
        log.println("First object creation average time: " + (float)extResults[0] / (float)nums[0] + " ms");
        log.println("Average object creation time (without transactions): " + (correction = (float) validTime / (float) validInst) + " ms");
        log.println("Average object creation time: " + (float) validExtTime / (float) validInst + " ms");
        log.println();
        
    }
    
    /** Tests performance of association creation.
     *  Associations are created randomly within a Java metamodel's extent.
     *  As a result, an average time to create an associations instance is recoreded
     *  both excuding and including time necessary to start and finish an
     *  appropriate transaction.
     */    
    public void testAssociationCreation () {
        initSupport();
        long[] results = new long[MAX_ITERATIONS];
        long[] extResults = new long[MAX_ITERATIONS];
        int[] nums = new int[MAX_ITERATIONS];
        System.out.println("Association creation performance test started (it takes some time to complete)...");
        random.prepareAssociationEnds();
        for (int i = 0; i < MAX_ITERATIONS; i++) {
            System.gc();
            long seed = System.currentTimeMillis();
            random.generateAssociations (seed, MAX_ASSOC, i == 0, true);
            System.gc();
            long warmUpStart = System.currentTimeMillis();
            random.generateAssociations (seed, MAX_ASSOC, i == 0, true);
            long warmUp = System.currentTimeMillis() - warmUpStart;
            System.gc();
            long extStart = System.currentTimeMillis();
            try {
                repository.beginTrans (true);            
                long start = System.currentTimeMillis();
                random.generateAssociations (seed, MAX_ASSOC, i == 0, false);
                results[i] = System.currentTimeMillis() - start - warmUp - (long)(correction * random.getInstancesCounter());
            } catch (Exception e) {
                fail (e.getMessage ());
            }
            finally {
                repository.endTrans ();
            }
            extResults[i] = System.currentTimeMillis() - extStart - warmUp - (long)(correction * random.getInstancesCounter());
            nums[i] = random.getLinksCounter();
        }
        System.out.println("Done.");
        
        int numberOfInst = 0;
        long totalTime = 0;
        long totalExtTime = 0;
        int validInst = 0;
        long validTime = 0;
        long validExtTime = 0;
        for (int i = 0; i < MAX_ITERATIONS; i++) {
            numberOfInst += nums[i];
            totalTime += results[i];
            totalExtTime += extResults[i];
            if (i >= 10) {
                validTime += results[i];
                validExtTime += extResults[i];
                validInst += nums[i];
                if (enabled && nums[i] > 0) {
                    PerformanceServerConnection conn = new PerformanceServerConnection();
                    conn.logTestCase("MDR - 100 associations creation time", results[i] * 100 / nums[i]);
                    conn.logTestCase("MDR - 100 associations creation time (including transactions)", extResults[i] * 100 / nums[i]);
                }
            }
        }
        
        PrintStream log = getLog();
        log.println("Random association creation test results:");
        log.println("------------------------------------");
        log.println("Number of created associations: " + numberOfInst);
        log.println("Total time (without transactions): " + (float)totalTime/(float)1000 + " s");
        log.println("Total time: " + (float)totalExtTime/(float)1000 + " s");
        log.println("Average association creation time (without transactions): " + (float)validTime / (float)validInst + " ms");
        log.println("Average association creation time: " + (float)validExtTime / (float)validInst + " ms");
        log.println();
    }
    
    /** Tests performance of the RefClass.refAllOfClass() operation.
     *  All classes of the Java metamodel are queried.
     *  As a result, an average time to obtain the requested collection of instances
     *  of the class is recorded.
     */    
    public void testAllOfClass() {
        initSupport();
        int numberOfClasses = random.getNumberOfClasses();
        long[] results = new long[numberOfClasses];
        System.out.println("refAllOfClass performance test started...");
        for (int i = 0; i < numberOfClasses; i++) {
            System.gc();
            long seed = System.currentTimeMillis();
            random.findAllOfClass(i);
            results[i] = System.currentTimeMillis() - seed;
        }
        System.out.println("Done.");

        long totalTime = 0;
        
        for (int i = 0; i < numberOfClasses; i++) {
            totalTime += results[i];
            if (enabled && results[i] > 0) {
                PerformanceServerConnection conn = new PerformanceServerConnection();
                conn.logTestCase("MDR - refAllOfClass invocations time", results[i]);
            }
        }

        PrintStream log = getLog();
        log.println("Random refAllOfClass test results:");
        log.println("------------------------------------");
        log.println("Number of queried classes: " + numberOfClasses);
        log.println("Total time: " + (float)totalTime/(float)1000 + " s");
        log.println("Average time: " + (float)totalTime / (float)numberOfClasses + " ms");
        log.println();
    }
    
    /** Tests performance of iterators iterating through the RefClass.refAllOfClass() operation's
     *  results.
     *  As a result, an average time to iterate through a collection of instances
     *  of the class is recorded.
     */    
    public void testIteratingAllOfClass () {
        initSupport();
        int numberOfClasses = random.getNumberOfClasses();
        long[] results = new long[numberOfClasses];
        int[] nums = new int[numberOfClasses];
        System.out.println("Iterating through refAllOfClass started...");
        for (int i = 0; i < numberOfClasses; i++) {
            System.gc();
            long seed = System.currentTimeMillis();
            nums[i] = random.iterateAllOfClass(i);
            results[i] = System.currentTimeMillis() - seed;
        }
        System.out.println("Done.");
        
        int numberOfInst = 0;
        long totalTime = 0;
        for (int i = 0; i < numberOfClasses; i++) {
            numberOfInst += nums[i];
            totalTime += results[i];
            if (enabled && nums[i] > 0) {
                PerformanceServerConnection conn = new PerformanceServerConnection();
                conn.logTestCase("MDR - Iterating through 100 objects returned by refAllOfClass", results[i] * 100 / nums[i]);
            }
        }
        
        PrintStream log = getLog();
        log.println("Iterating through refAllOfClass results:");
        log.println("------------------------------------");
        log.println("Number of iterated classes: " + numberOfClasses);
        log.println("Number of iterated instances: " + numberOfInst);
        log.println("Total time: " + (float)totalTime/(float)1000 + " s");
        log.println("Average number of instances per class: " + numberOfInst/numberOfClasses);
        log.println("Average time per iterated class: " + (float)totalTime/(float)numberOfClasses + " ms");
        log.println("Average time per iterated instance: " + (float)totalTime/(float)numberOfInst + " ms");
        log.println();
    }
    
    /** Tests performance of the RefAssociation.refQuery(...) operation.
     *  Randomly queries associations previously created within a Java metamodel's extent.
     *  As a result, an average time to query a randomly selected association
     *  both excluding and including time necessary to start and finish an appropriate
     *  transaction is recorded.
     */    
    public void testQueryAssociations() {
        initSupport();
        long[] results = new long[MAX_ITERATIONS];
        int[] nums = new int[MAX_ITERATIONS];
        System.out.println("Random query associations test started...");
        for (int i = 0; i < MAX_ITERATIONS; i++) {
            System.gc();
            long seed = System.currentTimeMillis();
            nums[i] = random.queryAssociations(seed, MAX_QUERY_ASSOC);
            results[i] = System.currentTimeMillis() - seed;
        }
        System.out.println("Done.");

        int others = random.getOthersCounter();
        int numberOfInst = 0;
        long totalTime = 0;
        for (int i = 0; i < MAX_ITERATIONS; i++) {
            numberOfInst += nums[i];
            totalTime += results[i];
            if (enabled && nums[i] > 0) {
                PerformanceServerConnection conn = new PerformanceServerConnection();
                conn.logTestCase("MDR - query 100 associations time", results[i] * 100 / nums[i]);
            }
        }
        
        PrintStream log = getLog();
        log.println("Random query associations test results:");
        log.println("------------------------------------");
        log.println("Number of queried association ends: " + numberOfInst);
        log.println("Number of traversed other ends: " + others);
        log.println("Total time: " + (float)totalTime/(float)1000 + " s");
        log.println("Average time per queried association end: " + (float)totalTime/(float)numberOfInst + " ms");
        log.println("Average time per other end traversed: " + (float)totalTime/(float)others + " ms");
        log.println();
    }
    
    /** Tests performance of object modifications.
     *  Randomly modifies objects previously created within a Java metamodel's extent.
     *  As a result, an average time to modify a randomly selected attribute of a 
     *  randomly selected object both excluding and including time necessary to
     *  start and finish an appropriate transaction is recorded.
     */    
    public void testObjectModification() {
        initSupport();
        long[] results = new long[MAX_ITERATIONS];
        long[] extResults = new long[MAX_ITERATIONS];
        int[] nums = new int[MAX_ITERATIONS];
        System.out.println("Object modification performance test started (it takes some time to complete)...");
        random.modifyObjects(0, MAX_INST_TO_MODIFY, true, true);
        for (int i = 0; i < MAX_ITERATIONS; i++) {
            System.gc();
            long seed = System.currentTimeMillis();
            random.modifyObjects(seed, MAX_INST_TO_MODIFY, false, true);
            long warmUp = System.currentTimeMillis() - seed;
            System.gc();
            long extStart = System.currentTimeMillis();
            try {
                repository.beginTrans (true);            
                long start = System.currentTimeMillis();
                random.modifyObjects(seed, MAX_INST_TO_MODIFY, false, false);
                results[i] = System.currentTimeMillis() - start - warmUp;
            } catch (Exception e) {
                fail (e.getMessage ());
            }
            finally {
                repository.endTrans ();
            }
            extResults[i] = System.currentTimeMillis() - extStart - warmUp;
            nums[i] = random.getInstancesCounter();
            if (results[i] < 0) {
                results[i] = 0;
                extResults[i] = 0;
                nums[i] = 0;
            }
        }
        System.out.println("Done.");

        int numberOfInst = 0;
        long totalTime = 0;
        long totalExtTime = 0;
        for (int i = 0; i < MAX_DEL_ITERATIONS; i++) {
            numberOfInst += nums[i];
            totalTime += results[i];
            totalExtTime += extResults[i];
            if (enabled && nums[i] > 0) {
                PerformanceServerConnection conn = new PerformanceServerConnection();
                conn.logTestCase("MDR - 100 object modifications time", results[i] * 100 / nums[i]);
                conn.logTestCase("MDR - 100 object modifications time (including transactions)", extResults[i] * 100 / nums[i]);
            }
        }

        PrintStream log = getLog();
        log.println("Random object modification test results:");
        log.println("------------------------------------");
        log.println("Number of modified instances: " + numberOfInst);
        log.println("Total time (without transactions): " + (float)totalTime/(float)1000 + " s");
        log.println("Total time: " + (float)totalExtTime/(float)1000 + " s");
        log.println("Average object modification time (without transactions): " + (float)totalTime/(float)numberOfInst + " ms");
        log.println("Average object modification time: " + (float)totalExtTime/(float)numberOfInst + " ms");
        log.println();
    }
    
    /** Tests performance of object modifications.
     *  Randomly modifies objects previously created within a Java metamodel's extent.
     *  As a result, an average time to modify a randomly selected attribute of a 
     *  randomly selected object within an autocommited transaction is recorded.
     */    
    public void testAutocommitedModifications() {
        initSupport();
        long[] results = new long[MAX_ITERATIONS];
        long[] extResults = new long[MAX_ITERATIONS];
        int[] nums = new int[MAX_ITERATIONS];
        System.out.println("Autocommited object modification performance test started (it takes some time to complete)...");
        random.modifyObjects(0, MAX_INST_TO_MODIFY, true, true);
        for (int i = 0; i < MAX_ITERATIONS; i++) {
            System.gc();
            long seed = System.currentTimeMillis();
            random.modifyObjects(seed, MAX_INST_TO_MODIFY, false, true);
            long warmUp = System.currentTimeMillis() - seed;
            System.gc();
            long start = System.currentTimeMillis();
            random.modifyObjects(seed, MAX_INST_TO_MODIFY, false, false);
            results[i] = System.currentTimeMillis() - start - warmUp;
            nums[i] = random.getInstancesCounter();
        }
        System.out.println("Done.");

        int numberOfInst = 0;
        long totalTime = 0;
        for (int i = 0; i < MAX_DEL_ITERATIONS; i++) {
            numberOfInst += nums[i];
            totalTime += results[i];
            if (enabled && nums[i] > 0) {
                PerformanceServerConnection conn = new PerformanceServerConnection();
                conn.logTestCase("MDR - 100 autocommited object modifications time", results[i] * 100 / nums[i]);
            }
        }

        PrintStream log = getLog();
        log.println("Random autocommited object modification test results:");
        log.println("------------------------------------");
        log.println("Number of modified instances: " + numberOfInst);
        log.println("Total time: " + (float)totalTime/(float)1000 + " s");
        log.println("Average object modification time: " + (float)totalTime/(float)numberOfInst + " ms");
        log.println();
    }
    
    /** Tests performance of object deletion.
     *  Ramdomly deletes objects previously created within a Java metamodel's extent.
     *  As a result, an average time to delete an object instance both excluding and
     *  including time necessary to start and finish an appropriate transaction
     *  is recorded.
     */    
    public void testObjectDeletion() {
        initSupport();
        long[] results = new long[MAX_DEL_ITERATIONS];
        long[] extResults = new long[MAX_DEL_ITERATIONS];
        int[] nums = new int[MAX_DEL_ITERATIONS];
        System.out.println("Object deletion performance test started (it takes some time to complete)...");
        for (int i = 0; i < MAX_DEL_ITERATIONS; i++) {
            System.gc();
            long seed = System.currentTimeMillis();
            nums[i] = random.prepareObjectsToDelete(seed, MAX_INST_TO_DEL);
            System.gc();
            long extStart = System.currentTimeMillis();
            try {
                repository.beginTrans (true);            
                long start = System.currentTimeMillis();
                random.deleteObjects ();
                results[i] = System.currentTimeMillis() - start;
            } catch (Exception e) {
                fail (e.getMessage ());
            }
            finally {
                repository.endTrans ();
            }
            extResults[i] = System.currentTimeMillis() - extStart;
        }
        System.out.println("Done.");

        int numberOfInst = 0;
        long totalTime = 0;
        long totalExtTime = 0;
        for (int i = 0; i < MAX_DEL_ITERATIONS; i++) {
            numberOfInst += nums[i];
            totalTime += results[i];
            totalExtTime += extResults[i];
            if (enabled && nums[i] > 0) {
                PerformanceServerConnection conn = new PerformanceServerConnection();
                conn.logTestCase("MDR - 100 objects deletion time", results[i] * 100 / nums[i]);
                conn.logTestCase("MDR - 100 objects deletion time (including transactions)", extResults[i] * 100 / nums[i]);
            }
        }

        PrintStream log = getLog();
        log.println("Random object deletion test results:");
        log.println("------------------------------------");
        log.println("Number of deleted instances: " + numberOfInst);
        log.println("Total time (without transactions): " + (float)totalTime/(float)1000 + " s");
        log.println("Total time: " + (float)totalExtTime/(float)1000 + " s");
        log.println("Average object deletion time (without transactions): " + (float)totalTime/(float)numberOfInst + " ms");
        log.println("Average object deletion time: " + (float)totalExtTime/(float)numberOfInst + " ms");
        log.println();
    }
            
    public void testXMIExportImport () {
        XMIReader reader = XMIReader.getDefault();
        if (reader == null)
            fail("Unable to lookup XMIReader");
        XMIWriter writer = XMIWriter.getDefault();
        if (writer == null)
            fail("Unable to lookup XMIWriter");
        ModelPackage mmExtent = (ModelPackage) repository.getExtent(METAMODEL);
        RefObject mmPackage = findPackage(mmExtent, PACKAGE);
        if (mmPackage == null)
            fail("Unable to find package: " + PACKAGE);
        
        RefPackage javaPackage = null;
        long time = 0, extTime = 0;
        long[] writeResults = new long[MAX_XMI_ITERATIONS];
        long[] readResults = new long[MAX_XMI_ITERATIONS];
        long[] extReadResults = new long[MAX_XMI_ITERATIONS];
        int[] linksCounts = new int[MAX_XMI_ITERATIONS];
        int[] objectsCounts = new int[MAX_XMI_ITERATIONS];
        
        File file = new File ("temp.xml");
        String uri = null;
        try {
            uri = file.toURL ().toExternalForm ();
        } catch (MalformedURLException e) {
            fail (e.getMessage ());
        }
        
        System.out.println("XMI export/import performance test started (it takes some time to complete)...");        
        for (int i = 0; i < MAX_XMI_ITERATIONS; i++) {
            try {
                javaPackage = repository.createExtent(MODEL, mmPackage);
            } catch (CreationFailedException e) {
                fail ("Package instantiation failed: " + MODEL);
            }
            
            long seed = System.currentTimeMillis();
            RandomDataSupport support = new RandomDataSupport (javaPackage);
            support.generateObjects (seed, MAX_INST, false, false);
            objectsCounts[i] = support.getInstancesCounter();
            support.generateAssociations (seed, MAX_INST, false, false);            
            linksCounts[i] = support.getLinksCounter();
            objectsCounts[i] += support.getInstancesCounter();
                        
            OutputStream os = null;
            try {
                os = new FileOutputStream (file);
            } catch (IOException e) {
                fail (e.getMessage ());
            }
            System.gc();
            
            try {
                time = System.currentTimeMillis();            
                writer.write (os, javaPackage);
                os.flush ();
                os.close ();
            } catch (Exception e) {
                fail (e.getMessage ());
            }
            writeResults[i] = System.currentTimeMillis() - time;
            
            javaPackage.refDelete ();
            try {
                javaPackage = repository.createExtent(MODEL, mmPackage);
            } catch (CreationFailedException e) {
                fail ("Package instantiation failed: " + MODEL);
            }            
            RefPackage [] extents = new RefPackage [] {javaPackage};
            System.gc();
            
            extTime = System.currentTimeMillis();
            try {                           
                repository.beginTrans (true);
                time = System.currentTimeMillis();
                reader.read (uri, extents);
                readResults[i] = System.currentTimeMillis() - time;
            } catch (Exception e) {
                fail (e.getMessage ());
            } finally {
                repository.endTrans ();
            }
            extReadResults[i] = System.currentTimeMillis() - extTime;
            javaPackage.refDelete ();
            
        } // for
        System.out.println("Done.");
        
        int numberOfInst = 0, numberOfLinks = 0;
        long totalWriteTime = 0, totalReadTime = 0, totalExtReadTime = 0;
        for (int i = 0; i < MAX_XMI_ITERATIONS; i++) {
            numberOfInst += objectsCounts[i];
            numberOfLinks += linksCounts[i];
            totalWriteTime += writeResults[i];
            totalReadTime += readResults[i];
            totalExtReadTime += extReadResults[i];
            if (enabled) {
                PerformanceServerConnection conn = new PerformanceServerConnection();
                conn.logTestCase("MDR - 100 objects import time", readResults[i] * 100 / objectsCounts[i]);
                conn.logTestCase("MDR - 100 objects import time (including transactions)", extReadResults[i] * 100 / objectsCounts[i]);
                conn.logTestCase("MDR - 100 objects export time", writeResults[i] * 100 / objectsCounts[i]);
            }
        }

        PrintStream log = getLog();
        log.println("Random data export/import test results:");
        log.println("---------------------------------------");
        log.println("Number of exported/imported instances: " + numberOfInst);
        log.println("Number of exported/imported links: " + numberOfLinks);
        log.println("Total import time (without transactions): " + (float)totalReadTime/(float)1000 + " s");
        log.println("Total import time: " + (float)totalExtReadTime/(float)1000 + " s");
        log.println("Total export time: " + (float)totalWriteTime/(float)1000 + " s");
        log.println("Average import time (without transactions) per instance: " + (float)totalReadTime/(float)numberOfInst + " ms");
        log.println("Average import time per instance: " + (float)totalExtReadTime/(float)numberOfInst + " ms");
        log.println("Average export time per instance: " + (float)totalWriteTime/(float)numberOfInst + " ms");        
        log.println();        
    }

}


