package org.netbeans.mdr.test;

import java.net.InetAddress;
/*
 * PerformanceServerConnection.java
 *
 *  Created on June 27, 2001, 5:38 PM
 */

import java.util.Properties;
import java.util.Enumeration;
import java.net.MalformedURLException;
import java.sql.*;
import java.util.ResourceBundle;

/** class maintaining connection to Performance Server Database
 *
 * @author <a href="mailto:adam.sotona@sun.com">Adam Sotona</a>
 * @version 1.0
 */
public class PerformanceServerConnection {

    static {
        try {
            Class.forName("org.gjt.mm.mysql.Driver");
        } catch (ClassNotFoundException ex) {
            ex.printStackTrace();
        }
    }

    private static PerformanceServerConnection connection;

    private Connection conn=null;
    private String URL;
    private String username;
    private String password;
    private String IDEbranch;
    private String buildNumber;
    private String testSuite;
    private int totalTestCases;
    private int executionId;


    /** performs lookup for property inside given properties and then inside System properties
     * @param p connection properties
     * @param key property key
     * @throws Exception throws Exception when key is missing
     * @return property value
     */    
    private static String readProperty (final Properties p, final String key) throws Exception {
        String value=null;
        if (p!=null) value=p.getProperty(key,System.getProperty(key));
        else value=System.getProperty(key);
        if (value==null) throw new Exception("PerformanceServerConnection is missing option : "+key);
        return value;
    }

    /** creates new connection with all informations stored in System properties
     */    
    public PerformanceServerConnection() {
        this(System.getProperties());
    }

    /** creates new connection with all informations stored in given properties properties merged with System properties
     * <br>
     * needed properties are:<br><pre>
     *       netbeans.performance.serverURL - URL of Performance Server Database
     *       netbeans.performance.serverUsername - user name
     *       netbeans.performance.serverPassword - pasword
     *       netbeans.build.branch - additional information about tested IDE
     *       org.openide.version - build number of tested IDE (set by default)
     *       netbeans.performance.testSuite - testsuite name
     *       netbeans.performance.totalTestCases - number of reported testcases for verification of complete testsuite
     *       os.name, os.version and os.arch - current OS informations (set by default)
     *       java.vendor, java.version - current Java informations (set by default)
     * @param p connection properties
     */    
    public PerformanceServerConnection(final Properties p) {
        try {
            executionId=0;
            URL=readProperty(p,"netbeans.performance.serverURL");
            username=readProperty(p,"netbeans.performance.serverUsername");
            password=readProperty(p,"netbeans.performance.serverPassword");
            IDEbranch=readProperty(p,"netbeans.build.branch");
            buildNumber=readProperty(p,"org.openide.version");
            testSuite=readProperty(p,"netbeans.performance.testSuite");
            totalTestCases=Integer.parseInt(readProperty(p,"netbeans.performance.totalTestCases"));
            String systemInfo=readProperty(p,"os.name")+" "+readProperty(p,"os.version")+" "+readProperty(p,"os.arch");
            String javaInfo=readProperty(p,"java.vendor")+" "+readProperty(p,"java.version");
            InetAddress local=InetAddress.getLocalHost();
            String computer=local.getHostName()+" "+local.getHostAddress();
            System.out.println(testSuite+", "+systemInfo+", "+javaInfo+", "+IDEbranch+", "+buildNumber+", "+String.valueOf(totalTestCases));
            executionId=getExecutionId(testSuite, computer, systemInfo, javaInfo, IDEbranch, buildNumber, totalTestCases);
        } catch (Exception ex) {
            ex.printStackTrace();
            executionId=-1;
        }
    }
    
    /** compares set of connection properties foe equality
     * @param p connection properties to comapre
     * @return boolean result
     */    
    private boolean equalOptions(final Properties p) {
        try {
            return
                URL.equals(readProperty(p,"netbeans.performance.serverURL"))&&
                username.equals(readProperty(p,"netbeans.performance.serverUsername"))&&
                password.equals(readProperty(p,"netbeans.performance.serverPassword"))&&
                IDEbranch.equals(readProperty(p,"netbeans.build.branch"))&&
                buildNumber.equals(readProperty(p,"org.openide.version"))&&
                testSuite.equals(readProperty(p,"netbeans.performance.testSuite"))&&
                (totalTestCases==Integer.parseInt(readProperty(p,"netbeans.performance.totalTestCases")));
        } catch (Exception ex) {
            ex.printStackTrace();
            return false;
        }
    }

    /** returns default connection holded or created according to System properties
     * @return PerformanceServerConnection
     */    
    public static PerformanceServerConnection getConnection() {
        return getConnection(System.getProperties());
    }
    
    /** returns default connection holded or created according to given and System properties
     * @param p connection properties
     * @return PerformanceServerConnection
     */    
    public static PerformanceServerConnection getConnection(final Properties p) {
        if (connection==null) {
            connection=new PerformanceServerConnection(p);
        } else {
            if (!connection.equalOptions(p)) {
                connection=new PerformanceServerConnection(p);
            }
        }
        return connection;
    }

    /** returns default connection holded or created according to properties stored in given file and System properties
     * @param clazz class neneded to determine propertiesFile location (lookup is performed by class loader)
     * @param propertiesFile connection properties file name
     * @return PerformanceServerConnection
     */    
    public static PerformanceServerConnection getConnection(final Class clazz, final String propertiesFile) {
        Properties props=new Properties();
        try {
            props.load(clazz.getClassLoader().getResourceAsStream(propertiesFile));
        } catch (Exception ex) {
            ex.printStackTrace();
        } 
        return getConnection(props);
    }

    
    /** sets default PerformanceServerConnection
     * @param newConnection new default PerformanceServerConnection
     */    
    public static void setConnection(PerformanceServerConnection newConnection) {
        connection=newConnection;
    }
    
    /** logs one testcase to performance server
     * @param testCase test case name
     * @param time time in ms
     */    
    public void logTestCase(final String testCase, final long time) {
        if (executionId>0) {
            try {
                logTestCase(executionId, testCase, time);
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        }
        System.out.println(testCase+": "+String.valueOf(time)+" ms.");
    }            

    /** logs several testcase stored in properties to performance server
     * @param testCases properties with testcases
     */    
    public void logTestCases(final Properties testCases) {
        if (executionId>0) {
            Enumeration keys=testCases.propertyNames();
            String key;
            while (keys.hasMoreElements()) {
                try {
                    key=(String)keys.nextElement();
                    logTestCase(executionId, key, Integer.parseInt(testCases.getProperty(key)));
                } catch (Exception ex) {
                    ex.printStackTrace();
                }
            }
        }
    } 
    
    private int getExecutionId(final String testSuite, final String computer, final String systemInfo, final String javaInfo, final String IDEbranch, final String buildNumber, final int totalTestCases) throws SQLException {
        try {
            if (conn==null)
                conn=DriverManager.getConnection(URL, username, password);
            PreparedStatement stat=conn.prepareStatement("insert into executions (branch,build,computer,java,rundate,system,testsuite,totalkeys) values (?,?,?,?,?,?,?,?)");
            stat.setString(1,IDEbranch);
            stat.setString(2,buildNumber);
            stat.setString(3,computer);
            stat.setString(4,javaInfo);
            stat.setTimestamp(5,new Timestamp((new java.util.Date()).getTime()));
            stat.setString(6,systemInfo);
            stat.setString(7,testSuite);
            stat.setInt(8,totalTestCases);
            stat.executeUpdate();
            SQLWarning w;
            while ((w=stat.getWarnings())!=null) {
                System.err.println("["+testSuite+" "+systemInfo+" "+javaInfo+" "+IDEbranch+" "+buildNumber+" "+totalTestCases+"]: "+w.getMessage());
            }
            ResultSet res=stat.executeQuery("select last_insert_id()");
            while ((w=res.getWarnings())!=null) {
                System.err.println("["+testSuite+" "+systemInfo+" "+javaInfo+" "+IDEbranch+" "+buildNumber+" "+totalTestCases+"]: "+w.getMessage());
            }
            res.first();
            //System.out.println(res.getInt(1)+" "+testSuite+" "+systemInfo+" "+javaInfo+" "+IDEbranch+" "+buildNumber+" "+totalTestCases);
            return res.getInt(1);
        } catch (SQLException ex) {
            System.err.println("["+testSuite+" "+systemInfo+" "+javaInfo+" "+IDEbranch+" "+buildNumber+" "+totalTestCases+"]: "+ex.getMessage());
            ex.printStackTrace();
            throw ex;
        }
    }
    
    private void logTestCase(final int executionId, final String testCase, final long time) throws SQLException {
        try {
            if (conn==null)
                conn=DriverManager.getConnection(URL, username, password);
            PreparedStatement stat=conn.prepareStatement("insert into data (id,testcase,time) values (?,?,?)");
            stat.setInt(1,executionId);
            stat.setString(2,testCase);
            stat.setLong(3,time);
            stat.executeUpdate();
            SQLWarning w;
            while ((w=stat.getWarnings())!=null) {
                System.err.println("["+executionId+" "+testCase+" "+time+"]: "+w.getMessage());
            }
            stat=conn.prepareStatement("update executions set finishedkeys=finishedkeys+1 where id=?");
            stat.setInt(1,executionId);
            stat.executeUpdate();
            while ((w=stat.getWarnings())!=null) {
                System.err.println("["+executionId+" "+testCase+" "+time+"]: "+w.getMessage());
            }        
        } catch (SQLException ex) {
            System.err.println("["+executionId+" "+testCase+" "+time+"]: "+ex.getMessage());
            ex.printStackTrace();
            throw ex;
        }
    }
    
    
    /** example of class usage
     * @param args command line arguments
     */    
    public static void main(String args[]) {
        System.setProperty("netbeans.performance.serverPassword","");
        System.setProperty("netbeans.performance.serverUsername","perfserver");
        System.setProperty("netbeans.performance.serverURL","jdbc:mysql://beetle.czech.sun.com:3306/performance");
        System.setProperty("netbeans.performance.testSuite","test");
        System.setProperty("netbeans.build.branch","test");
        System.setProperty("netbeans.performance.totalTestCases","1");

        getConnection().logTestCase("test value",100);
    }
}
