/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

/*
 * 
 */
package org.openide.explorer.view;

import java.awt.AWTException;
import java.awt.BorderLayout;
import java.awt.Robot;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.beans.PropertyVetoException;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;

import javax.swing.JList;
import javax.swing.SwingUtilities;

import junit.textui.TestRunner;

import org.netbeans.junit.NbTestCase;
import org.netbeans.junit.NbTestSuite;

import org.openide.explorer.ExplorerPanel;
import org.openide.nodes.AbstractNode;
import org.openide.nodes.Children;
import org.openide.nodes.Node;
import org.openide.nodes.Children.Array;

/**
 * Tests for class ContextTreeViewTest
 */
public class ContextTreeViewModelTest extends NbTestCase {
    
    private static final int NO_OF_NODES = 3;

    
    public ContextTreeViewModelTest(String name) {
        super(name);
    }

    protected boolean runInEQ () {
        return true;
    }
    
    
    public void testCheckThatTheModelFiresChangesAboutOnlyVisibleNodes () throws Throwable {
        final AbstractNode root = new AbstractNode (new Children.Array ());
        root.setName ("test root");
        
        root.getChildren ().add (new Node[] {
            createNode ("kuk", true),
            createNode ("huk", true),
        });
        
        
        NodeTreeModel m = new ContextTreeView.NodeContextModel ();
        m.setNode (root);
        javax.swing.tree.TreeNode visual = (javax.swing.tree.TreeNode)m.getRoot ();
        waitEQ ();
        
        
        assertEquals ("Leaf nodes are not counted", 0, m.getChildCount (visual));
        
        Listener listener = new Listener ();
        m.addTreeModelListener (listener);
        
        Node n = createNode ("blik", true);
        
        root.getChildren ().add (new Node[] { n });
        assertEquals ("Leaf nodes are not counted even when added", 0, m.getChildCount (visual));
        assertEquals ("Really added", n.getParentNode (), root);
        listener.assertEvents ("No events", 0);
        
        root.getChildren ().remove (new Node[] { n });
        listener.assertEvents ("Still no events", 0);
        assertNull ("Removed", n.getParentNode ());
        
        Node nonLeaf = createNode ("nonleaf", false);
        root.getChildren ().add (new Node[] { nonLeaf });
        assertEquals ("One child is there", 1, m.getChildCount (visual));
        listener.assertEvents ("This node is visible there", 1);
        listener.assertIndexes ("Added at position zero", new int[] { 0 });
        assertEquals ("Really added", nonLeaf.getParentNode (), root);
        
        root.getChildren ().remove (new Node[] { nonLeaf });
        assertEquals ("One child is away", 0, m.getChildCount (visual));
        assertNull ("Removed", nonLeaf.getParentNode ());
        listener.assertEvents ("And it has been removed", 1);
        listener.assertIndexes ("Removed from position zero", new int[] { 0 });
        
    }
    
    public void testABitMoreComplexAddAndRemoveEventCheck () throws Throwable {
        final AbstractNode root = new AbstractNode (new Children.Array ());
        root.setName ("test root");
        
        root.getChildren ().add (new Node[] {
            createNode ("kuk", false),
            createNode ("huk", false),
        });
        
        
        NodeTreeModel m = new ContextTreeView.NodeContextModel ();
        m.setNode (root);
        javax.swing.tree.TreeNode visual = (javax.swing.tree.TreeNode)m.getRoot ();
        waitEQ ();
        
        
        assertEquals ("Initial size is two", 2, m.getChildCount (visual));
        
        Listener listener = new Listener ();
        m.addTreeModelListener (listener);

        Node[] arr = {
            createNode ("add1", false), createNode ("add2", false)
        };
        
        root.getChildren ().add (arr);
        listener.assertEvents ("One addition", 1);
        listener.assertIndexes ("after the two first", new int[] { 2, 3 });
        
        root.getChildren ().remove (arr);
        listener.assertEvents ("One removal", 1);
        listener.assertIndexes ("from end", new int[] { 2, 3 });
        
    }
    
    public void testRemoveInMiddle () throws Throwable {
        final AbstractNode root = new AbstractNode (new Children.Array ());
        root.setName ("test root");
        root.getChildren ().add (new Node[] { createNode ("Ahoj", false) });
        
        Node[] first = {
            createNode ("kuk", false),
            createNode ("huk", false),
        };
        
        root.getChildren ().add (first);
        
        
        NodeTreeModel m = new ContextTreeView.NodeContextModel ();
        m.setNode (root);
        javax.swing.tree.TreeNode visual = (javax.swing.tree.TreeNode)m.getRoot ();
        waitEQ ();
        
        
        assertEquals ("Initial size is two", 3, m.getChildCount (visual));
        
        Listener listener = new Listener ();
        m.addTreeModelListener (listener);

        Node[] arr = {
            createNode ("add1", false), createNode ("add2", false)
        };
        
        root.getChildren ().add (arr);
        listener.assertEvents ("One addition", 1);
        listener.assertIndexes ("after the three first", new int[] { 3, 4 });
        
        root.getChildren ().remove (first);
        listener.assertEvents ("One removal", 1);
        listener.assertIndexes ("from end", new int[] { 1, 2 });
        
    }

    private static Node createNode (String name, boolean leaf) {
        AbstractNode n = new AbstractNode (leaf ? Children.LEAF : new Children.Array ());
        n.setName (name);
        return n;
    }
    
    private void waitEQ () throws Throwable {
        /*
        try {
            javax.swing.SwingUtilities.invokeAndWait (new Runnable () { public void run () { } } );
        } catch (java.lang.reflect.InvocationTargetException ex) {
            throw ex.getTargetException ();
        }
         */
    }

    private static class Panel extends javax.swing.JPanel 
    implements org.openide.explorer.ExplorerManager.Provider {
        private org.openide.explorer.ExplorerManager em = new org.openide.explorer.ExplorerManager ();

        public org.openide.explorer.ExplorerManager getExplorerManager() {
            return em;
        }
    }
    
    private class Listener implements javax.swing.event.TreeModelListener {
        private int cnt;
        private int[] indexes;
        
        public void assertEvents (String msg, int cnt) throws Throwable {
            waitEQ ();
            assertEquals (msg, cnt, this.cnt);
            this.cnt = 0;
        }
        public void assertIndexes (String msg, int[] arr) throws Throwable {
            waitEQ ();
            assertNotNull (msg + " there has to be some", indexes);
            boolean bad = false;
            if (arr.length != indexes.length) {
                bad = true;
            } else {
                for (int i = 0; i < arr.length; i++) {
                    if (arr[i] != indexes[i]) {
                        bad = true;
                    }
                }
            }
            if (bad) {
                fail (msg + " expected: " + toStr (arr) + " was: " + toStr (indexes));
            }
            
            this.indexes = null;
        }
        
        private String toStr (int[] arr) {
            StringBuffer sb = new StringBuffer ();
            String sep = "[";
            for (int i = 0; i < arr.length; i++) {
                sb.append (sep);
                sb.append (arr[i]);
                sep = ", ";
            }
            sb.append (']');
            return sb.toString ();
        }

        public void treeNodesChanged (javax.swing.event.TreeModelEvent treeModelEvent) {
            cnt++;
        }

        public void treeNodesInserted (javax.swing.event.TreeModelEvent treeModelEvent) {
            cnt++;
            indexes = treeModelEvent.getChildIndices ();
        }

        public void treeNodesRemoved (javax.swing.event.TreeModelEvent treeModelEvent) {
            cnt++;
            indexes = treeModelEvent.getChildIndices ();
        }

        public void treeStructureChanged (javax.swing.event.TreeModelEvent treeModelEvent) {
            cnt++;
        }
    }
}
