/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Leon Chiver. All Rights Reserved.
 */
package org.netbeans.modules.refactoring.experimental.plugins;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;
import javax.jmi.reflect.RefObject;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.refactoring.api.AbstractRefactoring;
import org.netbeans.modules.refactoring.api.Problem;
import org.netbeans.modules.refactoring.experimental.CleanUpRefactoring;
import org.netbeans.modules.refactoring.plugins.JavaRefactoringPlugin;
import org.netbeans.modules.refactoring.spi.RefactoringElementsBag;
import org.netbeans.modules.refactoring.spi.SimpleRefactoringElementImpl;
import org.openide.filesystems.FileObject;
import org.openide.text.PositionBounds;
import org.openide.util.NbBundle;

/**
 * @author leon
 */
public class CleanUpRefactoringPlugin extends JavaRefactoringPlugin {
    
    private CleanUpRefactoring refactoring;

    public CleanUpRefactoringPlugin(CleanUpRefactoring refactoring) {
        this.refactoring = refactoring;
    }
    
    public Problem preCheck() {
        return null;
    }
    
    public Problem checkParameters() {
        return null;
    }
    
    public Problem fastCheckParameters() {
        if (refactoring.getResources().size() == 0) {
            return new Problem(true, 
                    NbBundle.getMessage(CleanUpRefactoringPlugin.class, "ERR_NoFiles")); // NOI18N
        }
        if (!isRemoveUnusedCode() && !refactoring.isRemoveUnusedImports()) {
            return new Problem(true, 
                    NbBundle.getMessage(CleanUpRefactoringPlugin.class, "ERR_NoCleanUpActions")); // NOI18N
        }
        if (refactoring.getResources().size() > 1) {
            return new Problem(false, 
                    NbBundle.getMessage(CleanUpRefactoringPlugin.class, "WARN_SingleFileRefactoring")); // NOI18N
        } else {
            return new Problem(false, 
                    NbBundle.getMessage(CleanUpRefactoringPlugin.class, "WARN_ReviewChanges")); // NOI18N
        }
    }
    
    public Problem prepare(RefactoringElementsBag refactoringElements) {
        List/*<Resource>*/ resources = refactoring.getResources();
        boolean comment = refactoring.isCommentInsteadOfRemoving();
        boolean removeUnusedCode = isRemoveUnusedCode();
        boolean removeUnusedImports = refactoring.isRemoveUnusedImports();
        int steps = 0;
        int sz = resources.size();
        if (removeUnusedCode) {
            // Take 10 steps to process each resource
            steps = sz * 10; 
        } 
        if (removeUnusedImports) {
            // One step to remove unused imports
            steps += sz;
        }
        fireProgressListenerStart(AbstractRefactoring.PREPARE, steps);
        try {
            for (Iterator/*<Resource>*/ it = resources.iterator(); it.hasNext();) {
                Resource res = (Resource) it.next();
                if (removeUnusedCode) {
                    FileObject fo = JavaMetamodel.getManager().getFileObject(res);
                    Set/*<Resource>*/ candidates = new HashSet/*<Resource>*/();
                    Map/*<Element, List<Element>>*/ element2UsageList = new HashMap/*<Element, List<Element>>*/();
                    collectElements(res, candidates, element2UsageList, true);
                    // Go through all elements and remove those without usages
                    for (Iterator candidateIt = candidates.iterator(); candidateIt.hasNext();) {
                        Element el = (Element) candidateIt.next();
                        if (el instanceof TypeCast) {
                            refactoringElements.add(refactoring, new RemoveUnusedElement(fo, el, null, comment));
                            candidateIt.remove();
                        } else {
                            List usages = (List) element2UsageList.get(el);
                            if (usages.isEmpty()) {
                                refactoringElements.add(refactoring, new RemoveUnusedElement(fo, el, null, comment));
                                candidateIt.remove();
                            }
                        }
                    }
                    // For the elements with usages check if they may have usages between them. 
                    // If yes (a uses b and b uses a, but both are not needed, as noone else uses them)
                    // remove them in the right order (construct a dependency graph)
                    // TODO - implement
                }
                if (removeUnusedImports) {
                    refactoringElements.add(refactoring, new RemoveUnusedImportsElement(res));
                    fireProgressListenerStep();
                }
            }
        } finally {
            fireProgressListenerStop();
        }
        return null;
    }
    

    private boolean isRemoveUnusedCode() {
        boolean removeUnusedCode = 
              refactoring.isRemoveUnusedFields() || 
              refactoring.isRemoveUnusedCallableFeatures() ||
              refactoring.isRemoveUnusedClasses() ||
              refactoring.isRemoveUnusedLocalVars() ||
              refactoring.isRemoveRedundantCasts();
        return removeUnusedCode;
    }
    
    private static class RemoveUnusedImportsElement extends SimpleRefactoringElementImpl {
        
        private Resource resource;
        
        private String text;
        
        public RemoveUnusedImportsElement(Resource resource) {
            this.resource = resource;
            this.text = NbBundle.getMessage(CleanUpRefactoringPlugin.class, "TXT_RemoveUnusedImports");
        }
        
        public String getText() {
            return text;
        }
        
        public String getDisplayText() {
            return text;
        }
        
        public void performChange() {
            // TODO - remove imports from the same package
            Set/*<Type>*/ types = new HashSet/*<Type>*/();
            collectTypes(resource, types);
            Set importedNames = new HashSet();
            Set neededImportedNames = new HashSet();
            Import[] imports = (Import[]) resource.getImports().toArray(new Import[0]);
            
            for (int i = 0; i < imports.length; i++) {
                Import imp = imports[i];
                NamedElement importedNs = imp.getImportedNamespace();
                importedNames.add(importedNs.getName());
            }
            
            for (Iterator typesIt = types.iterator(); typesIt.hasNext();) {
                Type type = (Type) typesIt.next();
                String typeName = type.getName();
                String resourcePkgName = resource.getPackageName();
                if (resourcePkgName == null) {
                    resourcePkgName = "";
                }
                String typePkgName = type.getResource().getPackageName();
                if (typePkgName == null) {
                    typePkgName = "";
                }
                if (resourcePkgName.equals(typePkgName)) {
                    continue;
                }
                if (importedNames.contains(typeName)) {
                    neededImportedNames.add(typeName);
                } else  if (importedNames.contains(typePkgName)) {
                    neededImportedNames.add(typePkgName);
                }
            }
            
            for (ListIterator it = resource.getImports().listIterator(); it.hasNext();) {
                Import i = (Import) it.next();
                NamedElement importedNs = i.getImportedNamespace();
                if (importedNs instanceof ClassDefinition && "java.lang".equals(importedNs.getResource().getPackageName())) {
                    it.remove();
                } else if (!neededImportedNames.contains(importedNs.getName())) {
                    it.remove();
                }
            }
        }
        
        
        private void collectTypes(Element el, Set/*<Type>*/ types) {
            Iterator it;
            if (el instanceof ArrayReference) {
                el = ((ArrayReference) el).getParent();
            }
            if (el instanceof MultipartId) {
                MultipartId id = (MultipartId) el;
                List typeArgs = new ArrayList();
                while (id.getParent() != null) {
                    id = id.getParent();
                    typeArgs.addAll(id.getTypeArguments());
                }
                typeArgs.addAll(id.getTypeArguments());
                NamedElement ne = id.getElement();
                if (ne instanceof JavaClass) {
                    types.add(ne);
                }
                it = typeArgs.iterator();
            } else {
                it = el.getChildren().iterator();
            }
            while (it.hasNext()) {
                collectTypes((Element) it.next(), types);
            }
        }
        
        public Element getJavaElement() {
            return resource;
        }
        
        public FileObject getParentFile() {
            return JavaMetamodel.getManager().getFileObject(resource);
        }
        
        public PositionBounds getPosition() {
            return null;
        }
        
    }
    
    private static class RemoveUnusedElement extends SimpleRefactoringElementImpl {
        
        private Element element;
        
        private boolean commentOut;

        private FileObject fo;
        
        private String text;
        
        private Collection dependencies;
        
        public RemoveUnusedElement(FileObject fo, Element element, Collection dependencies, boolean commentOut) {
            this.element = element;
            this.fo = fo;
            this.commentOut = commentOut;
            this.dependencies = dependencies;
            if (element instanceof TypeCast) {
                initCastText();
            } else {
                initElementText();
            }
        }
        
        public String getText() {
            return text;
        }
        
        private void initCastText() {
            String action = commentOut ?
                NbBundle.getMessage(CleanUpRefactoringPlugin.class, "TXT_Comment") :
                NbBundle.getMessage(CleanUpRefactoringPlugin.class, "TXT_Remove");
            TypeCast tc = (TypeCast) element;
            String from = tc.getExpression().getType().getName();
            String to = tc.getType().getName();
            String code = new String(tc.getResource().getSourceText().substring(
                    tc.getStartOffset(), tc.getExpression().getEndOffset()));
            // Set the member variable
            text = NbBundle.getMessage(CleanUpRefactoringPlugin.class, "TXT_RemoveRedundantCast",
                    new Object[] { action, from, to, code});
        }
        
        private void initElementText() {
            String action = commentOut ? 
                NbBundle.getMessage(CleanUpRefactoringPlugin.class, "TXT_Comment") :
                NbBundle.getMessage(CleanUpRefactoringPlugin.class, "TXT_Remove");
            String type = null;
            if (element instanceof LocalVariable) {
                type = NbBundle.getMessage(CleanUpRefactoringPlugin.class, "TXT_LocalVariable");
            } else if (element instanceof Field) {
                type = NbBundle.getMessage(CleanUpRefactoringPlugin.class, "TXT_Field");
            } else if (element instanceof Method) {
                type = NbBundle.getMessage(CleanUpRefactoringPlugin.class, "TXT_Method");
            } else if (element instanceof Constructor) {
                type = NbBundle.getMessage(CleanUpRefactoringPlugin.class, "TXT_Constructor");
            } else if (element instanceof JavaClass) {
                type = NbBundle.getMessage(CleanUpRefactoringPlugin.class, "TXT_Class");
            }
            String name = getDisplayName((NamedElement) element);
            // Update the member variable
            text = NbBundle.getMessage(CleanUpRefactoringPlugin.class, 
                    "TXT_RemoveUnusedElement", // NOI18N
                    new Object[] { action, type, name, });
        }
        
        public String getDisplayText() {
            return text;
        }
        
        private String getDisplayName(NamedElement ne) {
            if (ne instanceof CallableFeature) {
                String name;
                if (ne instanceof Constructor) {
                    name = ((JavaClass) ((Constructor) ne).getDeclaringClass()).getSimpleName();
                } else {
                    name = ne.getName();
                }
                StringBuffer buff = new StringBuffer(name).append('(');
                Parameter[] p = (Parameter[]) ((CallableFeature) ne).getParameters().toArray(new Parameter[0]);
                for (int i = 0; i < p.length; i++) {
                    if (i > 0) {
                        buff.append(", "); // NOI18N
                    }
                    buff.append(p[i].getType().getName());
                }
                buff.append(')');
                return buff.toString();
            } else if (ne instanceof JavaClass) {
                return ((JavaClass) ne).getSimpleName();
            } else {
                return ne.getName();
            }
        }
        
        public void performChange() {
            if (dependencies != null) {
                for (Iterator it = dependencies.iterator(); it.hasNext();) {
                    Element el = (Element) it.next();
                    if (el.isValid()) {
                        return;
                    }
                }
            }
            if (element instanceof ClassMember) {
                if (commentOut) {
                    // TODO - implement
                } else {
                    ClassMember cm = (ClassMember) element;
                    cm.getDeclaringClass().replaceChild(cm, null);
                }
            } else if (element instanceof LocalVariable) {
                if (commentOut) {
                    // TODO - implement
                } else {
                    LocalVariable lv = (LocalVariable) element;
                    LocalVarDeclaration lvd = (LocalVarDeclaration) lv.refImmediateComposite();
                    List vars = lvd.getVariables();
                    if (vars.size() > 1) {
                        vars.remove(lv);
                    } else {
                        StatementBlock sb = (StatementBlock) lvd.refImmediateComposite();
                        sb.getStatements().remove(lvd);
                    }
                }
            } else if (element instanceof TypeCast) {
                TypeCast tc = (TypeCast) element;
                Object parent = tc.refImmediateComposite();
                while (parent != null && !(parent instanceof Element)) {
                    if (!(parent instanceof RefObject)) {
                        continue;
                    }
                    parent = ((RefObject) parent).refImmediateComposite();
                }
                if (parent != null) {
                    Element parentEl = (Element) parent;
                    Expression clone = (Expression) tc.getExpression().duplicate();
                    parentEl.replaceChild(tc, clone);
                }
            }
        }
        
        public Element getJavaElement() {
            if (element instanceof ClassMember) {
                return ((ClassMember) element).getDeclaringClass();
            } else if (element instanceof LocalVariable) {
                Object parent = element;
                while (parent != null && !(parent instanceof CallableFeature) && (parent instanceof RefObject)) {
                    parent = ((RefObject) parent).refImmediateComposite();
                }
                return parent != null ? (Element) parent : element.getResource();
            } else if (element instanceof TypeCast) {
                Object parent = element;
                while (parent != null && !(parent instanceof ClassMember) && (parent instanceof RefObject)) {
                    parent = ((RefObject) parent).refImmediateComposite();
                }
                return parent != null ? (Element) parent : element.getResource();
            } else {
                return element.getResource();
            }
        }
        
        public FileObject getParentFile() {
            return fo;
        }
        
        public PositionBounds getPosition() {
            return null;
        }
        
    }
        
    
    private void collectElements(
            Element el, Set/*<Resource>*/ removalCandidates, Map/*<Element, List<Element>>*/ element2UsageList, boolean fireProgress) {
        if (isRemovalCandidate(el, refactoring)) {
            addRemovalCandidate(el, null, removalCandidates, element2UsageList);
        } else if (el instanceof ElementReference) {
            Element actualElement = ((ElementReference) el).getElement();
            if (isRemovalCandidate(actualElement, refactoring)) {
                addRemovalCandidate(actualElement, el, removalCandidates, element2UsageList);
            }
        }
        Element[] children = (Element[]) el.getChildren().toArray(new Element[0]);
        double currentStep = 0;
        int lastStep = 0;
        try {
            // All iterations have to fire 10 progress steps. It doesn't matter
            // how many elements there are. 
            double increment = 10d / children.length;
            for (int i = 0; i < children.length; i++) {
                if (fireProgress) {
                    currentStep += increment;
                    if ((int) currentStep != lastStep) {
                        fireProgressListenerStep((int) currentStep - lastStep);
                        lastStep = (int) currentStep;
                    }
                }
                collectElements(children[i], removalCandidates, element2UsageList, false);
            }
        } finally {
            // Here we make sure that we didn't miss firing a progress step
            if (fireProgress && lastStep < 10) {
                fireProgressListenerStep(10 - lastStep);
            }
        }
    }
    
    private void addRemovalCandidate(Element candidate, Element user, 
            Set/*<Resource>*/ removalCandidates, Map/*<Element, List<Element>>*/ element2UsageList) {
        removalCandidates.add(candidate);
        addElementUser(candidate, user, element2UsageList);
    }

    private void addElementUser(Element element, Element user, Map element2UsageList) {
        List users = (List) element2UsageList.get(element);
        if (users == null) {
            users = new ArrayList();
            element2UsageList.put(element, users);
        }
        if (user != null) {
            users.add(user);
        }
    }
    
    private boolean isRemovalCandidate(Element element, CleanUpRefactoring refactoring) {
        if (element instanceof Field) {
            return isRemovableField((Field) element);
        } else if (element instanceof Method) {
            // Method
            boolean b = refactoring.isRemoveUnusedCallableFeatures() && (((Method) element).getModifiers() & Modifier.PRIVATE) != 0;
            if (!b) {
                return false;
            }
            // TODO - check return types && params
            return !isSerializationMethod((Method) element);
        } else if (element instanceof Constructor) {
            // Constructor
            Constructor constr = (Constructor) element;
            // constructor with 0 args could mean singleton
            return refactoring.isRemoveUnusedCallableFeatures() && ((constr.getModifiers() & Modifier.PRIVATE) != 0) && !constr.getParameters().isEmpty();
        } else if (element instanceof JavaClass) {
            // Class
            return refactoring.isRemoveUnusedClasses() && (((JavaClass) element).getModifiers() & Modifier.PRIVATE) != 0;
        } else if (element instanceof LocalVariable) {
            return isRemovableLocalVariable((LocalVariable) element);
        } else if (element instanceof TypeCast) {
            // Cast
            TypeCast tc = (TypeCast) element;
            return refactoring.isRemoveRedundantCasts() && isCastRedundant(tc.getExpression().getType(), tc.getType());
        } 
        return false;
    }
    
    private boolean isSerializationMethod(Method m) {
        String name = m.getName();
        boolean ro = "readObject".equals(name); // NOI18N
        boolean wo = "writeObject".equals(name); // NOI18N
        if (!(ro || wo)) {
            return false;
        }
        Parameter[] p = (Parameter[]) m.getParameters().toArray(new Parameter[0]);
        if (p.length != 1) {
            return false;
        }
        if (ro) {
            return "java.io.ObjectInputStream".equals(p[0].getType().getName()); // NOI18N
        } else {
            return "java.io.ObjectOutputStream".equals(p[0].getType().getName()); // NOI18N
        }
    }

    /**
     * Checks if a field may be removed
     */
    private boolean isRemovableField(Field f) {
        if (!refactoring.isRemoveUnusedFields()) {
            return false;
        }
        if ((f.getModifiers() & Modifier.PRIVATE) == 0) {
            return false;
        }
        if ("serialVersionUID".equals(f.getName())) { // NOI18N
            return false;
        }
        return isElementWithInitialValueRemovable(f.getInitialValue());
    }
    
    /** 
     * Local variables or fields with a initial value may not be removed
     * For example, one could have <code>boolean initialized = initialize();</code>
     * and do some initialization work in that method
     */
    private boolean isElementWithInitialValueRemovable(InitialValue val) {
        if (val == null) {
            return true;
        }
        if (val instanceof Literal) {
            return true;
        }
        if (val instanceof VariableAccess) {
            return true;
        }
        return false;
    }
    
    /**
     * Checks if a local variable may be removed
     */
    private boolean isRemovableLocalVariable(LocalVariable var) {
        if (!refactoring.isRemoveUnusedLocalVars()) {
            return false;
        }
        return isElementWithInitialValueRemovable(var.getInitialValue());
    }
    
    private boolean isCastRedundant(Type fromType, Type toType) {
        if (fromType instanceof PrimitiveType && toType instanceof PrimitiveType) {
            // TODO - implement
        } else if (fromType instanceof ParameterizedType && toType instanceof ClassDefinition) {
            ClassDefinition fromClass = ((ParameterizedType) fromType).getDefinition();
            ClassDefinition toClass = (ClassDefinition) toType;
            if (fromClass.isSubTypeOf(toClass)) {
                return true;
            }
        } else if (fromType instanceof ClassDefinition && toType instanceof ClassDefinition) {
            ClassDefinition fromClass = (ClassDefinition) fromType;
            ClassDefinition toClass = (ClassDefinition) toType;
            if (fromClass.isSubTypeOf(toClass)) {
                return true;
            }
        }
        return false;
    } 
    
}
