/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.refactoring.experimental.plugins;

import java.text.MessageFormat;
import java.util.*;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.jmi.javamodel.ClassDefinition;
import org.netbeans.jmi.javamodel.LocalVariable;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.netbeans.modules.javacore.internalapi.ProgressListener;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.refactoring.api.*;
import org.openide.filesystems.FileObject;
import org.openide.text.PositionBounds;
import org.openide.util.NbBundle;
import org.netbeans.modules.refactoring.spi.RefactoringElementsBag;
import org.netbeans.modules.refactoring.plugins.*;
import org.netbeans.modules.refactoring.experimental.IntroduceVariableRefactoring;
import org.netbeans.modules.refactoring.spi.SimpleRefactoringElementImpl;

/**
 *
 * @author Daniel Prusa
 */
public class IntroduceVariableRefactoringPlugin extends JavaRefactoringPlugin implements ProgressListener {
    private Expression expression;
    private String variableName;
    private Type variableType;
    private boolean replaceAll;
    private boolean declareFinal;
    private BehavioralFeature enclosingMethod = null;
    private Set localVariables = null;
    
    private IntroduceVariableRefactoring refactoring;
    
    public IntroduceVariableRefactoringPlugin(IntroduceVariableRefactoring introduce) {
        this.refactoring = introduce;
        expression = introduce.getExpression();
    }
    
    public Problem checkParameters() {
        return setParameters(
            refactoring.getVariableName(),
            refactoring.getVariableType(),
            refactoring.isReplaceAll(),
            refactoring.isFinal()
        );
    }
    
    public Problem fastCheckParameters() {
        String name = refactoring.getVariableName();
        if (name == null || name.length() == 0) {
            return new Problem(true, getString("ERR_EmptyIdentifier"));
        }
        if (!org.openide.util.Utilities.isJavaIdentifier(name)) {
            String msg = new MessageFormat(getString("ERR_InvalidIdentifier")).format(new Object[] {name}); //NOI18N
            return new Problem(true, msg);
        }
        return null;
    }
    
    public Problem preCheck() {
        fireProgressListenerStart(refactoring.PRE_CHECK, 2);
        Problem problem = null;
        try {
            if (expression == null) {
                problem = new Problem(true, getString("ERR_ExpressionNotSelected"));
                fireProgressListenerStep();
            } else {
                BehavioralFeature method = getMethod();
                fireProgressListenerStep();
                if (method == null) {
                    problem = new Problem(true, getString("ERR_NotInBehavioralFeature"));
                } else {
                    Object comp = expression.refImmediateComposite();
                    if ((comp instanceof ExpressionStatement) ||
                        (!(comp instanceof Expression) && expression instanceof TypeReference) ||
                        (comp instanceof NewClassExpression) || (comp instanceof NewArrayExpression)) {
                            problem = new Problem(true, getString("ERR_CannotReplaceExpression"));
                    }
                }
            }
            fireProgressListenerStep();
        } finally {
            fireProgressListenerStop();
        }
        return problem;
    }
    
    private Problem checkParameters(String name, Type type, boolean replaceAll, boolean isFinal) {
        collectLocalVariables();
        if (localVariables.contains(name)) {
            String msg = new MessageFormat(getString("ERR_NameClashes")).format(new Object[] {name});
            return new Problem(true, msg);
        }
        return null;
    }
    
    private Problem setParameters(String name, Type type, boolean replaceAll, boolean isFinal) {
        fireProgressListenerStart(refactoring.PARAMETERS_CHECK, 1);
        try {
            fireProgressListenerStep();
            
            variableName = name;
            variableType = type;
            this.replaceAll = replaceAll;
            declareFinal = isFinal;
            
            Problem result = checkParameters(name, type, replaceAll, isFinal);
            return result;
        } finally {
            fireProgressListenerStop();
        }
    }    
    
    public Problem prepare(RefactoringElementsBag elements) {
        JavaModelPackage modelPackage = (JavaModelPackage) expression.refOutermostPackage();
        IntroduceVariableElement elem = new IntroduceVariableElement(expression, variableName, variableType, declareFinal);
        elements.add(refactoring, elem);
        if (refactoring.isReplaceAll()) {
            BehavioralFeature method = getMethod();
            collectOccurrences(elements,  method, refactoring.getVariableName());
        } else {
            ReplaceExpressionElement replaceElem = new ReplaceExpressionElement(expression, refactoring.getVariableName());
            elements.add(refactoring, replaceElem);
        }
        return null;
    }
    
    // helper methods ..........................................................
    
    private BehavioralFeature getMethod() {
        if (enclosingMethod == null) {
            Element elem = expression;
            while (!(elem instanceof BehavioralFeature) && (elem != null)) {
                elem = (Element)elem.refImmediateComposite();
            }
            enclosingMethod = (BehavioralFeature)elem;
        }
        return enclosingMethod;
    }
    
    private void collectLocalVariables() {
        localVariables = new HashSet();
        BehavioralFeature method = getMethod();
        if (method != null) {
            collectLocalVariables(localVariables, method);
        }
    }
    
    private void collectLocalVariables(Set localVars, Element elem) {
        /*
        if (elem instanceof VariableAccess) {
            NamedElement nelem = ((VariableAccess) elem).getElement();
            if (nelem instanceof Field) {
                localVars.add(nelem.getName());
            }
        } else */ if (elem instanceof LocalVariable) {
            LocalVariable localVar = (LocalVariable) elem;
            localVars.add(localVar.getName());
            InitialValue initValue = localVar.getInitialValue();
            if (initValue != null) {
                collectLocalVariables(localVars, initValue);
            }
        } else {
            for (Iterator iter = elem.getChildren().iterator(); iter.hasNext(); ) {
                collectLocalVariables(localVars, (Element)iter.next());
            }
        }
    }
    
    private void collectOccurrences(RefactoringElementsBag elements, Element root, String name) {
        for (Iterator iter = root.getChildren().iterator(); iter.hasNext(); ) {
            Element elem = (Element) iter.next();
            if (elem instanceof Expression) {
                findMatchingExpressions(elements, (Expression) elem, name);
            } else if (!(elem instanceof ClassDefinition)) {
                collectOccurrences(elements, elem, name);
            }
        } // for
    }

    private void findMatchingExpressions(RefactoringElementsBag elements, Expression root, String name) {
        // [TODO] exclude all anonymous classes
        if (compareExpressions(root, expression)) {
            ReplaceExpressionElement replaceElem = new ReplaceExpressionElement(root, name);
            elements.add(refactoring, replaceElem);
            return;
        }
        for (Iterator iter = root.getChildren().iterator(); iter.hasNext(); ) {
            Element elem = (Element) iter.next();
            if (elem instanceof Expression) {
                findMatchingExpressions(elements, (Expression) elem, name);
            } else {
                return;
            }
        } // for
    }
    
    public static boolean compareExpressions(Expression expr_1, Expression expr_2) {
        if (expr_1 == null) {
            return expr_2 == null;
        }
        if (expr_2 == null) {
            return false; // the previous condition ensures that expr_1 is not null
        }
        if (expr_1.getClass() != expr_2.getClass()) {
            return false;
        }
        if (expr_1 instanceof NullLiteral) {
            return true;
        } else if (expr_1 instanceof BooleanLiteral) {
            return ((BooleanLiteral) expr_1).isValue() == ((BooleanLiteral) expr_2).isValue();
        } else if (expr_1 instanceof StringLiteral) {
            return ((StringLiteral) expr_1).getValue().equals(((StringLiteral) expr_2).getValue());
        } else if (expr_1 instanceof IntLiteral) {
            return ((IntLiteral) expr_1).getValue() == ((IntLiteral) expr_2).getValue();
        } else if (expr_1 instanceof DoubleLiteral) {
            return ((DoubleLiteral) expr_1).getValue() == ((DoubleLiteral) expr_2).getValue();
        } else if (expr_1 instanceof LongLiteral) {
            return ((LongLiteral) expr_1).getValue() == ((LongLiteral) expr_2).getValue();
        } else if (expr_1 instanceof CharLiteral) {
            return ((CharLiteral) expr_1).getValue() == ((CharLiteral) expr_2).getValue();
        } else if (expr_1 instanceof FloatLiteral) {
            return ((FloatLiteral) expr_1).getValue() == ((FloatLiteral) expr_2).getValue();
        } else if (expr_1 instanceof MultipartId) {
            NamedElement elem_1 = ((MultipartId) expr_1).getElement();
            NamedElement elem_2 = ((MultipartId) expr_2).getElement();
            if (!(elem_1 instanceof UnresolvedClass)) {
                return elem_1 == elem_2;
            }
            if (!(elem_2 instanceof UnresolvedClass)) {
                return false;
            }
            return elem_1.getName().equals(elem_2.getName());
        } else if (expr_1 instanceof ArrayReference) {
            if (((ArrayReference) expr_1).getDimCount() != ((ArrayReference) expr_2).getDimCount()) {
                return false;
            }
            NamedElement elem_1 = ((TypeReference) expr_1).getElement();
            NamedElement elem_2 = ((TypeReference) expr_2).getElement();
            if (!(elem_1 instanceof UnresolvedClass)) {
                return elem_1 == elem_2;
            }
            if (!(elem_2 instanceof UnresolvedClass)) {
                return false;
            }
            return elem_1.getName().equals(elem_2.getName());
        } else if (expr_1 instanceof PrefixExpression) {
            Operator op_1 = ((PrefixExpression) expr_1).getOperator();
            Operator op_2 = ((PrefixExpression) expr_2).getOperator();
            if (!op_1.equals(op_2)) {
                return false;
            }
        } else if (expr_1 instanceof InfixExpression) {
            Operator op_1 = ((InfixExpression) expr_1).getOperator();
            Operator op_2 = ((InfixExpression) expr_2).getOperator();
            if (!op_1.equals(op_2)) {
                return false;
            }
        } else if (expr_1 instanceof MethodInvocation) {
            String name_1 = ((MethodInvocation) expr_1).getName();
            String name_2 = ((MethodInvocation) expr_2).getName();
            if (!name_1.equals(name_2)) {
                return false;
            }
        }
        List list_1 = expr_1.getChildren();
        List list_2 = expr_2.getChildren();
        if (list_1.size() != list_2.size()) {
            return false;
        }
        Iterator iter_1 = list_1.iterator();
        Iterator iter_2 = list_2.iterator();
        while (iter_1.hasNext()) {
            if (!compareExpressions((Expression) iter_1.next(), (Expression) iter_2.next())) {
                return false;
            }
        }
        return true;
    }
    
    private static final String getString(String key) {
        return NbBundle.getMessage(IntroduceVariableRefactoringPlugin.class, key);
    }
    
    public void start(org.netbeans.modules.javacore.internalapi.ProgressEvent event) {
        fireProgressListenerStart(event.getOperationType(), event.getCount());
    }
    
    public void step(org.netbeans.modules.javacore.internalapi.ProgressEvent event) {
        fireProgressListenerStep();
    }
    
    public void stop(org.netbeans.modules.javacore.internalapi.ProgressEvent event) {
        fireProgressListenerStop();
    }

    // IntroduceVariableElement .................................................
    private class IntroduceVariableElement extends SimpleRefactoringElementImpl {
        
        private final String text;
        private PositionBounds bounds = null;
        
        private Expression initialExpr;
        private String name;
        private Type type;
        private boolean isFinal;
        
        public IntroduceVariableElement(Expression expr, String name, Type type, boolean isFinal) {
            initialExpr = expr;
            this.name = name;
            this.type = type;
            this.isFinal = isFinal;
            text = MessageFormat.format(getString("TXT_IntroduceVariable"), new Object[] {name});
        }
            
        private CallableFeature getMethod() {
            Element elem = initialExpr;
            while (!(elem instanceof CallableFeature) && (elem != null)) {
                elem = (Element)elem.refImmediateComposite();
            }
            return (CallableFeature)elem;
        }
        
        public String getDisplayText() {
            return text;
        }
        
        public Element getJavaElement() {
            return JavaModelUtil.getDeclaringFeature(initialExpr);
        }
        
        public PositionBounds getPosition() {
            if (bounds == null) {
                bounds = JavaMetamodel.getManager().getElementPosition(initialExpr);
            }
            return null;
        }
        
        public String getText() {
            return getDisplayText();
        }
        
        public void performChange() {
            CallableFeature method = getMethod();
            if (method == null) {
                return;
            }
            JavaModelPackage modelPackage = (JavaModelPackage) method.refOutermostPackage();
            LocalVarDeclarationClass localVarDeclProxy = modelPackage.getLocalVarDeclaration();
            TypeReference typeRef = typeToTypeReference(modelPackage, type);
            LocalVariableClass localVarProxy = modelPackage.getLocalVariable();
            InitialValue initValue = (InitialValue)initialExpr.duplicate();
            LocalVariable localVar = localVarProxy.createLocalVariable(name, null, isFinal, typeRef, 0, initValue, null);
            List varList = new ArrayList();
            varList.add(localVar);
            LocalVarDeclaration localVarDecl = localVarDeclProxy.createLocalVarDeclaration(isFinal, typeRef, varList);
            method.getBody().getStatements().add(0, localVarDecl);
        }
        
        public FileObject getParentFile() {
            Resource res = getMethod().getResource();
            return JavaMetamodel.getManager().getFileObject(res);
        }
        
        private TypeReference typeToTypeReference(JavaModelPackage modelPackage, Type type) {
            if (type instanceof Array) {
                ArrayReferenceClass arrayProxy = modelPackage.getArrayReference();
                int dims = 0;
                Type elemType = type;
                while (elemType instanceof Array) {
                    dims++;
                    elemType = ((Array) elemType).getType();
                }
                String name = elemType instanceof JavaClass ? ((JavaClass) elemType).getSimpleName() : elemType.getName();
                MultipartId id = modelPackage.getMultipartId().createMultipartId(name, null, null);
                return arrayProxy.createArrayReference(null, id, dims);
            } else {
                MultipartIdClass idProxy = modelPackage.getMultipartId();
                String name = type instanceof JavaClass ? ((JavaClass) type).getSimpleName() : type.getName();
                return idProxy.createMultipartId(name, null, null);
            }
        }
        
    } // IntroduceVariableElement
    
    // ReplaceExpressionElement .................................................
    private class ReplaceExpressionElement extends SimpleRefactoringElementImpl {
        
        private final String text;
        private PositionBounds bounds = null;
        private Resource res = null;
        private Expression expression;
        private String localVarName;
        
        public ReplaceExpressionElement(Expression expression, String localVarName) {
            this.expression = expression;
            this.localVarName = localVarName;
            text = MessageFormat.format(getString("TXT_ReplaceExpression"), new Object[] {localVarName});
        }
         
        public String getDisplayText() {
            return text;
        }
        
        public Element getJavaElement() {
            return JavaModelUtil.getDeclaringFeature(expression);
        }
        
        public PositionBounds getPosition() {
            if (bounds == null) {
                bounds = JavaMetamodel.getManager().getElementPosition(expression);
            }
            return bounds;
        }
        
        public String getText() {
            return getDisplayText();
        }
        
        public void performChange() {
            JavaModelPackage modelPackage = (JavaModelPackage) expression.refOutermostPackage();
            VariableAccessClass proxy = modelPackage.getVariableAccess();
            VariableAccess varAccess = proxy.createVariableAccess(localVarName, null, false);
            Element comp = (Element) expression.refImmediateComposite();
            comp.replaceChild(expression, varAccess);
        }

        public FileObject getParentFile() {
            return JavaMetamodel.getManager().getFileObject(expression.getResource());
        }

    } // ReplaceExpressionElement
    
}
