/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.refactoring;

import java.lang.reflect.Modifier;
import java.util.*;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ui.OpenProjects;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.GuardedQuery;
import org.netbeans.modules.javacore.jmiimpl.javamodel.MethodImpl;
import org.netbeans.modules.refactoring.spi.RefactoringElementImplementation;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataFolder;
import org.openide.loaders.DataObject;
import org.openide.text.PositionBounds;
import org.openide.util.Utilities;


/**
 * Methods for validating parameters. Used in different refactorings, e.g.
 * rename class, change method signature etc.
 *
 * @author  Dan Prusa
 * @author  Pavel Flaska
 */
public final class CheckUtils {
    
    //Copied from org.netbeans.modules.editor.java.JMIUtils
    private static boolean is15Enabled = true;
    private static final boolean[][] assignVals = new boolean[][] {
        new boolean[] { true, false, false, false, false, false, false, false, false}, // boolean
        new boolean[] { false, true, false, true, true, true, true, true, false}, // byte
        new boolean[] { false, false, true, true, true, true, true, false, false}, // char
        new boolean[] { false, false, false, true, false, false, false, false, false}, // double
        new boolean[] { false, false, false, true, true, false, false, false, false}, // float
        new boolean[] { false, false, false, true, true, true, true, false, false}, // int
        new boolean[] { false, false, false, true, true, false, true, false, false}, // long
        new boolean[] { false, false, false, true, true, true, true, true, false}, // short
        new boolean[] { false, false, false, false, false, false, false, false, true} // void
    };
    
    
    /**
     *
     * @param feature
     * @return returns all variables used in given scope
     */
    public static Set getAllVariableNames(Feature feature) {
        Set names=new HashSet();
        LinkedList children=new LinkedList();
        
        children.add(feature);
        while(!children.isEmpty()) {
            Element el=(Element)children.removeFirst();
            
            if (el instanceof VariableAccess) {
                VariableAccess varAcc=(VariableAccess)el;
                
                if (varAcc.getParentClass()==null && !varAcc.isHasSuper()) {
                    names.add(varAcc.getName());
                }
            }
            children.addAll(el.getChildren());
        }
        return names;
    }
    
    public static boolean isFromLibrary(Resource res) {
        CodebaseClass cbClass=((JavaModelPackage)res.refImmediatePackage()).getCodebase();
        Codebase cb=(Codebase)cbClass.refAllOfClass().iterator().next();
        return cb.isLibrary();
    }
    
    public static boolean membersEqual(NamedElement m1, NamedElement m2) {
        if (!m1.getClass().equals(m2.getClass())) {
            return false;
        }
        if (m1 instanceof Method) {
            return ((MethodImpl) m1).signatureEquals((Method) m2);
        }
        return m1.getName().equals(m2.getName());
    }
    
    /**
     * Check if the provided method is overriden by another method.
     *
     * @param method
     * @param name
     * @param argTypes
     *
     * @return  method which overrides provided method, otherwise null
     */
    public static Collection isOverridden(Method method, String name, List argTypes) {
        Collection res = new HashSet();
        
        if (!isVirtual(method))
            return res;
        
        ClassDefinition jc = method.getDeclaringClass();
        LinkedList subtypes = new LinkedList();
        addSubtypes(jc, subtypes);
        while (subtypes.size() > 0) {
            jc = (ClassDefinition) subtypes.removeFirst();
            Method m = jc.getMethod(name, argTypes, false);
            if ((m != null) && isVirtual(m))
                res.add(m);
            addSubtypes(jc, subtypes);
        }
        return res;
    }
    
    /**
     * @param method
     * @param name
     * @param argTypes
     * @param findFinal
     * @return
     */
    public static Collection overrides(Method method, String name, List argTypes, boolean findFinal) {
        Collection res = new HashSet();
        
        if (!isVirtual(method))
            return res;
        
        ClassDefinition jc = method.getDeclaringClass();
        LinkedList supertypes = new LinkedList();
        
        supertypes.addAll(jc.getInterfaces());
        jc = jc.getSuperClass();
        if (jc != null)
            supertypes.add(jc);
        while (supertypes.size() > 0) {
            jc = (ClassDefinition) supertypes.removeFirst();
            Method m =  jc.getMethod(name, argTypes, false);
            if ((m != null) && isVirtual(m)) {
                if ((m.getModifiers() & Modifier.FINAL) > 0) {
                    res.add(m);
                    continue;
                } else if (res == null) {
                    res.add(m);
                    if (!findFinal)
                        continue;
                }
            }
            supertypes.addAll(jc.getInterfaces());
            jc = jc.getSuperClass();
            if (jc != null)
                supertypes.add(jc);
        }
        return res;
    }
    
    /**
     * Checks, if method or constructor has variable arguments paramater.
     * If so, returns true, otherwise false.
     *
     * @param   constructor or method to check
     * @return  true, if callable contain variable argument parameter
     */
    public static boolean hasVarArgs(CallableFeature callable) {
        for (Iterator it = callable.getParameters().iterator(); it.hasNext(); ) {
            if (((Parameter) it.next()).isVarArg()) {
                return true;
            }
        }
        return false;
    }
    
    public static boolean isVirtual(Feature feature) {
        int mod = feature.getModifiers();
        return !Modifier.isPrivate(mod);
    }
            
    public static JavaClass getClass(JavaPackage pkg, String clsName) {
        for (Iterator it = pkg.getResources().iterator(); it.hasNext();) {
            Resource res = (Resource) it.next();
            for (Iterator it2 = res.getClassifiers().iterator(); it2.hasNext();) {
                Object obj = it2.next();
                if (obj instanceof JavaClass) {
                    if (clsName.equals(((JavaClass) obj).getName())) {
                        return (JavaClass) obj;
                    }
                }
            } // for
        } // for
        return null;
    }
    
    public static boolean isFolderEmpty(FileObject f) {
        if (f==null || f.isVirtual() || !f.isValid()) {
            return true;
        }
        DataFolder dob = DataFolder.findFolder(f);
        DataObject children[] = dob.getChildren();
        for (int i = 0; i < children.length; i++) {
            if (!(children[i] instanceof DataFolder))
                return false;
        }
        return true;
    }
    
    public static boolean isElementReadOnly(Element e) {
        if (e==null) {
            throw new IllegalArgumentException("Cannot pass null as an argument"); //NOI18N
        }
        Resource res = e.getResource();
        if (res!=null) {
            FileObject f = JavaModel.getFileObject(res);
            if (f != null)
                return !f.canWrite();
        }
        return false;
    }
    
    public static boolean isValidPackageName(String name) {
        StringTokenizer tokenizer = new StringTokenizer(name, "."); // NOI18N
        while (tokenizer.hasMoreTokens()) {
            if (!Utilities.isJavaIdentifier(tokenizer.nextToken())) {
                return false;
            }
        }
        return true;
    }
    
    
    public static boolean isRefactoringElementReadOnly(RefactoringElementImplementation el) {
        Element elem = el.getJavaElement();
        if (elem == null) {
            return !el.getParentFile().canWrite();
        } else {
            return isElementReadOnly(elem);
        }
    }
    
    public static boolean isRefactoringElementGuarded(RefactoringElementImplementation el) {
        Element elem = el.getJavaElement();
        if (elem != null) {
            Resource res = elem.getResource();
            if (res!=null) {
                PositionBounds pos = el.getPosition();
                if (pos!=null)
                    return GuardedQuery.isSectionGuarded(res, pos);
            }
        }
        return false;
    }
    
    public static boolean isElementInOpenProject(Element el) {
        assert el != null;
        Resource r = el.getResource();
        assert r != null: "el.getResource() returned null, el = " + el.toString();
        FileObject f = JavaModel.getFileObject(r);
        assert f!=null : "JavaModel.getFileObject(r) returned null, r = " + r.getName();
        Project p = FileOwnerQuery.getOwner(f);
        Project[] opened = OpenProjects.getDefault().getOpenProjects();
        for (int i = 0; i<opened.length; i++) {
            if (p==opened[i])
                return true;
        }
        return false;
    }
    
    
    public static String htmlize(String input) {
        String temp = org.openide.util.Utilities.replaceString(input, "<", "&lt;"); // NOI18N
        temp = org.openide.util.Utilities.replaceString(temp, ">", "&gt;"); // NOI18N
        return temp;
    }
    
    public static String trimToEnd(String text) {
        char[] str=text.toCharArray();
        int end = str.length;
        while ((end > 0) && Character.isWhitespace(str[end - 1])) {
            end--;
        }
        return text.substring(0, end);
    }
    
    /**
     * @return true if folder has children, which are not folders
     */
    public static boolean hasChildren(DataFolder folder) {
        for (Enumeration en = folder.children(true); en.hasMoreElements();) {
            if (!(en.nextElement() instanceof DataFolder))
                return true;
        }
        return false;
    }
    
    //Copied from JMIUtils under org.netbeans.modules.editor.java. There are
    //9 private static methods later in this file that support this method.
    
    public static boolean isAccessible(ClassDefinition cls, JavaClass from) {
        if (cls == null)
            return false;
        if (from == null)
            return true;
        while (cls instanceof Array) {
            Type typ = ((Array)cls).getType();
            if (typ instanceof ClassDefinition)
                cls = (ClassDefinition)typ;
            else
                return true;
        }
        return ((JavaClass)cls).isInner() ? isAccessible((JavaClass)cls, ((JavaClass)cls).getDeclaringClass(), from) : isTopLevelClassAccessible((JavaClass)cls, from);
    }
    
    public static boolean isSubTypeOf(JavaClass thisCls, JavaClass otherCls) {
        if (otherCls instanceof TypeParameter) {
            if (!isSubTypeOf(thisCls, otherCls.getSuperClass()))
                return false;
            for (Iterator it = otherCls.getInterfaces().iterator(); it.hasNext();) {
                if (!isSubTypeOf(thisCls, (JavaClass)it.next()))
                    return false;
            }
            return true;
        }
        if (thisCls.isSubTypeOf(otherCls)) {
            List thisParams = thisCls instanceof ParameterizedType ? ((ParameterizedType)thisCls).getParameters() : new ArrayList();
            List otherParams = otherCls instanceof ParameterizedType ? ((ParameterizedType)otherCls).getParameters() : new ArrayList();
            if (thisParams.size() != otherParams.size())
                return false;
            for (Iterator it = thisParams.iterator(), itt = otherParams.iterator(); it.hasNext();) {
                if (!isAssignable((Type)it.next(), (Type)itt.next()))
                    return false;
            }
            return true;
        }
        return false;
    }
    
    public static boolean isAssignable(Type from, Type to) {
        if (from == null || to == null)
            return false;
        if (from.equals(to))
            return true;
        if (from instanceof Array && to instanceof Array)
            return isAssignable(((Array)from).getType(), ((Array)to).getType());
        if (to instanceof PrimitiveType && (is15Enabled || from instanceof PrimitiveType))
            return assignVals[getPrimitiveTypeIdx(from)][getPrimitiveTypeIdx(to)];
        if (to.getName().equals("java.lang.Object") && (is15Enabled || !(from instanceof PrimitiveType))) // NOI18N
            return true;
        if (to instanceof JavaClass && is15Enabled && from instanceof PrimitiveType)
            return isAssignable(getObjectType((PrimitiveType)from), to);
        if (from instanceof JavaClass && to instanceof JavaClass)
            return isSubTypeOf((JavaClass)from, (JavaClass)to);
        return false;
    }
    
    
    
    ////////////////////////////////////////////////////////////////////////////
    // INTERNAL METHODS
    ////////////////////////////////////////////////////////////////////////////
    private static void addSubtypes(ClassDefinition cd, List list) {
        if (!(cd instanceof JavaClass)) {
            return;
        }
        JavaClass jc = (JavaClass) cd;
        Collection desc = null;
        if (jc.isInterface()) {
            desc = jc.getImplementors();
        } else {
            desc = jc.getSubClasses();
        }
        list.addAll(desc);
    }
    
    //What follows is a blind, blatant copy of methods from JMIUtils under
    //org.netbeans.modules.editor.java. It's unfortunate that we need to
    //have duplication of check utilities across plugins.
    
    private static boolean isAccessible(Feature feature, ClassDefinition cls, JavaClass from) {
        ClassDefinition decCls = feature.getDeclaringClass();
        AccessibilityDescriptor desc = new AccessibilityDescriptor(decCls, cls, from);
        
        return desc.isAccessible(feature.getModifiers());
    }
    
    private static class AccessibilityDescriptor {
        int mprotected=-1, mpackprivate=-1, mprivate=-1;
        ClassDefinition decCls;
        ClassDefinition cls;
        JavaClass from;
        
        AccessibilityDescriptor(ClassDefinition decCls,ClassDefinition cls, JavaClass from) {
            if (from == null) {
                mprotected = mpackprivate = mprivate = 1;
                return;
            }
            if (cls instanceof TypeParameter)
                cls = decCls;
            if (cls != null && !from.equals(cls) && !(CheckUtils.isAccessible(cls, from))) {
                mprotected = mpackprivate = mprivate = 0;
                return;
            }
            this.decCls = decCls;
            this.cls = cls;
            this.from = from;
        }
        
        private void computeProtectedAccess() {
//            JavaClass jc = from;
//            while (jc != null) {
//                NOTE: We do not check for assignability. Visibility is our primary concern
//                if (isAssignable(jc, decCls) && (cls == null || isAssignable(cls, jc))) {
//                    mprotected = 1;
//                    return;
//                }
//                jc = (JavaClass)jc.getDeclaringClass();
//            }
//            mprotected = 0;
        }
        
        private void computePackagePrivateAccess() {
            String fromPkgName = getPackageName(from);
            String pkgName = getPackageName(decCls);
            boolean mpackpriv = fromPkgName.equals(pkgName);
            
            if (mpackpriv)
                mprotected = mpackprivate = 1;
            else
                mpackprivate = 0;
        }
        
        private void computePrivateAccess() {
            boolean mpriv = getOutermostClass(from).equals(decCls instanceof JavaClass ? getOutermostClass((JavaClass)decCls) : null);
            if (mpriv)
                mprotected = mpackprivate = mprivate = 1;
            else
                mprivate = 0;
        }
        
        private boolean isAccessible(int modifiers) {
            if (Modifier.isPublic(modifiers)) {
                return true;
            }
            if (mprivate == -1)
                computePrivateAccess();
            if (mprivate == 1) {
                return true;
            }
            if (Modifier.isPrivate(modifiers)) {
                return false;
            }
            if (mpackprivate == -1)
                computePackagePrivateAccess();
            if (mpackprivate == 1) {
                return true;
            }
            if (!Modifier.isProtected(modifiers)) {     // pavkage private
                return false;
            }
//            We do not check for protected access through  computeProtectedAccess
//            as that only uses an assignability test to determine accesibility
//            if (mprotected == -1)
//                computeProtectedAccess();
            return mprotected == 1;
        }
    }
    
    private static String getPackageName(ClassDefinition jc) {
        if (jc instanceof UnresolvedClass) {
            String name = jc.getName();
            int index = name.lastIndexOf('.');
            return index < 0 ? "" : name.substring(0, index);
        }
        if (jc instanceof JavaClass) {
            Resource res = jc.getResource();
            if (res != null){
                String result = res.getPackageName();
                if (result != null) {
                    return result;
                }
            }
        }
        return "";
    }
    
    private static JavaClass getOutermostClass(JavaClass jc) {
        while (jc.isInner()) {
            JavaClass decl = (JavaClass)jc.getDeclaringClass();
            if (decl == null)
                return jc;
            jc = decl;
        }
        return jc;
    }
    
    private static boolean isTopLevelClassAccessible(JavaClass cls, JavaClass from) {
        if (cls == null)
            return false;
        if (from == null)
            return true;
        if ((cls.getModifiers() & Modifier.PUBLIC) != 0)
            return true;
        String pkgName = getPackageName(cls);
        String fromPkgName = getPackageName(from);
        return pkgName.equals(fromPkgName);
    }
    
    private static Type getObjectType(PrimitiveType type) {
        PrimitiveTypeKind kind = type.getKind();
        if (PrimitiveTypeKindEnum.BOOLEAN.equals(kind)) return resolveType("java.lang.Boolean"); // NOI18N
        if (PrimitiveTypeKindEnum.BYTE.equals(kind)) return resolveType("java.lang.Byte"); // NOI18N
        if (PrimitiveTypeKindEnum.CHAR.equals(kind)) return resolveType("java.lang.Char"); // NOI18N
        if (PrimitiveTypeKindEnum.DOUBLE.equals(kind)) return resolveType("java.lang.Double"); // NOI18N
        if (PrimitiveTypeKindEnum.FLOAT.equals(kind)) return resolveType("java.lang.Float"); // NOI18N
        if (PrimitiveTypeKindEnum.INT.equals(kind)) return resolveType("java.lang.Integer"); // NOI18N
        if (PrimitiveTypeKindEnum.LONG.equals(kind)) return resolveType("java.lang.Long"); // NOI18N
        if (PrimitiveTypeKindEnum.SHORT.equals(kind)) return resolveType("java.lang.Short"); // NOI18N
        return null;
    }
    
    private static Type resolveType(String typeName) {
        if (!"null".equals(typeName)) { // NOI18N
            return JavaModel.getDefaultExtent().getType().resolve(typeName);
        }
        return null;
    }
    
    private static byte getPrimitiveTypeIdx(Type type) {
        PrimitiveTypeKind kind = type instanceof PrimitiveType ? ((PrimitiveType)type).getKind() : null;
        if ("java.lang.Boolean".equals(type.getName()) || PrimitiveTypeKindEnum.BOOLEAN.equals(kind)) return 0; // NOI18N
        if ("java.lang.Byte".equals(type.getName()) || PrimitiveTypeKindEnum.BYTE.equals(kind)) return 1; // NOI18N
        if ("java.lang.Char".equals(type.getName()) || PrimitiveTypeKindEnum.CHAR.equals(kind)) return 2; // NOI18N
        if ("java.lang.Double".equals(type.getName()) || PrimitiveTypeKindEnum.DOUBLE.equals(kind)) return 3; // NOI18N
        if ("java.lang.Float".equals(type.getName()) || PrimitiveTypeKindEnum.FLOAT.equals(kind)) return 4; // NOI18N
        if ("java.lang.Integer".equals(type.getName()) || PrimitiveTypeKindEnum.INT.equals(kind)) return 5; // NOI18N
        if ("java.lang.Long".equals(type.getName()) || PrimitiveTypeKindEnum.LONG.equals(kind)) return 6; // NOI18N
        if ("java.lang.Short".equals(type.getName()) || PrimitiveTypeKindEnum.SHORT.equals(kind)) return 7; // NOI18N
        return 8;
    }
    
}
