/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.refactoring.api;

import java.lang.reflect.Modifier;
import org.netbeans.jmi.javamodel.Feature;
import org.netbeans.jmi.javamodel.Field;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.Method;
import org.netbeans.jmi.javamodel.NamedElement;
import org.netbeans.jmi.javamodel.ParameterizedType;
import org.netbeans.modules.refactoring.classpath.Util;

/** Extract Interface Refactoring implementation class.
 *
 * @author Martin Matula
 */
public final class ExtractInterfaceRefactoring extends AbstractRefactoring {
    private static final NamedElement[] EMPTY_MEMBERS = new NamedElement[0];

    // source type
    private final JavaClass sourceType;
    // name of the new class to be created
    private String ifcName;
    private NamedElement[] members;
    
    /** Creates a new instance of ExtractInterfaceRefactoring 
     * @param sourceType Type the members of which should be extracted into an interface.
     */
    public ExtractInterfaceRefactoring(JavaClass sourceType) {
        // check if the sourceType is a ParameterizedType - if so, unwrap it
        if (sourceType instanceof ParameterizedType) {
            this.sourceType = ((ParameterizedType) sourceType).getDefinition();
        } else {
            this.sourceType = sourceType;
        }
    }
    
    /** Method called by the refactoring framework - gives the refactoring implementation
     * a chance to narrow the scope of the refactoring.
     */
    protected void setClassPath() {
        // leave the complete classpath
        Util.setDefaultClassPath();
    }

    /** Returns the type the members of which should be extracted into an interface
     * by this refactoring.
     * @return Source of the members to be extracted.
     */
    public JavaClass getSourceType() {
        return sourceType;
    }

    // --- PARAMETERS ----------------------------------------------------------
    
    /** Returns name of the interface to be created.
     * @return Name of the new interface or null if it is not set.
     */
    public String getIfcName() {
        return ifcName;
    }

    /** Sets the name of the interface to be created.
     * @param ifcName Name of the new interface.
     */
    public void setIfcName(String ifcName) {
        this.ifcName = ifcName;
    }

    /** Returns members to extract.
     * @return Array of members.
     */
    public NamedElement[] getMembers() {
        // never return null
        return members == null ? EMPTY_MEMBERS : members;
    }

    /** Sets members to extract.
     * @param members Array of members to extract.
     */
    public void setMembers(NamedElement[] members) {
        this.members = members;
    }
    
    // --- HELPER METHODS ------------------------------------------------------
    
    public boolean acceptFeature(Feature feature) {
        int modifiers = feature.getModifiers();
        if (Modifier.isPublic(modifiers)) {
            if (feature instanceof JavaClass) {
                if (Modifier.isStatic(modifiers)) {
                    return true;
                }
            } else if (feature instanceof Field) {
                if (Modifier.isStatic(modifiers) && Modifier.isFinal(modifiers) && ((Field) feature).getInitialValueText() != null) {
                    return true;
                }
            } else if (feature instanceof Method) {
                if (!Modifier.isStatic(modifiers)) {
                    return true;
                }
            }
        }
        return false;
    }
}
