/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.refactoring.api;

import java.lang.reflect.Modifier;
import org.netbeans.jmi.javamodel.Feature;
import org.netbeans.jmi.javamodel.Field;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.Method;
import org.netbeans.jmi.javamodel.MultipartId;
import org.netbeans.jmi.javamodel.NamedElement;
import org.netbeans.jmi.javamodel.ParameterizedType;
import org.netbeans.modules.refactoring.classpath.Util;

/** Extract Super Class Refactoring implementation class.
 *
 * @author Martin Matula
 */
public final class ExtractSuperClassRefactoring extends AbstractRefactoring {
    private static final MemberInfo[] EMPTY_MEMBERS = new MemberInfo[0];

    // source type
    private final JavaClass sourceType;
    // parameters of the refactoring
    private String scName;
    private MemberInfo[] members;
    
    /**
     * Creates a new instance of ExtractSuperClassRefactoring 
     * 
     * @param sourceType Type the members of which should be pulled up.
     */
    public ExtractSuperClassRefactoring(JavaClass sourceType) {
        // check if the sourceType is a ParameterizedType - if so, unwrap it
        if (sourceType instanceof ParameterizedType) {
            this.sourceType = ((ParameterizedType) sourceType).getDefinition();
        } else {
            this.sourceType = sourceType;
        }
    }
    
    /** Method called by the refactoring framework - gives the refactoring implementation
     * a chance to narrow the scope of the refactoring.
     */
    protected void setClassPath() {
        // leave the complete classpath
        Util.setDefaultClassPath();
    }

    /** Returns the type the members of which should be pulled up
     * by this refactoring.
     * @return Source of the members to be pulled up.
     */
    public JavaClass getSourceType() {
        return sourceType;
    }

    // --- PARAMETERS ----------------------------------------------------------
    
    /** Returns the name for the super class to be created.
     * @return Super class name.
     */
    public String getSuperClassName() {
        return scName;
    }

    /** Sets name for the super class to be created.
     * @param scName Super class name.
     */
    public void setSuperClassName(String scName) {
        this.scName = scName;
    }

    /** Returns descriptors of the members to extract into the new super class.
     * @return Member descriptors.
     */
    public MemberInfo[] getMembers() {
        // never return null
        return members == null ? EMPTY_MEMBERS : members;
    }

    /** Sets members (using their descriptors) to extract into the new super class.
     * @param members Descriptors of members to be extracted into the new super class.
     */
    public void setMembers(MemberInfo[] members) {
        this.members = members;
    }
    
    // --- HELPER METHODS ------------------------------------------------------
    
    public boolean acceptFeature(Feature feature) {
        return (((feature instanceof JavaClass) || (feature instanceof Field) || (feature instanceof Method)) && (!(Modifier.isPrivate(feature.getModifiers()))));
    }
    
    // --- HELPER CLASSES ------------------------------------------------------
    
    /** Class describing a member to be extracted into the new super class.
     */
    public static final class MemberInfo {
        public final NamedElement member;
        public final boolean makeAbstract;
        
        private MemberInfo(NamedElement member, boolean makeAbstract) {
            this.member = member;
            this.makeAbstract = makeAbstract;
        }

        /** Creates a new instance of MemberInfo describing a method.
         * @param method Method to be extracted.
         * @param makeAbstract Indicates whether the method should be made abstract
         *              in the supertype.
         */
        public MemberInfo(Method method, boolean makeAbstract) {
            this((NamedElement) method, makeAbstract);
        }
        
        /** Creates a new instance of MemberInfo describing an inner class 
         * to be extracted.
         * @param innerClass Inner class to be extracted.
         */
        public MemberInfo(JavaClass innerClass) {
            this(innerClass, false);
        }
        
        /** Creates a new instance of MemberInfo describing a field 
         * to be extracted.
         * @param field Field to be extracted.
         */
        public MemberInfo(Field field) {
            this(field, false);
        }
        
        /** Creates a new instance of MemberInfo describing an interface name
         * from the implements clause that should be extracted.
         * @param interfaceName Interface name to be extracted.
         */
        public MemberInfo(MultipartId interfaceName) {
            this(interfaceName, false);
        }
    }
}
