/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.refactoring.api;

import java.util.*;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.modules.java.JavaDataObject;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.refactoring.classpath.Util;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.loaders.DataFolder;
import org.openide.loaders.DataObject;
import org.openide.loaders.DataObjectNotFoundException;
import org.netbeans.api.queries.*;

/**
 * Move class/classes refactoring.
 * @author Jan Becicka
 */
public final class MoveClassRefactoring extends AbstractRefactoring {
    
    private Collection/*<Resource>*/        resources;
    private Collection                      otherDataObjects = Collections.EMPTY_LIST;
    private FileObject                      folder = null;
    private Map                             packagePostfix;
    private Collection dataObjects = Collections.EMPTY_LIST;
    private Map checkedSourcesMap;
    private boolean packageRename;
    private boolean isPackageRename = false;
    private String newPackageName;
    private FileObject target;
    
    /**
     * Creates a new instance of MoveClassRefactoring. All classes within given folder will be moved.
     * @param sourceFolder Folder where classes are held
     * @param isPackage true - given sourceFolder is considered as java package - only direct children will be moved.
     *                  false - given sourceFolder is considered as folder - move fill be performed recursively
     * @since 1.4.0
     */
    public MoveClassRefactoring(FileObject sourceFolder, boolean isPackage) {
        isPackageRename = true;
        this.packageRename = isPackage;
        if (packageRename) {
            this.folder = sourceFolder;
            DataObject[] children = DataFolder.findFolder(folder).getChildren();
            resources = new HashSet();
            otherDataObjects = new HashSet();
            for (int i = 0; i < children.length; i++) {
                if (children[i] instanceof JavaDataObject) {
                    resources.add(JavaModel.getResource(children[i].getPrimaryFile()));
                } else if (!(children[i] instanceof DataFolder)) {
                    otherDataObjects.add(children[i]);
                }
            }
        } else {
            this.folder = sourceFolder;
            LinkedList children = new LinkedList(Arrays.asList(DataFolder.findFolder(folder).getChildren()));
            resources = new HashSet();
            otherDataObjects = new HashSet();
            while(!children.isEmpty()) {
                DataObject dob = (DataObject) children.removeFirst();
                if (dob instanceof JavaDataObject) {
                    resources.add(JavaModel.getResource(dob.getPrimaryFile()));
                } else if (dob instanceof DataFolder) {
                    if (VisibilityQuery.getDefault().isVisible(dob.getPrimaryFile())) {
                        //CVS folders are ignored
                        children.addAll(Arrays.asList(((DataFolder)dob).getChildren()));
                    }
                } else {
                    otherDataObjects.add(dob);
                }
            }
        }
    }

    /**
     * Creates a new instance of MoveClassRefactoring.
     * @param javaClass a JavaClass which will be moved
     */
    public MoveClassRefactoring(JavaClass javaClass) {
        resources = new ArrayList(1);
        resources.add(javaClass.getResource());
    }
    
    /**
     * Creates a new instance of MoveClassRefactoring.
     * @param resources All Resources from this collection will be moved
     */
    public MoveClassRefactoring(Collection resources) {
        this.resources = resources;
    }
    
    /**
     * Creates a new instance of MoveClassRefactoring.
     * @param dataObjects DataObjects that are being moved
     * @param target target folder
     * @since 1.4.0
     */
    public MoveClassRefactoring(Collection dataObjects, FileObject target) {
        this.target = target;
        resources = new ArrayList(1);
        packagePostfix = new HashMap(dataObjects.size());
        otherDataObjects = new HashSet();
        this.dataObjects=dataObjects;
        setup(dataObjects, "");
    }
    
    /**
     * Getter for property sourceFolder
     * @return Value of sourceFolder or null if no surce folder was specified
     */
    public FileObject getSourceFolder() {
        return folder;
    }

    protected void setClassPath() {
        Util.setClassPath(resources);
    }

    /**
     * Getter for property resources
     * @return Collection<Resource>
     */
    public Collection getResources() {
        return resources;
    }
    
    /**
     * Getter for property targetPackageName
     * @return Value of property targetPackageName
     * @since 1.4.0
     */
    public String getTargetPackageName() {
        return newPackageName;
    }

    /**
     * Getter for targetPackageName for specified Resource
     * @return Value of property targetPackageName
     * @param r Resource for which target package name is requested
     */
    public String getTargetPackageName(Resource r) {
        return getTargetPackageName(JavaModel.getFileObject(r));
    }
    
    /**
     * Getter for targetPackageName for specified FileObject
     * @return Value of property targetPackageName
     * @param fo FileObject for which target package name is requested
     * @since 1.4.0
     */
    public String getTargetPackageName(FileObject fo) {
        if (isPackageRename) {
            if (packageRename)
                return newPackageName;
            else {
                //folder rename
                ClassPath cp = ClassPath.getClassPath(folder, ClassPath.SOURCE);
                FileObject root = cp.findOwnerRoot(folder);
                String prefix = FileUtil.getRelativePath(root, folder.getParent()).replace('/','.');
                String postfix = FileUtil.getRelativePath(folder, fo.getParent()).replace('/', '.');
                String t = concat(prefix, newPackageName, postfix);
                return t;
            }
        }

        if (packagePostfix != null) {
            String postfix = (String) packagePostfix.get(fo);
            String packageName = concat(null, newPackageName, postfix);
            return packageName;
        } else
            return newPackageName;
    }

    /**
     * Getter for property otherDataObjects
     * @return Value of property otherDataObjects
     */
    public Collection getOtherDataObjects() {
        return otherDataObjects;
    }
    
    /**
     * Getter for property targetClassPathRoot
     * @return returns target class path root
     * @since 1.4.0
     */
    public FileObject getTargetClassPathRoot() {
        FileObject f = isPackageRename ? folder : target;
        return ClassPath.getClassPath(f, ClassPath.SOURCE).findOwnerRoot(f);
    }
    
    /**
     * Setter for property targetClassPath root
     * @param target New value of property targetClassPath root
     * @since 1.4.0
     */
    public void setTargetClassPathRoot(FileObject target) {
        this.target = target;
    }
    
    /**
     * Setter for property newPackageName
     * @param name New value of property newPackageName
     * @since 1.4.0 
     */
    public void setTargetPackageName(String name) {
        this.newPackageName = name;
    }
    
    /**
     * Getter for property selectedDataObjects
     * @return Value of property selectedDataObjects
     * @since 1.4.0
     */
    public Collection getSelectedDataObjects() {
        if (isPackageRename) {
            try {
                if (!packageRename)
                    return Collections.singletonList(DataObject.find(folder));
                else {
                    FileObject f = folder;
                    FileObject cpRoot = getTargetClassPathRoot();
                    if (f!=cpRoot) {
                        while (f.getParent() != cpRoot) {
                            f = f.getParent();
                        }
                    }
                    return Collections.singletonList(DataObject.find(f));
                }
            } catch (DataObjectNotFoundException notFound) {
                
            }
        }
        return dataObjects;
    }
    
    private void setup(Collection dataObjects, String postfix) {
        for (Iterator i = dataObjects.iterator(); i.hasNext(); ) {
            DataObject o = (DataObject) i.next();
            FileObject fo = o.getPrimaryFile();
            if (o instanceof JavaDataObject) {
                Resource r = JavaModel.getResource(fo);
                resources.add(r);
                packagePostfix.put(fo, postfix.replace('/', '.'));
            } else if (!(o instanceof DataFolder)) {
                otherDataObjects.add(o);
                packagePostfix.put(fo, postfix.replace('/', '.'));
            } else if (VisibilityQuery.getDefault().isVisible(((DataFolder) o).getPrimaryFile())) {
                //o instanceof DataFolder
                //CVS folders are ignored
                boolean addDot = !"".equals(postfix);
                Collection col = new ArrayList();
                for (Enumeration en = ((DataFolder) o).children(); en.hasMoreElements(); col.add(en.nextElement()));
                
                setup(col, postfix +(addDot?".":"") +o.getName()); // NOI18N
            }
        }
    }
 
    private String concat(String s1, String s2, String s3) {
        String result = "";
        if (s1 != null && !"".equals(s1)) {
            result += s1 + "."; // NOI18N
        }
        result +=s2;
        if (s3 != null && !"".equals(s3)) {
            result += ("".equals(result)? "" : ".") + s3; // NOI18N
        }
        return result;
    }
}    
