/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.refactoring.classpath;
import java.lang.reflect.Modifier;
import java.net.URL;
import java.util.*;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.java.classpath.GlobalPathRegistry;
import org.netbeans.api.java.project.JavaProjectConstants;
import org.netbeans.api.java.queries.SourceForBinaryQuery;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ProjectUtils;
import org.netbeans.api.project.SourceGroup;
import org.netbeans.api.project.Sources;
import org.netbeans.api.project.ui.OpenProjects;
import org.netbeans.jmi.javamodel.ClassMember;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.JavaPackage;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.jmi.javamodel.Variable;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.modules.javacore.api.JavaModel;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;

/**
 *
 * @author  Jan Becicka
 */
public class Util {
    
    private static WeakHashMap superProjectsCache = null;
    
    private static WeakHashMap getSuperProjectsCache() {
        if (superProjectsCache==null)
            superProjectsCache = new WeakHashMap();
        return superProjectsCache;
    }
    
    public static Set getSuperprojects(Project project) {
        Set result = (Set) getSuperProjectsCache().get(project);
        if (result!=null) {
            //return cached result
            return result;
        }
        result = new HashSet();
        Project openedProjects[] = OpenProjects.getDefault().getOpenProjects();
        HashSet sourcePath = new HashSet();
        Sources sources = ProjectUtils.getSources(project);
        SourceGroup[] sourceGroup = sources.getSourceGroups(JavaProjectConstants.SOURCES_TYPE_JAVA);
        for(int sourceGroupsIndex=0; sourceGroupsIndex<sourceGroup.length; sourceGroupsIndex++) {
            //add each directory on ClassPath.SOURCE to sourcePath
            ClassPath cp = ClassPath.getClassPath(sourceGroup[sourceGroupsIndex].getRootFolder(), ClassPath.SOURCE);
            sourcePath.addAll(Arrays.asList(cp.getRoots()));
        }
        
        for (int i = 0; i<openedProjects.length; i++) {
            //for all opened projects
            Project p = openedProjects[i];
            Sources src = ProjectUtils.getSources(p);
            SourceGroup[] groups = src.getSourceGroups(JavaProjectConstants.SOURCES_TYPE_JAVA);
            for(int j=0; j<groups.length; j++) {
                //for all groups get COMPILE classpath
                ClassPath cp = ClassPath.getClassPath(groups[j].getRootFolder(), ClassPath.COMPILE);
                List entries = cp.entries();
                for (Iterator it = cp.entries().iterator(); it.hasNext();) {
                    //try to find source for each class path item
                    URL binURL = ((ClassPath.Entry) it.next()).getURL();
                    FileObject[] roots = SourceForBinaryQuery.findSourceRoots(binURL).getRoots();
                    for (int z = 0; z<roots.length; z++) {
                        if (sourcePath.contains(roots[z])) {
                            //if source of class path item is on sourcePath - this project depend on given project
                            result.add(p);
                        }
                    }
                }
            }
        }
        getSuperProjectsCache().put(project, result);
        return result;
    }
    
    static void resetCache() {
        superProjectsCache = null;
    }
    
    public static void setDefaultClassPath() {
        setClassPath(RefactoringClassPathImplementation.getDefault());
    }

    public static void setClassPath(Element el) {
        if (el == null) {
            ErrorManager.getDefault().log(ErrorManager.INFORMATIONAL,
                    "Refactoring: setClassPath(null) should not be called. Setting default classpath"); //NOI18N
            setDefaultClassPath();
            return ;
        }
        if (el != null && el.isValid()) {
            if (el instanceof JavaPackage) {
                setDefaultClassPath();
            } else {
                Resource resource = el.getResource();
                if (resource == null) {
                    ErrorManager.getDefault().log(ErrorManager.INFORMATIONAL,
                            "Refactoring: el.getResource() returned null. el = " + el); //NOI18N
                    setDefaultClassPath();
                    return ;
                }
                if (isLocal(el)) {
                    JavaModel.setClassPath(resource);
                    return;
                }                
                FileObject fo = JavaModel.getFileObject(resource);
                if (fo == null) {
                    ErrorManager.getDefault().log(ErrorManager.INFORMATIONAL, 
                            "Refactoring: JavaModel.getFileObject(res) returned null. res = " + resource.getName());//NOI18N
                    setDefaultClassPath();
                    return ;
                }
                
                setClassPath(RefactoringClassPathImplementation.getCustom(Collections.singleton(fo)));
            }
        }
    }
    
    public static void setClassPath(Collection elements) {
        Set s = new HashSet(elements.size());
        for (Iterator i = elements.iterator(); i.hasNext();) {
            Element el = (Element) i.next();
            
                Resource resource = el.getResource();
                if (resource == null) {
                    ErrorManager.getDefault().log(ErrorManager.INFORMATIONAL, 
                            "Refactoring: el.getResource() returned null. el = " + el);//NOI18N
                    setDefaultClassPath();
                    return ;
                }
                
                FileObject fo = JavaModel.getFileObject(resource);
                if (fo == null) {
                    ErrorManager.getDefault().log(ErrorManager.INFORMATIONAL, 
                            "Refactoring: JavaModel.getFileObject(res) returned null. res = " + resource.getName());//NOI18N
                    setDefaultClassPath();
                    return ;
                }
            
            s.add(fo);
        }
        setClassPath(RefactoringClassPathImplementation.getCustom(s));
    }
    
    private static void setClassPath(ClassPath cp) {
        JavaModel.setClassPath(cp);
        JMManager.getTransactionMutex().setSearchScope(Collections.singletonList(cp));
    }
    
    public static boolean isUnderSourceRootOfOpenProjects(FileObject fo) {
        Project p = FileOwnerQuery.getOwner(fo);
        if (p==null) return false;
        Project[] opened = OpenProjects.getDefault().getOpenProjects();
        for (int i = 0; i<opened.length; i++) {
            if (p==opened[i]) {
                SourceGroup[] gr = ProjectUtils.getSources(p).getSourceGroups(JavaProjectConstants.SOURCES_TYPE_JAVA);
                for (int j = 0; j < gr.length; j++) {
                    if (fo==gr[j].getRootFolder()) return true;
                    if (FileUtil.isParentOf(gr[j].getRootFolder(), fo))
                        return true;
                }
                return false;
            }
        }
        return false;
    }
    
    public static boolean isClassPathRoot(FileObject fo) {
        return fo.equals(((JMManager) JMManager.getManager()).getMergedClassPath().findOwnerRoot(fo));
    }

    private static boolean isLocal(Element el) {
        if (el instanceof ClassMember) {
            return Modifier.isPrivate(((ClassMember)el).getModifiers());
        } else if (el instanceof Variable)
            return true;
        return false;
    }
}
