/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.refactoring.plugins;

import java.lang.reflect.Modifier;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.jmi.javamodel.ClassDefinition;
import org.netbeans.modules.javacore.internalapi.ProgressListener;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.refactoring.CheckUtils;
import org.netbeans.modules.refactoring.api.*;
import org.netbeans.modules.refactoring.api.AnonymousToInnerRefactoring;
import org.netbeans.modules.refactoring.api.AnonymousToInnerRefactoring.ParameterInfo;
import org.netbeans.modules.refactoring.ui.AnonymousToInnerAction;
import org.openide.filesystems.FileObject;
import org.openide.text.PositionBounds;
import org.openide.util.NbBundle;
import org.netbeans.modules.refactoring.spi.RefactoringElementsBag;
import org.netbeans.modules.refactoring.spi.SimpleRefactoringElementImpl;

/**
 *
 * @author Daniel Prusa
 */
public class AnonymousToInnerRefactoringPlugin extends JavaRefactoringPlugin implements ProgressListener {
    private ClassDefinition classDefinition;
    private AnonymousToInnerRefactoring refactoring;
    private List outerLocalVars;
    private List constructorParameters;
    private boolean outerClassAccessed;
    
    public AnonymousToInnerRefactoringPlugin(AnonymousToInnerRefactoring convert) {
        this.refactoring = convert;
        classDefinition = convert.getClassDefinition();
    }
    
    public Problem checkParameters() {
        return setParameters(refactoring.getName(), refactoring.getModifiers(), refactoring.getParamTable());
    }
    
    public Problem fastCheckParameters() {
        String name = refactoring.getName();
        if (name == null || name.length() == 0) {
            return new Problem(true, NbBundle.getMessage(AnonymousToInnerRefactoring.class, "ERR_NameNotSet"));
        }
        if (!org.openide.util.Utilities.isJavaIdentifier(name)) {
            String msg = new MessageFormat(NbBundle.getMessage(AnonymousToInnerRefactoring.class, "ERR_InvalidIdentifier")).format(new Object[] {name}); //NOI18N
            return new Problem(true, msg);
        }
        return null;
    }
    
    public Problem preCheck() {
        fireProgressListenerStart(refactoring.PRE_CHECK, 4);
        try {
            if (classDefinition == null) {
                String msg = getString("ERR_AnonymousToInnerNoClassSelected"); //NOI18N
                return new Problem(true, msg);
            }
            if (!CheckUtils.isElementInOpenProject(classDefinition)) {
                return new Problem(true, NbBundle.getMessage(JavaRefactoringPlugin.class, "ERR_ProjectNotOpened"));
            }
            
            fireProgressListenerStep();
            
            collectLocalVariables();
            fireProgressListenerStep();
            fireProgressListenerStep();
            
            ParameterInfo[] pinfo = computeConstructorParams();
            fireProgressListenerStep();
            
            if (refactoring.getParamTable() == null) {
                refactoring.setParamTable(pinfo);
            }
            
        } finally {
            fireProgressListenerStop();
        }
        return null;
    }
    
    private Problem checkParameters(String name, int modifiers, ParameterInfo[] pinfo) {
        Problem problem = null;
        JavaClass jc = getEnclosingJavaClass(classDefinition);
        if (jc.getInnerClass(name, false) != null) {
            String msg = new MessageFormat(getString("ERR_InnerClassExists")).format(new Object[] {name, jc.getSimpleName()}); //NOI18N
            problem = new Problem(true, msg);
        }

        if ((modifiers & Modifier.STATIC) != 0 && outerClassAccessed) {
            String msg = getString("ERR_AnonymousToInnerCannotBeStatic"); //NOI18N
            Problem p = new Problem(true, msg);
            p.setNext(problem);
            problem = p;
        }
        
        for (int x = 0; x < pinfo.length; x++) {
            String pname = pinfo[x].getName();
            if (!org.openide.util.Utilities.isJavaIdentifier(pname)) {
                String msg = new MessageFormat(getString("ERR_AnonymousToInnerInavlidParamName")).format(new Object[] {pname}); //NOI18N
                Problem p = new Problem(true, msg);
                p.setNext(problem);
                problem = p;
            }
        }
        
        Set names = new HashSet(pinfo.length);
        for (int x = 0; x < pinfo.length; x++) {
            String pname = pinfo[x].getName();
            if (names.contains(pname)) {
                String msg = new MessageFormat(getString("ERR_AnonymousToInnerDuplicateParamName")).format(new Object[] {pname}); //NOI18N
                Problem p = new Problem(true, msg);
                p.setNext(problem);
                problem = p;
                break;
            } else {
                names.add(pname);
            }
        }
        return problem;
    }
    
    private Problem setParameters(String name, int modifiers, ParameterInfo[] pinfo) {
        fireProgressListenerStart(refactoring.PARAMETERS_CHECK, 1);
        try {
            fireProgressListenerStep();
            Problem result = checkParameters(name, modifiers, pinfo);
            return result;
        } finally {
            fireProgressListenerStop();
        }
    }    
    
    public Problem prepare(RefactoringElementsBag elements) {
        elements.add(refactoring, new ConvertToInnerClassElement(classDefinition, refactoring.getName(), refactoring.getModifiers(), refactoring.getParamTable()));
        return null;
    }
    
    // helper methods ..........................................................
    
    private static final String getString(String key) {
        return NbBundle.getMessage(AnonymousToInnerRefactoringPlugin.class, key);
    }
    
    public void start(org.netbeans.modules.javacore.internalapi.ProgressEvent event) {
        fireProgressListenerStart(event.getOperationType(), event.getCount());
    }
    
    public void step(org.netbeans.modules.javacore.internalapi.ProgressEvent event) {
        fireProgressListenerStep();
    }
    
    public void stop(org.netbeans.modules.javacore.internalapi.ProgressEvent event) {
        fireProgressListenerStop();
    }
    
    private void collectLocalVariables() {
        outerLocalVars = new ArrayList();
        Set accessedClasses = new HashSet();
        collectLocalVariables(outerLocalVars, classDefinition, new HashSet(), accessedClasses);
        Set supertypes = new HashSet();
        collectSupertypes(supertypes, getEnclosingJavaClass(classDefinition));
        outerClassAccessed = false;
        for (Iterator iter = accessedClasses.iterator(); iter.hasNext(); ) {
            if (supertypes.contains(iter.next())) {
                outerClassAccessed = true;
                break;
            }
        }
    }
    
    private void collectLocalVariables(List localVars, Element elem, Set localVarNames, Set accessedClasses) {
        if (elem instanceof VariableAccess) {
            NamedElement nelem = ((VariableAccess) elem).getElement();
            if (nelem instanceof LocalVariable || nelem instanceof Parameter) {
                Variable variable = (Variable) nelem;
                ClassDefinition cd = getEnclosingClass(variable);
                if (cd != classDefinition && !localVarNames.contains(variable.getName())) {
                    localVars.add(variable);
                    localVarNames.add(variable.getName());
                }
            } else if ((nelem instanceof Field) && ((((Field) nelem).getModifiers() & Modifier.STATIC) == 0)) {
                accessedClasses.add(((Field)nelem).getDeclaringClass());
            }
        } else {
            if (elem instanceof Invocation) {
                Feature bf = (Feature) ((Invocation) elem).getElement();
                if (bf != null && (bf.getModifiers() & Modifier.STATIC) == 0) {
                    accessedClasses.add(bf.getDeclaringClass());
                }
            }
            for (Iterator iter = elem.getChildren().iterator(); iter.hasNext(); ) {
                collectLocalVariables(localVars, (Element)iter.next(), localVarNames, accessedClasses);
            }
        }
    }

    private ParameterInfo[] computeConstructorParams() {
        constructorParameters = new ArrayList();
        Set paramNames = new HashSet();
        JavaClass superType = getSupertype(classDefinition);
        if (!superType.isInterface()) {
            NewClassExpression newClsExpr = (NewClassExpression) classDefinition.refImmediateComposite();
            Constructor constr = (Constructor) newClsExpr.getElement();
            if (constr != null) {
                for (Iterator iter = constr.getParameters().iterator(); iter.hasNext(); ) {
                    constructorParameters.add(iter.next());
                }
            }
        }
        
        ParameterInfo[] pinfo = new ParameterInfo[outerLocalVars.size() + constructorParameters.size()];
        int index = 0;
        for (Iterator iter = outerLocalVars.iterator(); iter.hasNext(); ) {
            Variable localVar = (Variable) iter.next();
            paramNames.add(localVar.getName());
            pinfo[index++] = new ParameterInfo(localVar);
        }
        for (Iterator iter = constructorParameters.iterator(); iter.hasNext(); ) {
            Parameter param = (Parameter) iter.next();
            String pname = param.getName();
            String name = pname;
            int x = 1;
            while (paramNames.contains(name)) {
                name = pname + x;
                x++;
            }
            pinfo[index] = new ParameterInfo(param);
            pinfo[index].setName(name);
            index++;
        }
        return pinfo;
    }
    
    private ClassDefinition getEnclosingClass(Element elem) {
        Element comp = (Element) elem.refImmediateComposite();
        while (comp != null && !(comp instanceof ClassDefinition)) {
            comp = (Element) comp.refImmediateComposite();
        }
        return (ClassDefinition) comp;
    }
    
    private JavaClass getEnclosingJavaClass(Element elem) {
        Element comp = (Element) elem.refImmediateComposite();
        while (comp != null && !(comp instanceof JavaClass)) {
            comp = (Element) comp.refImmediateComposite();
        }
        return (JavaClass) comp;
    }
    
    private JavaClass getSupertype(ClassDefinition cd) {
        List list = cd.getInterfaces();
        if (list.size() > 0) {
            return (JavaClass) list.get(0);
        } else {
            return cd.getSuperClass();
        }
    }
    
    private void collectSupertypes(Set supertypes, ClassDefinition jc) {
        if (jc == null)
            return;
        supertypes.add(jc);
        collectSupertypes(supertypes, jc.getSuperClass());
        for (Iterator iter = jc.getInterfaces().iterator(); iter.hasNext();) {
            collectSupertypes(supertypes, (ClassDefinition)iter.next());
        }
    }
    
    // ConvertToInnerClassElement ...............................................
    private class ConvertToInnerClassElement extends SimpleRefactoringElementImpl {
        
        private final String text;
        private PositionBounds bounds = null;
        
        private ClassDefinition classDef;
        private String name;
        private int modifiers;
        private ParameterInfo[] pinfo;
        
        public ConvertToInnerClassElement(ClassDefinition classDef, String name, int modifiers, ParameterInfo[] pinfo) {
            this.classDef = classDef;
            this.name = name;
            this.modifiers = modifiers;
            this.pinfo = pinfo;
            text = MessageFormat.format(getString("TXT_ConvertAnonymousToInner"), new Object[] {name});
        }
            
        public String getDisplayText() {
            return text;
        }
        
        public Element getJavaElement() {
            return classDef;
        }
        
        public PositionBounds getPosition() {
            if (bounds == null) {
                bounds = JavaMetamodel.getManager().getElementPosition(classDef);
            }
            return bounds;
        }
        
        public String getText() {
            return getDisplayText();
        }
        
        public void performChange() {
            JavaModelPackage modelPackage = (JavaModelPackage) classDef.refOutermostPackage();
            
            JavaClassClass jcProxy = modelPackage.getJavaClass();
            MultipartIdClass idProxy = modelPackage.getMultipartId();
            FieldClass fieldProxy = modelPackage.getField();
            ConstructorClass constructorProxy = modelPackage.getConstructor();
            ParameterClass parameterProxy = modelPackage.getParameter();
            AssignmentClass assignmentProxy = modelPackage.getAssignment();
            VariableAccessClass varAccessProxy = modelPackage.getVariableAccess();
            ThisExpressionClass thisProxy = modelPackage.getThisExpression();
            ExpressionStatementClass exprStatementProxy = modelPackage.getExpressionStatement();
            StatementBlockClass blockProxy = modelPackage.getStatementBlock();
            ConstructorInvocationClass constrInvocationProxy = modelPackage.getConstructorInvocation();
            NewClassExpressionClass newClassProxy = modelPackage.getNewClassExpression();
            
            NewClassExpression newClsExpr = (NewClassExpression) classDef.refImmediateComposite();
            JavaClass superType = getSupertype(classDef);
            
            MultipartId superClassName = null;
            List interfaceNames = null;
            if (superType != null) {
                MultipartId exprSuperTypeName = newClsExpr.getClassName();
                MultipartId superTypeName = (MultipartId) exprSuperTypeName.duplicate();
                if (!superType.isInterface()) {
                    superClassName = superTypeName;
                } else {
                    interfaceNames = new ArrayList();
                    interfaceNames.add(superTypeName);
                }
            }
            JavaClass innerClass = jcProxy.createJavaClass(
                name, null, modifiers, null, null, null, superClassName, interfaceNames, null
            );
            List contents = innerClass.getContents();
            for (Iterator iter = classDef.getContents().iterator(); iter.hasNext(); ) {
                Element elem = (Element) iter.next();
                contents.add(elem.duplicate());
            }
            List statements = new ArrayList();
            int featureIndex = 0;
            
            Map paramToName = new HashMap();
            for (int x = 0; x < pinfo.length; x++) {
                paramToName.put(pinfo[x].getVariable(), pinfo[x].getName());
            }
            
            // if local final variables from the outer class are accessed, correspondent fields and constructor must be added
            if (outerLocalVars.size() > 0) {
                for (Iterator iter = outerLocalVars.iterator(); iter.hasNext(); ) {
                    Variable localVar = (Variable) iter.next();
                    String varName = localVar.getName();
                    Field field = fieldProxy.createField(
                        varName, null, Modifier.PRIVATE, null, null, true, 
                        (TypeReference) localVar.getTypeName().duplicate(), localVar.getDimCount(), null, null
                    );
                    contents.add(featureIndex, field);
                    featureIndex++;
                    ThisExpression thisExpr = thisProxy.createThisExpression(null);
                    VariableAccess leftSide = varAccessProxy.createVariableAccess(varName, thisExpr, false);
                    VariableAccess rightSide = varAccessProxy.createVariableAccess((String) paramToName.get(localVar), null, false);
                    Assignment assignment = assignmentProxy.createAssignment(leftSide, OperatorEnum.ASSIGN, rightSide);
                    ExpressionStatement exprSt = exprStatementProxy.createExpressionStatement(assignment);
                    statements.add(exprSt);
                }
            }

            if (pinfo.length > 0) {
                List paramsToDelegate = new ArrayList();
                int index = 0;
                for (Iterator iter = constructorParameters.iterator(); iter.hasNext(); ) {
                    Parameter par = (Parameter) iter.next();
                    String pname = (String) paramToName.get(par);
                    paramsToDelegate.add(varAccessProxy.createVariableAccess(pname, null, false));
                }
                ConstructorInvocation cinv = constrInvocationProxy.createConstructorInvocation(
                    null, paramsToDelegate, true, null
                );
                statements.add(0, cinv);

                StatementBlock stBlock = blockProxy.createStatementBlock(statements);
                List constrParams = new ArrayList();
                for (int x = 0; x < pinfo.length; x++) {
                    String pName = pinfo[x].getName();
                    Variable var = pinfo[x].getVariable();
                    Parameter param = parameterProxy.createParameter(
                        pName, null, false, (TypeReference) var.getTypeName().duplicate(),
                        var.getDimCount(), false
                    );
                    constrParams.add(param);
                }
                Constructor constr = constructorProxy.createConstructor(
                    name, null, Modifier.PUBLIC, null, null, stBlock, null, null, constrParams, null
                );
                contents.add(featureIndex, constr);
            }
            
            Element jc = classDef;
            while (jc != null && !(jc instanceof JavaClass)) {
                jc = (Element) jc.refImmediateComposite();
            }
            if (jc != null) {
                ((JavaClass) jc).getContents().add(innerClass);
            }
            
            Map paramToValue = new HashMap(pinfo.length);
            Iterator parIter = constructorParameters.iterator();
            for (Iterator iter = newClsExpr.getParameters().iterator(); iter.hasNext(); ) {
                Expression expr = (Expression)iter.next();
                paramToValue.put(parIter.next(), expr.duplicate());
            }
            for (Iterator iter = outerLocalVars.iterator(); iter.hasNext(); ) {
                Variable locVar = (Variable)iter.next();
                VariableAccess vacc = varAccessProxy.createVariableAccess(locVar.getName(), null, false);
                paramToValue.put(locVar, vacc);
            }
            
            List values = new ArrayList(pinfo.length);
            for (int x = 0; x < pinfo.length; x++) {
                values.add(paramToValue.get(pinfo[x].getVariable()));
            }
            NewClassExpression newClassEpr2 = newClassProxy.createNewClassExpression(
                name, values, null, null, null
            );
            Element composite = (Element) newClsExpr.refImmediateComposite();
            composite.replaceChild(newClsExpr, newClassEpr2);
        }
                
        public FileObject getParentFile() {
            return JavaMetamodel.getManager().getFileObject(classDef.getResource());
        }
        
    } // ConvertToInnerClassElement
    
}