/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.refactoring.plugins;

import java.lang.reflect.Modifier;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.netbeans.modules.refactoring.CheckUtils;
import org.netbeans.modules.refactoring.CreateMethodElement;
import org.netbeans.modules.refactoring.api.EncapsulateFieldRefactoring;
import org.netbeans.modules.refactoring.spi.RefactoringElementsBag;
import org.netbeans.modules.refactoring.spi.RefactoringElementImplementation;
import org.openide.util.NbBundle;
import javax.jmi.reflect.RefObject;
import java.util.Collections;
import java.util.List;
import java.util.ListIterator;
import org.netbeans.jmi.javamodel.Field;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.refactoring.ChangeModElement;
import org.netbeans.modules.refactoring.EncapsulateFieldElement;
import org.netbeans.modules.refactoring.api.AbstractRefactoring;
import org.netbeans.modules.refactoring.api.Problem;
import org.openide.util.Utilities;

/**
 *
 * @author  Tomas Hurka, Jan Becicka
 */
public class EncapsulateFieldRefactoringPlugin extends JavaRefactoringPlugin {
    private final RefObject jmiObject;
    private Field encapsulatedField;
    private String getterName,setterName;
    private int methodModifiers,fieldModifiers;
    private Method currentGetter,currentSetter;
    private static int accessModifiers = Modifier.PRIVATE | Modifier.PROTECTED | Modifier.PUBLIC;
    private boolean alwaysUseAccessors;
    private EncapsulateFieldRefactoring refactoring;
    public static final String CLASS_FIELD_PREFIX="_"; // NOI18N
    
    /** Creates a new instance of RenameRefactoring */
    public EncapsulateFieldRefactoringPlugin(EncapsulateFieldRefactoring refactoring) {
        this.refactoring = refactoring;
        jmiObject = (RefObject) refactoring.getRefactoredObject();
    }
    
    public Problem preCheck() {
        fireProgressListenerStart(AbstractRefactoring.PRE_CHECK, 2);
        try {
            Problem result = isElementAvail((Element) jmiObject);
            if (result != null) {
                return result;
            }

            fireProgressListenerStep();
            if ((jmiObject instanceof Field) && !(jmiObject instanceof EnumConstant)) {
               encapsulatedField=(Field)jmiObject;
            } else {
                return createProblem(result, true, NbBundle.getMessage(EncapsulateFieldRefactoring.class, "ERR_EncapsulateWrongType"));
            }
            if (!CheckUtils.isElementInOpenProject((Element) jmiObject)) {
                return new Problem(true, NbBundle.getMessage(JavaRefactoringPlugin.class, "ERR_ProjectNotOpened"));
            }
            return result;
        } finally {
            fireProgressListenerStop();
        }
    }
    
    public Problem fastCheckParameters() {
        return checkParameters(refactoring.getGetterName(), refactoring.getSetterName(), refactoring.getMethodModifiers(), refactoring.getFieldModifiers(), refactoring.isAlwaysUseAccessors());
    }
    
    public Problem checkParameters() {
        return setParameters(refactoring.getGetterName(), refactoring.getSetterName(), refactoring.getMethodModifiers(), refactoring.getFieldModifiers(), refactoring.isAlwaysUseAccessors());
    }
    
    private Problem checkParameters(String getter,String setter,int accessModifier,int fieldModifier, boolean alwaysUseAccessors) {
        if ((getter!=null && !Utilities.isJavaIdentifier(getter)) ||
            (setter!=null && !Utilities.isJavaIdentifier(setter)) || (getter == null && setter == null)) {
            // user doesn't use valid java identifier, it cannot be used
            // as getter/setter name
            return new Problem(true, NbBundle.getMessage(EncapsulateFieldRefactoring.class, "ERR_EncapsulateMethods"));
        } else {
            // we have no problem :-)
            return null;
        }
    }
    
    private Problem setParameters(String getter,String setter,int accessModifier,int fieldModifier, boolean alwaysUseAccessors) {
        Problem result = null;
        fireProgressListenerStart(AbstractRefactoring.PARAMETERS_CHECK, 8);
        
        try {
            setterName=setter;
            getterName=getter;
            this.methodModifiers=accessModifier;
            this.fieldModifiers=fieldModifier;
            this.alwaysUseAccessors = alwaysUseAccessors;
        } finally {
            fireProgressListenerStop();
        }
        return null;
    }
    
    private TypeReference getTypeReference() {
        TypeReference ref=encapsulatedField.getTypeName();
        int dims=encapsulatedField.getDimCount();
        
        ref=(TypeReference)ref.duplicate();
        if (dims>0) {            
            if (ref instanceof ArrayReference) {
                ArrayReference arrRef=(ArrayReference)ref;
                
                arrRef.setDimCount(arrRef.getDimCount()+dims);
            } else {
                ArrayReferenceClass arrClass=((JavaModelPackage)encapsulatedField.refImmediatePackage()).getArrayReference();
                
                ref=arrClass.createArrayReference(null,(MultipartId)ref,dims);
            }
            
        }
        return ref;
    }
    
    private void createGetterAndSetter(RefactoringElementsBag elms) {
        String fieldName=encapsulatedField.getName();
        int staticMod=encapsulatedField.getModifiers()&Modifier.STATIC;
        String parName=staticMod!=0?"a"+getCapitalizedName(encapsulatedField):stripPrefix(fieldName); //NOI18N
        String getterBody="return ".concat(fieldName).concat(";"); //NOI18N
        String setterBody=(staticMod==0?"this.":"").concat(fieldName).concat(" = ").concat(parName).concat(";"); //NOI18N
        Type fieldType=encapsulatedField.getType();
        ClassDefinition jcls=encapsulatedField.getDeclaringClass();
        JavaModelPackage modelExtent=(JavaModelPackage)jcls.refImmediatePackage();
        TypeReference fieldTypeName=getTypeReference();
        Parameter par=modelExtent.getParameter().createParameter(parName, null, false, null, 0, false);
        int newmods=methodModifiers | staticMod;        
        List features = jcls.getContents();
        ListIterator iter = features.listIterator(features.size());
        ClassMember insertPoint = null;
        JavaMetamodel manager = JavaMetamodel.getManager();
        
        while (iter.hasPrevious()) {
            ClassMember cm = (ClassMember)iter.previous();
            if (!manager.isElementGuarded(cm)) {
                break;
            } else {
                insertPoint = cm;
            }
        }
        par.setTypeName(fieldTypeName);        
        currentGetter=getterName != null ? jcls.getMethod(getterName,Collections.EMPTY_LIST,false) : null;
        if (currentGetter==null && getterName != null) {
            RefactoringElementImplementation getter=new CreateMethodElement(newmods, getterName, fieldTypeName,
                                            Collections.EMPTY_LIST, getterBody, insertPoint, jcls, true);
            elms.add(refactoring, getter);
        }
        currentSetter=setterName != null ? jcls.getMethod(setterName,Collections.singletonList(fieldType),false): null;
        if (currentSetter==null && setterName != null) {
            boolean enabled = !Modifier.isFinal(encapsulatedField.getModifiers());
            RefactoringElementImplementation setter=new CreateMethodElement(newmods, setterName, null,
                                        Collections.singletonList(par), setterBody, insertPoint, jcls, enabled);
            elms.add(refactoring, setter);
        }
    }
    
    /**
     * Removes the class field prefix from  the identifer of a field.
     * For example, if the class field prefix is "_", the identifier "_name" 
     * is stripped to become "name".
     * @param identifierString The identifer to strip.
     * @return The stripped identifier.
     */
    private static String stripPrefix(String identifierString){
        String stripped;
        if(identifierString.startsWith(CLASS_FIELD_PREFIX)){
            stripped = identifierString.substring(CLASS_FIELD_PREFIX.length());
        }
        else{
             stripped = identifierString;
        }
        return stripped;
    }
    
    private static StringBuffer getCapitalizedName(Field field) {        
        StringBuffer name=new StringBuffer(stripPrefix(field.getName()));
        
        name.setCharAt(0,Character.toUpperCase(name.charAt(0)));
        return name;
    }
    
    
    public static String computeSetterName(Field field) {
        if (Modifier.isFinal(field.getModifiers())) {
            return null;
        }

        StringBuffer name=getCapitalizedName(field);
        
        name.insert(0,"set"); //NOI18N
        return name.toString();
    }
    
    public static String computeGetterName(Field field) {
        StringBuffer name=getCapitalizedName(field);
        Type retVal=field.getType();
        
        if (retVal instanceof PrimitiveType && ((PrimitiveType)retVal).getKind().equals(PrimitiveTypeKindEnum.BOOLEAN))
            name.insert(0,"is"); //NOI18N
        else 
            name.insert(0,"get"); //NOI18N
        return name.toString();
    }
    
    private boolean isOutsideGetterSetter(Element el) {
        Feature f=JavaModelUtil.getDeclaringFeature(el);
        if (f.equals(currentGetter) || f.equals(currentSetter))
            return false;
        return true;
    }
    
    public Problem prepare(RefactoringElementsBag elements) {
        
        fireProgressListenerStart(AbstractRefactoring.PREPARE, 9);
        try {
            fireProgressListenerStep();
            
            referencesIterator = encapsulatedField.getReferences().iterator();
            // if the field modifier is changed, create the element which
            // changes field's modifier and informs user by item in the tree
            int mods=encapsulatedField.getModifiers();
            int targetMods = mods;
            
            if ((mods&accessModifiers) != fieldModifiers) {
                int newMods=(mods & ~accessModifiers)|fieldModifiers;
                targetMods = newMods;
                elements.add(refactoring, new ChangeModElement(encapsulatedField,newMods));
            }
            createGetterAndSetter(elements);            
            while (referencesIterator.hasNext()) {
                if (cancelRequest) {
                    return null;
                }
                Element usage=(Element)referencesIterator.next();
                
                if (isOutsideGetterSetter(usage)) {
                    boolean enabled = true;
                    VariableAccess varAcc=(VariableAccess)usage;
                    boolean writeAccess = varAcc.isWrite();
                    
                    if (!alwaysUseAccessors) {
                        enabled = !isAccessible(varAcc, targetMods);
                    }

                    if ((writeAccess && setterName!=null) || (!writeAccess && getterName != null))
                        elements.add(refactoring, new EncapsulateFieldElement(encapsulatedField, (VariableAccess)usage, getterName,setterName, enabled, writeAccess));
                }
            }
            return null;
        } finally {
            referencesIterator = null;
            fireProgressListenerStop();
        }
    }
    
    private static final boolean isAccessible (VariableAccess usage, int modifiers) {
        if (Modifier.isPublic(modifiers))
            return true;
        ClassDefinition declaring = ((Field) usage.getElement()).getDeclaringClass();
        if (Modifier.isPrivate(modifiers)) {
            Element element = getClassDefinition(usage);
            return element.equals(declaring);
        }
        
        if (Modifier.isProtected(modifiers)) {
            // (jb) TODO: should be done more precisely
            // visiblity of protected fields is not that simple as 
            // this algorithm
            if (isTheSamePackage(usage))
                return true;
            ClassDefinition cd = getClassDefinition(usage);
            do {
                if (cd.isSubTypeOf(declaring))
                    return true;
                cd = getClassDefinition(cd);
            } while (cd != null);
            return false;
        } else {
            //<default> access
            return isTheSamePackage(usage);
        }
    }
    
    private static final boolean isTheSamePackage(VariableAccess usage) {
        return usage.getResource().getPackageName().equals(usage.getElement().getResource().getPackageName());
    }
    
    private static final ClassDefinition getClassDefinition(Element usage) {
        Element element = (Element) usage.refImmediateComposite();
        if (element instanceof Resource || element instanceof JavaPackage)
            return null;
        while (!(element instanceof ClassDefinition)) {
            element = (Element) element.refImmediateComposite();
        }
        return (ClassDefinition) element;
    }
    
    public Field getEncapsulatedField() {
        return encapsulatedField;
    }
}
