/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.refactoring.plugins;

import java.text.MessageFormat;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.jmi.javamodel.ClassDefinition;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.EnumConstant;
import org.netbeans.jmi.javamodel.Feature;
import org.netbeans.jmi.javamodel.Field;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.jmi.javamodel.UnresolvedClass;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.refactoring.CheckUtils;
import org.netbeans.modules.refactoring.api.AbstractRefactoring;
import org.netbeans.modules.refactoring.api.EncapsulateFieldRefactoring;
import org.netbeans.modules.refactoring.api.EncapsulateFieldsRefactoring;
import org.netbeans.modules.refactoring.api.Problem;
import org.netbeans.modules.refactoring.spi.RefactoringElementsBag;
import org.openide.filesystems.FileObject;
import org.openide.util.NbBundle;
import org.netbeans.modules.refactoring.api.ProgressListener;
import org.netbeans.modules.refactoring.api.ProgressEvent;
import org.netbeans.modules.refactoring.api.RefactoringSession;
import org.netbeans.modules.refactoring.api.RenameRefactoring;

/** Encapsulate fields refactoring. This is a composed refactoring (uses instances of {@link org.netbeans.modules.refactoring.api.EncapsulateFieldRefactoring}
 * to encapsulate several fields at once.
 *
 * @author  Pavel Flaska, Jan Becicka
 */
public class EncapsulateFieldsPlugin extends JavaRefactoringPlugin {
    
    // represents selected fields in explorer
    private Element[] selectedObjects;
    private EncapsulateFieldRefactoring[] refactorings;
    private EncapsulateFieldsRefactoring refactoring;
    
    private ProgressListener listener = new ProgressListener() {
        public void start(ProgressEvent event) {
            fireProgressListenerStart(event.getOperationType(),event.getCount());
        }

        public void step(ProgressEvent event) {
            fireProgressListenerStep();
        }

       public void stop(ProgressEvent event) {
            fireProgressListenerStop();
        }
    };

    /** Creates a new instance of EcapsulateFields.
     * @param selectedObjects Array of objects (fields) that should be encapsulated.
     */
    public EncapsulateFieldsPlugin(EncapsulateFieldsRefactoring refactoring) {
        this.refactoring = refactoring;
        selectedObjects = refactoring.getSelectedObjects();
    }

    public Problem checkParameters() {
        return setParameters(false, refactoring.getRefactorFields(), refactoring.getMethodModifier(), refactoring.getFieldModifier(), refactoring.isAlwaysUseAccessors());
    }
    
    public Problem fastCheckParameters() {
        return setParameters(true, refactoring.getRefactorFields(), refactoring.getMethodModifier(), refactoring.getFieldModifier(), refactoring.isAlwaysUseAccessors());
    }

    
    public Problem preCheck() {
        fireProgressListenerStart(AbstractRefactoring.PRE_CHECK, 2);
        try {
            Problem result = null;

            fireProgressListenerStep();
            for (int i = 0; i < selectedObjects.length; i++) {
                Element jmiObject = selectedObjects[i];
                Problem p = isElementAvail(jmiObject);
                if (p != null) {
                    return p;
                }
                if (jmiObject instanceof UnresolvedClass) {
                    return new Problem(true, NbBundle.getMessage(JavaRefactoringPlugin.class, "DSC_ElNotAvail")); // NOI18N
                }
                
                if (!(jmiObject instanceof Field) || (jmiObject instanceof EnumConstant)) {
                    if (selectedObjects.length == 1 && selectedObjects[0] instanceof JavaClass) {
                        Element[] fields = refactoring.getFields();
                        if (fields.length == 0) { // there are no fields in class
                            String pattern = NbBundle.getMessage(EncapsulateFieldsRefactoring.class, "ERR_EncapsulateNoFields");
                            String text = new MessageFormat(pattern).format(new Object[] { ((JavaClass) selectedObjects[0]).getSimpleName() });
                            result = createProblem(result, true, text);
                        } else {
                            selectedObjects = fields;
                        }
                    } else {
                        result = createProblem(result, true, NbBundle.getMessage(EncapsulateFieldsRefactoring.class, "ERR_EncapsulateWrongType"));
                    }
                } else {
                    if (jmiObject instanceof Feature) {
                        Feature f = (Feature) jmiObject;
                        ClassDefinition cd = f.getDeclaringClass();
                        if (cd != null && cd instanceof JavaClass && ((JavaClass) cd).isInterface()) {
                            result = createProblem(result, true, NbBundle.getMessage(EncapsulateFieldsRefactoring.class, "ERR_EncapsulateInIntf"));
                        }
                    }
                }
                if (result != null && result.isFatal())
                    return result;
                if (!CheckUtils.isElementInOpenProject(jmiObject)) {
                    return new Problem(true, NbBundle.getMessage(JavaRefactoringPlugin.class, "ERR_ProjectNotOpened"));
                }
            }
// canWrite() check disabled. It can be handled later by refactoring/vcs module
//            if (selectedObjects.length>0) {
//                FileObject f = JavaModel.getFileObject(selectedObjects[0].getResource());
//                if (f!=null && !f.canWrite()) {
//                    result = createProblem(result, true, getCannotRefactor(selectedObjects[0].getResource()));
//                }
//            }
            return result;
        } finally {
            fireProgressListenerStop();
        }
    }
    
    private static final String getCannotRefactor(Resource r) {
        return new MessageFormat(NbBundle.getMessage(RenameRefactoring.class, "ERR_CannotRefactorFile")).format(new Object[] {r.getName()});
    }
    
    
    public Problem prepare(RefactoringElementsBag elements) {
        Problem problem = null, lastProblem = null;
        RefactoringSession session = elements.getSession();
        for (int i = 0; i < refactorings.length; i++) {
            if (cancelRequest) {
                return null;
            }
            Problem p = refactorings[i].prepare(session);
            if (p != null) {
                if (problem == null)
                    problem = lastProblem = p;
                else {
                    lastProblem.setNext(p);
                    lastProblem = p;
                }
            }
        }
        referencesIterator = null;
        return problem;
    }
    
    public void cancelRequest() {
        if (refactorings != null) {
            // if there were initialized refactorings for some fields,
            // cancel their prepare phase
            for (int i = 0; i < refactorings.length; i++) {
                refactorings[i].cancelRequest();
            }
        }
    }
    
    private Problem setParameters(boolean checkOnly, EncapsulateFieldsRefactoring.EncapsulateFieldInfo[] refactorFields, int methodModifier, int fieldModifier, boolean alwaysUseAccessors) {
        if (!checkOnly) {
            refactorings = new EncapsulateFieldRefactoring[refactorFields.length];
        }
        Problem problem = null, lastProblem = null;
        for (int i = 0; i < refactorFields.length; i++) {
            EncapsulateFieldsRefactoring.EncapsulateFieldInfo info = refactorFields[i];
            EncapsulateFieldRefactoring ref;
            
              ref = new EncapsulateFieldRefactoring(info.getField());
            if (!checkOnly) {
                refactorings[i] = ref;
            }
            Problem p;
            ref.setGetterName(info.getGetterName());
            ref.setSetterName(info.getSetterName());
            ref.setMethodModifiers(methodModifier);
            ref.setFieldModifiers(fieldModifier);
            ref.setAlwaysUseAccessors(alwaysUseAccessors);
            if (checkOnly) {
                p = ref.fastCheckParameters();
            } else {
                ref.addProgressListener(listener);
                p = ref.checkParameters();
            }
            if (p != null) {
                if (problem == null)
                    problem = lastProblem = p;
                else {
                    lastProblem.setNext(p);
                    lastProblem = p;
                }
            }
        }
        return problem;
    }
}    
