/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.refactoring.plugins;

import java.io.IOException;
import java.lang.reflect.Modifier;
import java.util.Iterator;
import java.util.List;
import org.netbeans.jmi.javamodel.ClassDefinition;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.Feature;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.JavaEnum;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.Method;
import org.netbeans.jmi.javamodel.MultipartId;
import org.netbeans.jmi.javamodel.MultipartIdClass;
import org.netbeans.jmi.javamodel.NamedElement;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.jmi.javamodel.TypeParameter;
import org.netbeans.jmi.javamodel.TypeReference;
import org.netbeans.jmi.javamodel.UnresolvedClass;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.ExternalChange;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.netbeans.modules.javacore.jmiimpl.javamodel.MetadataElement;
import org.netbeans.modules.refactoring.CheckUtils;
import org.netbeans.modules.refactoring.UndoWatcher;
import org.netbeans.modules.refactoring.api.AbstractRefactoring;
import org.netbeans.modules.refactoring.api.Problem;
import org.netbeans.modules.refactoring.api.RenameRefactoring;
import org.netbeans.modules.refactoring.api.ExtractSuperClassRefactoring;
import org.netbeans.modules.refactoring.Utilities;
import org.netbeans.modules.refactoring.ui.UIUtilities;
import org.netbeans.modules.refactoring.spi.SimpleRefactoringElementImpl;
import org.netbeans.modules.refactoring.spi.RefactoringElementsBag;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileSystem;
import org.openide.filesystems.Repository;
import org.openide.loaders.DataFolder;
import org.openide.loaders.DataObject;
import org.openide.loaders.DataObjectNotFoundException;
import org.openide.text.PositionBounds;
import org.openide.util.NbBundle;

/** Plugin that implements the core functionality of Extract Super Class refactoring.
 *
 * @author Martin Matula
 */
public class ExtractSuperClassRefactoringPlugin extends JavaRefactoringPlugin {
    /** Reference to the parent refactoring instance */
    private final ExtractSuperClassRefactoring refactoring;
    
    /** Creates a new instance of ExtractSuperClassRefactoringPlugin
     * @param refactoring Parent refactoring instance.
     */
    ExtractSuperClassRefactoringPlugin(ExtractSuperClassRefactoring refactoring) {
        this.refactoring = refactoring;
    }
    
    /** Checks pre-conditions of the refactoring.
     * @return Problems found or <code>null</code>.
     */
    public Problem preCheck() {
        // fire operation start on the registered progress listeners (2 step)
        fireProgressListenerStart(AbstractRefactoring.PRE_CHECK, 2);
        try {
            JavaClass sourceType = refactoring.getSourceType();
            
            // check whether the element is valid
            Problem result = isElementAvail(sourceType);
            if (result != null) {
                // fatal error -> don't continue with further checks
                return result;
            }

            if (!CheckUtils.isElementInOpenProject(sourceType)) {
                return new Problem(true, NbBundle.getMessage(JavaRefactoringPlugin.class, "ERR_ProjectNotOpened"));
            }
            
            // check whether the element is an unresolved class
            if (sourceType instanceof UnresolvedClass) {
                // fatal error -> return
                return new Problem(true, NbBundle.getMessage(JavaRefactoringPlugin.class, "ERR_ElementNotAvailable")); // NOI18N
            }
            
            // increase progress (step 1)
            fireProgressListenerStep();
            
            // check whether the element is not an interface or enum
            if (sourceType.isInterface() || (sourceType instanceof JavaEnum)) {
                return new Problem(true, NbBundle.getMessage(ExtractSuperClassRefactoringPlugin.class, "ERR_ExtractSC_MustBeClass")); // NOI18N
            }
            
            // all checks passed -> return null
            return null;
        } finally {
            // fire operation end on the registered progress listeners
            fireProgressListenerStop();
        }
    }
    
    public Problem fastCheckParameters() {
        Problem result = null;
        
        JavaClass sourceType = refactoring.getSourceType();
        String oldName = sourceType.getSimpleName();
        String newName = refactoring.getSuperClassName();
        
        if (!org.openide.util.Utilities.isJavaIdentifier(newName)) {
            result = createProblem(result, true, NbBundle.getMessage(RenameRefactoring.class, "ERR_InvalidIdentifier", newName)); // NOI18N
            return result;
        }
        
        Resource resource = sourceType.getResource();
        FileObject primFile = JavaModel.getFileObject(resource);
        FileObject folder = primFile.getParent();
        FileObject[] children = folder.getChildren();
        for (int x = 0; x < children.length; x++) {
            if (!children[x].isVirtual() && children[x].getName().equals(newName) && "java".equals(children[x].getExt())) { // NOI18N
                result = createProblem(result, true, NbBundle.getMessage(RenameRefactoring.class, "ERR_ClassClash", newName, resource.getPackageName())); // NOI18N
                return result;
            }
        }

        return null;
    }

    public Problem checkParameters() {
        ExtractSuperClassRefactoring.MemberInfo[] members = refactoring.getMembers();

        fireProgressListenerStart(AbstractRefactoring.PARAMETERS_CHECK, members.length);
        try {
            // #1 - check whether all the members are legal members that can be pulled up
            Problem problems = null;
            for (int i = 0; i < members.length; i++) {
                ClassDefinition cls;
                NamedElement member = members[i].member;
                if (member instanceof Feature) {
                    // member is a feature (inner class, field or method)
                    cls = ((Feature) member).getDeclaringClass();
                } else {
                    // member is an interface from implements clause
                    MultipartId ifcName = (MultipartId) member;
                    // get parent of the element (should be class if this is really
                    // a name from implements clause
                    Object parent = ifcName.refImmediateComposite();
                    // if parent is not a class, member is invalid
                    if (!(parent instanceof JavaClass)) {
                        cls = null;
                    } else {
                        // check if the parent class contains this MultipartId
                        // in interfaceNames
                        if (!((JavaClass) parent).getInterfaceNames().contains(ifcName)) {
                            cls = null;
                        } else {
                            cls = (ClassDefinition) parent;
                        }
                    }
                }
                if (!refactoring.getSourceType().equals(cls)) {
                    // if the declaring class of a feature is not the source class, 
                    // then this member is illegal
                    return createProblem(problems, true, NbBundle.getMessage(ExtractSuperClassRefactoringPlugin.class, "ERR_ExtractSC_IllegalMember", member.getName())); // NOI18N
                }
                
                fireProgressListenerStep();
            }

            // TODO: implement non-fatal checks

            return null;
        } finally {
            fireProgressListenerStop();
        }
    }

    public Problem prepare(RefactoringElementsBag refactoringElements) {
        ExtractSuperClassRefactoring.MemberInfo[] members = refactoring.getMembers();

        NamedElement[] membs = new NamedElement[members.length];
        for (int x = 0; x < members.length; x++) {
            membs[x] = members[x].member;
        }
        List typeParams = ExtractInterfaceRefactoringPlugin.findUsedGenericTypes(membs, refactoring.getSourceType());
        
        boolean makeAbstract = Modifier.isAbstract(refactoring.getSourceType().getModifiers());
        for (int i = 0; i < members.length && !makeAbstract; i++) {
            if (members[i].makeAbstract) {
                makeAbstract = true;
            }
        }
        CreateSCElement createSCElement = new CreateSCElement(refactoring.getSourceType().getResource(), refactoring.getSuperClassName(), makeAbstract, typeParams);
        refactoringElements.add(refactoring, createSCElement);
        
        String superClassName = refactoring.getSourceType().getSuperClass().getName();
        refactoringElements.add(refactoring, new AddExtendsElement(refactoring.getSourceType(), refactoring.getSuperClassName(), typeParams));
        if (!"java.lang.Object".equals(superClassName)) { // NOI18N
            refactoringElements.add(refactoring, new AddExtendsElement(createSCElement, superClassName));
        }

        
        for (int i = 0; i < members.length; i++) {
            if (members[i].makeAbstract) {
                // TODO: compute new modifiers and add a refactoring element to
                // change modifiers of the method in necessary
                refactoringElements.add(refactoring, new AddAbstractMethodElement((Method) members[i].member, createSCElement, 0));
            } else {
                // TODO: compute new modifiers
                refactoringElements.add(refactoring, new MoveMemberElement(members[i].member, createSCElement, 0));
            }
        }
        
        // TODO: add refactoring element for changing modifiers of the target class
        // if necessary
        
        return null;
    }
    
    // --- REFACTORING ELEMENTS ------------------------------------------------
    
    private static class CreateSCElement extends SimpleRefactoringElementImpl {
        private final String scName;
        private final Resource source;
        private final List typeParams;
        private final String text;
        private final boolean makeAbstract;
        
        private JavaClass newSC = null;
        
        CreateSCElement(Resource source, String scName, boolean makeAbstract, List typeParams) {
            this.source = source;
            this.scName = scName;
            this.typeParams = typeParams;
            this.makeAbstract = makeAbstract;
            this.text = NbBundle.getMessage(ExtractInterfaceRefactoringPlugin.class, "TXT_ExtractSC_CreateSC", scName); // NOI18N
        }

        public void performChange() {
            ExternalChange ec = new ExternalChange() {
                private FileSystem fs;
                private String newSCName;
                private String folderName;
                
                public void performExternalChange() {
                    try {
                        FileObject tempFO = Repository.getDefault().getDefaultFileSystem().findResource("Templates/Classes/Class.java"); // NOI18N
                        
                        FileObject folderFO;
                        if (fs == null) {
                            FileObject sourceFO = JavaModel.getFileObject(source);
                            folderFO = sourceFO.getParent();
                            folderName = folderFO.getPath();
                            fs = folderFO.getFileSystem();
                        } else {
                            folderFO = fs.findResource(folderName);
                        }
                            
                        DataFolder folder = (DataFolder) DataObject.find(folderFO);
                        DataObject template = DataObject.find(tempFO);
                        DataObject newSCDO = template.createFromTemplate(folder, scName);
                        UndoWatcher.watch(newSCDO);
                        FileObject newSCFO = newSCDO.getPrimaryFile();
                        newSCName = newSCFO.getPath();
                        newSC = (JavaClass) JavaMetamodel.getManager().getResource(newSCFO).getClassifiers().iterator().next();
                    } catch (DataObjectNotFoundException e) {
                        ErrorManager.getDefault().notify(e);
                    } catch (IOException e) {
                        ErrorManager.getDefault().notify(e);
                    }
                }
                
                public void undoExternalChange() {
                    try {
                        FileObject newSCFO = fs.findResource(newSCName);
                        DataObject newSCDO = DataObject.find(newSCFO);
                        newSCDO.delete();
                    } catch (DataObjectNotFoundException e) {
                        ErrorManager.getDefault().notify(e);
                    } catch (IOException e) {
                        ErrorManager.getDefault().notify(e);
                    }
                }
            };
            ec.performExternalChange();
            JavaMetamodel.getManager().registerUndoElement(ec);
            if (makeAbstract) {
                newSC.setModifiers(newSC.getModifiers() | Modifier.ABSTRACT);
            }
            
            List scTypeParams = newSC.getTypeParameters();
            for (Iterator iter = typeParams.iterator(); iter.hasNext(); ) {
                scTypeParams.add(((TypeParameter) iter.next()).duplicate());
            }
        }
        
        JavaClass getNewSuperClass() {
            return newSC;
        }

        public String getText() {
            return text;
        }

        public String getDisplayText() {
            return text;
        }

        public FileObject getParentFile() {
            return JavaModel.getFileObject(source).getParent();
        }

        public Element getJavaElement() {
            return null;
        }

        public PositionBounds getPosition() {
            return null;
        }
    }
    
    /** Refactoring element that takes care of adding an abstract method declaration
     * to the target class.
     */
    private static class AddAbstractMethodElement extends SimpleRefactoringElementImpl {
        private final Method methodToAdd;
        private final CreateSCElement element;
        private final int newModifiers;
        private final String text;
        
        AddAbstractMethodElement(Method methodToAdd, CreateSCElement element, int newModifiers) {
            this.methodToAdd = methodToAdd;
            this.element = element;
            this.newModifiers = newModifiers;
            this.text = NbBundle.getMessage(ExtractSuperClassRefactoringPlugin.class, "TXT_ExtractSC_Method", UIUtilities.getDisplayText(methodToAdd)); // NOI18N
        }

        public void performChange() {
            if (element.getNewSuperClass() == null) return;
            
            // get extent of the target method
            JavaModelPackage extent = (JavaModelPackage) element.getNewSuperClass().refImmediatePackage();
            // create the abstract method in this extent (duplicating the header
            // of the existing method
            Method newMethod = extent.getMethod().createMethod(
                    methodToAdd.getName(),
                    Utilities.duplicateList(methodToAdd.getAnnotations(), extent),
                    (newModifiers == 0 ? methodToAdd.getModifiers() : newModifiers) | Modifier.ABSTRACT,
                    methodToAdd.getJavadocText(),
                    null,
                    null,
                    null, 
                    Utilities.duplicateList(methodToAdd.getTypeParameters(), extent),
                    Utilities.duplicateList(methodToAdd.getParameters(), extent),
                    Utilities.duplicateList(methodToAdd.getExceptionNames(), extent),
                    (TypeReference) ((MetadataElement) methodToAdd.getTypeName()).duplicate(extent),
                    methodToAdd.getDimCount()
                );
            // add the new method to the target class
            element.getNewSuperClass().getContents().add(newMethod);
            ((MetadataElement) newMethod).fixImports(element.getNewSuperClass(), methodToAdd);
        }

        public String getText() {
            return text;
        }

        public String getDisplayText() {
            return text;
        }

        public FileObject getParentFile() {
            return JavaMetamodel.getManager().getFileObject(methodToAdd.getResource());
        }

        public Element getJavaElement() {
            return methodToAdd;
        }

        public PositionBounds getPosition() {
            return JavaMetamodel.getManager().getElementPosition(methodToAdd);
        }
    }
    
    /** Refactoring element that takes care of moving an element to the target type.
     */
    private static class MoveMemberElement extends SimpleRefactoringElementImpl {
        private final NamedElement elementToMove;
        private final CreateSCElement element;
        private final int newModifiers;
        private final String text;
        
        /** Creates a new instance of this refactoring element.
         * @elementToMove Element to be moved to the target type.
         * @target The target type the element should be moved to.
         * @newModifiers New modifiers of the element or 0 if the modifiers should
         *      remain unchanged.
         */
        MoveMemberElement(NamedElement elementToMove, CreateSCElement element, int newModifiers) {
            this.elementToMove = elementToMove;
            this.element = element;
            this.newModifiers = newModifiers;
            this.text = NbBundle.getMessage(ExtractSuperClassRefactoringPlugin.class, "TXT_ExtractSC_Member", UIUtilities.getDisplayText(elementToMove)); // NOI18N
        }

        public void performChange() {
            // remove element from the original type
            Element parent = (Element) elementToMove.refImmediateComposite();
            parent.replaceChild(elementToMove, null);
            // add the element to the super class
            if (elementToMove instanceof MultipartId) {
                element.getNewSuperClass().getInterfaceNames().add(JavaModelUtil.resolveImportsForClass(element.getNewSuperClass(), (JavaClass) ((MultipartId) elementToMove).getElement()));
                elementToMove.refDelete();
            } else {
                Feature feature = (Feature) elementToMove;
                if (newModifiers != 0) {
                    feature.setModifiers(newModifiers);
                }
                element.getNewSuperClass().getContents().add(elementToMove);
                ((MetadataElement) elementToMove).fixImports(element.getNewSuperClass(), elementToMove);
            }
        }

        public String getText() {
            return text;
        }

        public String getDisplayText() {
            return text;
        }

        public FileObject getParentFile() {
            return JavaMetamodel.getManager().getFileObject(elementToMove.getResource());
        }

        public Element getJavaElement() {
            return JavaModelUtil.getDeclaringFeature(elementToMove);
        }

        public PositionBounds getPosition() {
            return JavaMetamodel.getManager().getElementPosition(elementToMove);
        }
    }
    
    private static class AddExtendsElement extends SimpleRefactoringElementImpl {
        private final CreateSCElement element;
        private final String text;
        private JavaClass sourceType;
        private final List typeParams;
        private String scName;
        
        AddExtendsElement(JavaClass sourceType, String scName, List typeParams) {
            this(sourceType, null, scName, typeParams);
        }
        
        AddExtendsElement(CreateSCElement element, String scName) {
            this(null, element, scName, null);
        }
        
        private AddExtendsElement(JavaClass sourceType, CreateSCElement element, String scName, List typeParams) {
            this.sourceType = sourceType;
            this.element = element;
            this.scName = scName;
            this.typeParams = typeParams;
            this.text = NbBundle.getMessage(ExtractInterfaceRefactoringPlugin.class, "TXT_ExtractSC_AddExtends", scName); // NOI18N
        }
        
        public void performChange() {
            MultipartId mpi;
            if (sourceType == null) {
                sourceType = element.getNewSuperClass();
                mpi = JavaModelUtil.resolveImportsForClass(sourceType, (JavaClass) JavaModel.getDefaultExtent().getType().resolve(scName));
            } else {
                JavaModelPackage extent = (JavaModelPackage) sourceType.refImmediatePackage();
                mpi = extent.getMultipartId().createMultipartId(scName, null, null);
            }
            if (typeParams != null && typeParams.size() > 0) {
                List typeArgs = mpi.getTypeArguments();
                MultipartIdClass idProxy = ((JavaModelPackage) sourceType.refImmediatePackage()).getMultipartId();
                for (Iterator iter = typeParams.iterator(); iter.hasNext(); ) {
                    typeArgs.add(idProxy.createMultipartId(((TypeParameter) iter.next()).getName(), null, null));
                }
            }
            sourceType.setSuperClassName(mpi);
        }

        public String getText() {
            return text;
        }

        public String getDisplayText() {
            return text;
        }

        public FileObject getParentFile() {
            return sourceType == null ? element.getParentFile() : JavaMetamodel.getManager().getFileObject(sourceType.getResource());
        }

        public Element getJavaElement() {
            return sourceType == null ? element.getJavaElement() : sourceType;
        }

        public PositionBounds getPosition() {
            return sourceType == null ? element.getPosition() : JavaMetamodel.getManager().getElementPosition(sourceType);
        }
    }
}
