/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.refactoring.plugins;

import java.io.File;
import java.io.IOException;
import java.io.SyncFailedException;
import java.net.URL;
import java.text.MessageFormat;
import java.util.*;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.jmi.javamodel.Import;
import org.netbeans.jmi.javamodel.NamedElement;
import org.netbeans.modules.javacore.ClassIndex;
import org.netbeans.modules.javacore.internalapi.ExternalChange;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.refactoring.CheckUtils;
import org.netbeans.modules.refactoring.RenameFullNameElement;
import org.netbeans.modules.refactoring.api.AbstractRefactoring;
import org.netbeans.modules.refactoring.api.MoveClassRefactoring;
import org.netbeans.modules.refactoring.api.Problem;
import org.netbeans.modules.refactoring.spi.RefactoringElementsBag;
import org.netbeans.modules.refactoring.api.RenameRefactoring;
import org.netbeans.modules.refactoring.classpath.RefactoringClassPathImplementation;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileStateInvalidException;
import org.openide.filesystems.FileSystem;
import org.openide.filesystems.URLMapper;
import org.openide.loaders.DataFolder;
import org.openide.loaders.DataObject;
import org.openide.text.PositionBounds;
import org.openide.util.NbBundle;
import org.openide.util.Utilities;
import java.lang.reflect.Modifier;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.refactoring.spi.RefactoringElementImplementation;
import org.netbeans.modules.refactoring.spi.SimpleRefactoringElementImpl;

public class MoveClassRefactoringPlugin extends JavaRefactoringPlugin {
    
    private Map/*<Resource, DataObject>*/   sourceObjectsMap;
    private Collection/*<Resource>*/        resources;
    private Collection                      otherDataObjects = Collections.EMPTY_LIST;
    private Map /*<Resource, JavaClassp[]>*/sourceClassesMap;
    private Collection /*<JavaClass>*/      allClasses;
    private boolean                         particularClassSelected;
    private Map /*<Resource, Boolean>*/     accessesOriginalPackageMap;
    private Map /*<Resource, Boolean>*/     accessedByOriginalPackageMap;
    private Map /*<Resource, Boolean>*/     movingToDefaultPackageMap;
    private Map /*<Resource, Boolean>*/     movingFromDefaultPackageMap;
    private FileObject                      folder = null;
    private MoveClassRefactoring            refactoring;
    private ClassPath                       classPath;

    private Map checkedSourcesMap;
    
    private int size;
    
    public MoveClassRefactoringPlugin(MoveClassRefactoring refactoring) {
        this.refactoring = refactoring;
        this.resources = refactoring.getResources();
        this.otherDataObjects = refactoring.getOtherDataObjects();
        classPath = RefactoringClassPathImplementation.getDefault();
        folder = refactoring.getSourceFolder();
    }
            
    
    private Problem init() {
        this.sourceObjectsMap = new HashMap(resources.size());

        Problem prob = null;
        for (Iterator i = resources.iterator(); i.hasNext();) {
            Resource r = (Resource) i.next();
            if (r.isValid()) {
                DataObject dob = JavaMetamodel.getManager().getDataObject(r);
                sourceObjectsMap.put(r, dob);
                
                if (CheckUtils.isFromLibrary(r)) {
                    prob = createProblem(prob, true, getCannotMove(dob.getPrimaryFile()));
                }

            } else {
                i.remove();
            }
        }
        
// isMoveAllowed() check disabled. It can be handled later by refactoring/vcs module
//        for (Iterator i = otherDataObjects.iterator(); i.hasNext(); ) {
//            DataObject dob = (DataObject) i.next();
//            if (!dob.isMoveAllowed()) {
//                prob = createProblem(prob, true, getCannotMove(dob.getPrimaryFile()));
//            }
//        }

        size = resources.size();
        this.accessesOriginalPackageMap = new HashMap(size);
        this.accessedByOriginalPackageMap = new HashMap(size);
        this.movingToDefaultPackageMap = new HashMap(size);
        this.movingFromDefaultPackageMap = new HashMap(size);
        this.sourceClassesMap = new HashMap(size*2);
        this.checkedSourcesMap = new HashMap(size);
        this.allClasses = new ArrayList(10);
       
        for (Iterator i = resources.iterator(); i.hasNext();) {
            Resource r = (Resource) i.next();
            List list = new LinkedList();
            list.addAll(r.getClassifiers());
            sourceClassesMap.put(r, (JavaClass []) list.toArray(new JavaClass[list.size()]));
            checkedSourcesMap.put(r, new HashSet());
        }
        
        for (Iterator i = refactoring.getSelectedDataObjects().iterator(); i.hasNext();) {
            DataObject dob = (DataObject) i.next();
            if (dob instanceof DataFolder) {
                putSubfolders(dob.getPrimaryFile());
            }
        }
        return prob;
    }
    
    private static final String getCannotMove(FileObject f) {
        ClassPath cp = ClassPath.getClassPath(f, ClassPath.SOURCE);
        if (cp==null) 
            cp = ClassPath.getClassPath(f, ClassPath.BOOT);
        
        String name = cp.getResourceName(f);
        return new MessageFormat(NbBundle.getMessage(RenameRefactoring.class, "ERR_CannotMoveFile")).format(new Object[] {name});
    }

    public Problem preCheck() {
        Problem prob = init();
        if (prob != null)
            return prob;
        for (Iterator i = resources.iterator(); i.hasNext(); ) {
            Resource res = (Resource) i.next();
            if (!CheckUtils.isElementInOpenProject(res)) {
                return new Problem(true, NbBundle.getMessage(JavaRefactoringPlugin.class, "ERR_ProjectNotOpened"));
            }
            JavaClass[] sourceClasses = (JavaClass[]) sourceClassesMap.get(res);
            for (int x = 0; x < sourceClasses.length; x++)
                collectAllClasses(allClasses, sourceClasses [x]);
            
        }
        fireProgressListenerStart(AbstractRefactoring.PRE_CHECK, 3*size + 3);
        
        for (Iterator i = resources.iterator(); i.hasNext();) {
            if (cancelRequest)
                return null;
            Resource r = (Resource) i.next();
            accessedByOriginalPackageMap.put(r, Boolean.valueOf(isAccessedByOriginalPackage(r)));
            JavaPackage sourcePackage = (JavaPackage) r.refImmediateComposite();
            movingFromDefaultPackageMap.put(r, Boolean.valueOf("".equals(sourcePackage.getName())));
            accessesOriginalPackageMap.put(r, Boolean.FALSE);
        }
        
        try {
            for (Iterator i = resources.iterator(); i.hasNext();) {
                Resource r = (Resource) i.next();
                JavaClass[] sourceClasses = (JavaClass[]) sourceClassesMap.get(r);
                
//                this check can be omitted to allow to refactor "empty files"
//                if (sourceClasses.length == 0) {
//                    return new Problem(true, getString("ERR_SourceWithoutClass")); // NOI18N
//                }
                fireProgressListenerStep();
                if ((sourceClasses.length > 1) && particularClassSelected) {
                    return new Problem(true, getString("ERR_SourceWithMoreClasses")); // NOI18N
                }
                fireProgressListenerStep();
            }
            
            Problem problem = checkUsedElements(null);
            
            boolean defaultPackageWarned = false;
            for (Iterator i = resources.iterator(); i.hasNext();) {
                Resource r = (Resource) i.next();
                fireProgressListenerStep();
                Boolean boolVal = (Boolean) accessesOriginalPackageMap.get(r);
                boolean accessesOriginalPackage = boolVal != null ? boolVal.booleanValue() : false;
                boolean movingFromDefaultPackage = ((Boolean) movingFromDefaultPackageMap.get(r)).booleanValue();
                if (accessesOriginalPackage && movingFromDefaultPackage && !defaultPackageWarned) {
                    Problem p = new Problem(false, getString("ERR_ClassToMoveInDefaultPackage")); // NOI18N
                    p.setNext(problem);
                    problem = p;
                    defaultPackageWarned = true;
                }
            }
            
            fireProgressListenerStep();
            fireProgressListenerStep();
            problem = checkAccessesMyFeature(problem);
            
            fireProgressListenerStep();
            return problem;
        } finally {
            fireProgressListenerStop();
        }
    }        
    
    public Problem fastCheckParameters() {
        return checkParameters(refactoring.getTargetClassPathRoot(), refactoring.getTargetPackageName());
    }
    
    public Problem checkParameters() {
        return setParameters(refactoring.getTargetClassPathRoot(), refactoring.getTargetPackageName());
    }
    
    public Problem prepare(RefactoringElementsBag elements) {
        Problem problem = null;
        Problem p;
                
        fireProgressListenerStart(refactoring.PREPARE, 2 + 2 * resources.size());
        
        try {
//            if (targetFile == null) {
//                p = new Problem(true, getString("ERR_TargetFolderNotSet")); // NOI18N
//                p.setNext(problem);
//                return p;
//            }        
            fireProgressListenerStep();

            ClassPath thisCP = null;
            FileObject targetRoot = null;

//            thisCP = ClassPath.getClassPath(folder, ClassPath.SOURCE);
//            targetRoot = thisCP.findOwnerRoot(folder);
            checkWhereUsed(elements);                        
            fireProgressListenerStep();
            
            for (Iterator i = resources.iterator(); i.hasNext();) {
                if (cancelRequest) {
                    return null;
                }
                Resource resource = (Resource) i.next();
                JavaPackage sourcePackage = (JavaPackage) resource.refImmediateComposite();
                Boolean boolVal = (Boolean) accessesOriginalPackageMap.get(resource);
                boolean accessesOriginalPackage = boolVal != null ? boolVal.booleanValue() : false;
                boolean movingFromDefaultPackage = ((Boolean) movingFromDefaultPackageMap.get(resource)).booleanValue();
                if (accessesOriginalPackage && !movingFromDefaultPackage && !containsImport(resource, sourcePackage.getName())) {
                    elements.add(refactoring, new InsertImport(resource, sourcePackage.getName(), false));
                }
                fireProgressListenerStep();
                
                DataObject sourceObject = (DataObject) sourceObjectsMap.get(resource);
                elements.add(refactoring, new MoveClass(sourceObject, refactoring.getTargetClassPathRoot(), refactoring.getTargetPackageName(resource), (JavaClass[]) sourceClassesMap.get(resource)));
                
                fireProgressListenerStep();
            }
            
            removeUnusedImports(elements);
            for (Iterator i = otherDataObjects.iterator(); i.hasNext();) {
                DataObject dob = (DataObject) i.next();
                elements.add(refactoring, new MoveClass(dob,refactoring.getTargetClassPathRoot(), refactoring.getTargetPackageName(dob.getPrimaryFile()), null));
            }
            
            return null;
        } finally {
            referencesIterator = null;
            fireProgressListenerStop();
        }
    }
    
    private Collection getOldPackages() {
        Collection set = new HashSet(3);
        for (Iterator i = resources.iterator(); i.hasNext();) {
            Resource r = (Resource) i.next();
            set.add(r.refImmediateComposite());
        }
        return set;
    }
    
    private Collection getEmptyPackages() {
        Collection packages = new HashSet(3);
        for (Iterator i = getOldPackages().iterator(); i.hasNext();) {
            JavaPackage pack = (JavaPackage) i.next();
            Collection packResources = new HashSet(pack.getResources());
            packResources.removeAll(resources);
            packResources.remove(null);
            if (packResources.isEmpty()) {
                packages.add(pack);
            }
        }
        return packages;
    }
    private void removeUnusedImports(RefactoringElementsBag elements) {
        for (Iterator iter = getEmptyPackages().iterator(); iter.hasNext();) {
        JavaPackage oldPackage = (JavaPackage) iter.next();
        Resource referenced[] = findReferencedResources(oldPackage.getName());
        for (int i = 0; i < referenced.length; i++) {
            List imports = referenced[i].getImports();
            for (Iterator it = imports.iterator(); it.hasNext();) {
                Import imp = (Import) it.next();
                NamedElement el = imp.getImportedNamespace();
                if (oldPackage.equals(el)) {
                    elements.add(refactoring, new RemoveImport(imp));
                }
            }
        }
        }
        
        for (Iterator i = resources.iterator(); i.hasNext();) {
            Resource r = (Resource) i.next();
            String targetPkgName = refactoring.getTargetPackageName(r);
            
            for (Iterator importsIterator = r.getImports().iterator(); importsIterator.hasNext();) {
                Import imp = (Import) importsIterator.next();
                NamedElement ne = imp.getImportedNamespace();
                if (ne instanceof UnresolvedClass)
                    continue;
                if (ne instanceof JavaClass)
                    ne = (NamedElement) ne.getResource().refImmediateComposite();
                if (!(ne instanceof JavaPackage))
                    continue;
                if (ne.getName().equals(targetPkgName)) {
                    elements.add(refactoring, new RemoveImport(imp));
                }
            }
        }
    }
    
    private static Resource[] findReferencedResources(String packageName) {
        if ("".equals(packageName)) {
            return new Resource[0];
        }
        StringTokenizer tokenizer = new StringTokenizer(packageName, "."); //NOI18N
        String part = tokenizer.nextToken();
        List result = new ArrayList(Arrays.asList((Object[]) ClassIndex.findResourcesForIdentifier(part)));

        while (tokenizer.hasMoreTokens()) {
            part = tokenizer.nextToken();
            result.retainAll(Arrays.asList((Object[]) ClassIndex.findResourcesForIdentifier(part)));
        }
        return (Resource[]) result.toArray(new Resource[result.size()]);
    }


    private static String getPackageName(ClassPath cp, FileObject f) {
        return cp.getResourceName(f, '.', false);
    }
    
    private boolean isFullName(MultipartId id) {
        while (id != null) {
            if (id.getElement() instanceof JavaPackage)
                return true;
            id = id.getParent();
        }
        return false;
    }
    
    private Problem checkUsedElements(Problem problem) {
        for (Iterator iter = allClasses.iterator(); iter.hasNext();) {
            if (cancelRequest)
                return null;
            JavaClass jc = (JavaClass) iter.next(); // [TODO] anonymous classes
            problem = checkUsedElements(jc, new HashSet(), problem);
        }
        return problem;
    }
    
    private Problem checkUsedElements(Element elem, Set markedResources, Problem problem) {
        List elements = elem.getChildren();
        Iterator iter = elements.iterator();
        
        while (iter.hasNext()) {
            if (cancelRequest)
                return null;
            Object el = iter.next();
            if ((el instanceof JavaClass) || (el instanceof Feature && !(elem instanceof Feature)))
                continue;
            if (el instanceof ElementReference) {
                NamedElement namedElem = ((ElementReference) el).getElement();

                //workaround for #45796 .. is it necessary to handle arrays differently?
                while(namedElem instanceof Array) {
                    namedElem = ((Array) namedElem).getType();
                }
                if (namedElem instanceof Feature) {
                    Feature f = (Feature) namedElem;
                    Resource res = f.getResource();
                    if (res != null && !resources.contains(res) && !markedResources.contains(res)) {
                        JavaPackage pkg = (JavaPackage) res.refImmediateComposite();
                        
                        boolean accessesOriginalPackage = false;
                        for (Iterator i = resources.iterator(); i.hasNext();) {
                            Resource resource = (Resource) i.next();
                            JavaPackage sourcePackage = (JavaPackage) resource.refImmediateComposite();
                            if (sourcePackage == pkg) {
                                accessesOriginalPackageMap.put(resource, Boolean.TRUE);
                                accessesOriginalPackage = true;
                            }
                        }

//                        if (!accessesOriginalPackage)
//                            continue;
                        
                        int modifiers = f.getModifiers();
                        if (!(f instanceof Initializer) && ((modifiers & Modifier.PUBLIC) == 0) &&
                            ((modifiers & Modifier.PRIVATE) == 0) && 
                            (!Modifier.isProtected(modifiers) || (!getEnclosingClass((Element) el).isSubTypeOf((JavaClass)f.getDeclaringClass()) && Modifier.isProtected(modifiers) && !allClasses.contains(getEnclosingClass((Element) f))))) {
                            markedResources.add(res);
                            String modStr = (modifiers & Modifier.PROTECTED) != 0 ? getString ("LBL_Protected") : getString ("LBL_PackagePrivate"); // NOI18N
                            Problem p;
                            if (f instanceof JavaClass) {
                                p = new Problem (false,
                                    new MessageFormat(getString("ERR_AccessesPackagePrivateClass2")).format ( // NOI18N
                                        new Object[] {
                                            getEnclosingClass((Element)el).getSimpleName(),
                                            modStr,
                                            ((JavaClass) f).getSimpleName ()
                                        }
                                    )
                                );
                            } else if (f instanceof Constructor) {
                                p = new Problem (false,
                                    new MessageFormat(getString("ERR_AccessesPackagePrivateConstructor2")).format ( // NOI18N
                                        new Object[] {
                                            getEnclosingClass((Element)el).getSimpleName(),
                                            modStr,
                                            ((JavaClass)f.getDeclaringClass()).getSimpleName()
                                        }
                                    )
                                );
                            } else {
                                p = new Problem (false,
                                    new MessageFormat(getString("ERR_AccessesPackagePrivateFeature2")).format ( // NOI18N
                                        new Object[] {
                                            getEnclosingClass((Element)el).getSimpleName(),
                                            modStr,
                                            getFeatureKindName(f),
                                            f.getName (),
                                            ((JavaClass)f.getDeclaringClass()).getSimpleName()
                                        }
                                    )
                                );
                            }
                            p.setNext(problem);
                            problem = p;
                        } // if
                    } // if
                } // if
            } // if
            if (!(el instanceof TypeReference))
                problem = checkUsedElements((Element) el, markedResources, problem);
        } // while
        return problem;
    }
    
    private void checkWhereUsed(RefactoringElementsBag elements) {
        for (Iterator i = resources.iterator(); i.hasNext();) {
            Resource resource = (Resource) i.next();
            if (!refactoring.getTargetPackageName(resource).equals(((JavaPackage)resource.refImmediateComposite()).getName())) {
                JavaClass[] sourceClasses = (JavaClass[]) sourceClassesMap.get(resource);
                for (int x = 0; x < sourceClasses.length; x++) {
                    checkWhereUsed(elements, sourceClasses[x], resource);
                }
            }
        }
    }
    
    private Element getImport(Element e) {
        Element orig = e;
        while (e instanceof MultipartId) {
            e = (Element) e.refImmediateComposite();
        }
        if (e instanceof Import) {
            return e;
        } else {
            return orig;
        }
    }
    
    private void checkWhereUsed(RefactoringElementsBag elements, JavaClass jc, Resource r) {
        String targetPkgName = refactoring.getTargetPackageName(r);
        JavaPackage targetPackage = ((JavaModelPackage) jc.refImmediatePackage()).getJavaPackage().resolvePackage(targetPkgName);
        Collection users = jc.getReferences();
        boolean movingToDefaultPackage = ((Boolean) movingToDefaultPackageMap.get(r)).booleanValue();
        for (referencesIterator = users.iterator(); referencesIterator.hasNext();) {
            if (cancelRequest) {
                return ;
            }
            Element elem = getImport((Element) referencesIterator.next());
            if (elem instanceof MultipartId) {
                if (isFullName((MultipartId)elem)) {
                    elements.add(refactoring, new RenameFullNameElement(jc, (MultipartId)elem, targetPkgName));
                } else {
                    // check if an import is required to be added
                    Resource res = elem.getResource();
                    JavaPackage resPkg = (JavaPackage) res.refImmediateComposite();
                    Set checkedSources = (Set) checkedSourcesMap.get(r);
                    if (!movingToDefaultPackage && !resources.contains(res) && !checkedSources.contains(res)) {
                        checkedSources.add(res);
                        
                        if ((resPkg != targetPackage) && !containsImport(res, jc.getName())) {
                        //[FIXME] - this code needs to be checked if it is correct
                        //if ((isPackageRename || resPkg != targetPackage) && !containsImport(res, jc.getName())) {
                            elements.add(refactoring, new InsertImport(res, targetPkgName + '.' + jc.getSimpleName(), true));
                        }
                    }
                }
            } else if (elem instanceof Import) {
                if (movingToDefaultPackage) {
                    elements.add(refactoring, new RemoveImport((Import)elem));
                } else {
                    MultipartId id = ((Import) elem).getIdentifier();
                    while (id != null && !jc.equals(id.getElement())) {
                        id = id.getParent();
                    }
                    assert id != null: "id is null, resource = " + elem.getResource().getName() + " jc = " + jc;
                    elements.add(refactoring, new RenameFullNameElement(jc, id, targetPkgName));
                }
            }
        }
    }

    private JavaClass getEnclosingClass(Element elem) {
        while (elem != null && !(elem instanceof JavaClass))
            elem = (Element) elem.refImmediateComposite();
        return (JavaClass)elem;
    }
    
    private boolean containsImport(Resource res, JavaClass jc) {
        JavaPackage pkg = (JavaPackage) jc.getResource().refImmediateComposite();
        for (Iterator iter = res.getImports().iterator(); iter.hasNext();) {
            Import imp = (Import) iter.next();
            NamedElement elem = imp.getImportedNamespace();
            if (elem == jc || elem == pkg)
                return true;
        }
        return false;
    }

    private boolean containsImport(Resource res, String impName) {
        for (Iterator iter = res.getImports().iterator(); iter.hasNext();) {
            Import imp = (Import) iter.next();
            if (impName.equals(imp.getName()))            
                return true;
        }
        return false;
    }
    
    private Problem checkAccessesMyFeature(Problem problem) {
        for (Iterator iter = allClasses.iterator(); iter.hasNext();) {
            if (cancelRequest)
                return null;
            JavaClass jc = (JavaClass) iter.next();
            problem = checkAccessesMyFeature(problem, jc);
        }
        return problem;
    }
    
    private Problem checkAccessesMyFeature(Problem problem, JavaClass jc) {
        boolean classChecked = false;
        Set set = new HashSet();
        for (Iterator iter = jc.getFeatures().iterator(); iter.hasNext() || !classChecked;) {
            Feature f;
            if (!classChecked) {
                f = jc;
                classChecked = true;
            } else {        
                f = (Feature) iter.next();
                if (f instanceof JavaClass)
                    continue;
            }
            int modifiers = f.getModifiers();
            if (!(f instanceof Initializer) && ((modifiers & Modifier.PUBLIC) == 0) && ((modifiers & Modifier.PRIVATE) == 0)) {
                boolean isProtected = (modifiers & Modifier.PROTECTED) != 0;
                Collection users = f.getReferences();
                for (referencesIterator = users.iterator(); referencesIterator.hasNext();) {
                    if (cancelRequest) {
                        return null;
                    }
                    Element elem = (Element) referencesIterator.next();
                    Resource res = elem.getResource();
                    String modStr = (modifiers & Modifier.PROTECTED) != 0 ? getString ("LBL_Protected") : getString ("LBL_PackagePrivate"); // NOI18N
                    if (!resources.contains(res) && set.add(res)) {
                        JavaPackage pkg = (JavaPackage) res.refImmediateComposite();
                        boolean cont = false;
                        for (Iterator i = resources.iterator(); i.hasNext();) {
                            JavaPackage sourcePackage = (JavaPackage) ((Resource)i.next()).refImmediateComposite();
                            if (pkg != sourcePackage) {
                                cont = true;
                            }
                        }
                        if (cont)
                            continue;
                        if (getEnclosingClass(elem).isSubTypeOf(jc))
                            continue;
                        Problem p;
                        if (f instanceof JavaClass) {
                            p = new Problem (false,
                            new MessageFormat(getString("ERR_AccessesPackagePrivateClass")).format ( // NOI18N
                                new Object[] {
                                    getEnclosingClass(elem).getSimpleName(),
                                    modStr,
                                    ((JavaClass) f).getSimpleName ()
                                }
                            ));
                        } else if (f instanceof Constructor) {
                            p = new Problem (false,
                                new MessageFormat(getString("ERR_AccessesPackagePrivateConstructor")).format ( // NOI18N
                                    new Object[] {
                                        getEnclosingClass(elem).getSimpleName(),
                                        modStr,
                                        ((JavaClass)f.getDeclaringClass()).getSimpleName()
                                    }
                                )
                            );
                        } else {
                            p = new Problem (false,
                            new MessageFormat(getString("ERR_AccessesPackagePrivateFeature")).format ( // NOI18N
                                new Object[] {
                                    getEnclosingClass(elem).getSimpleName(),
                                    modStr,
                                    getFeatureKindName(f),
                                    f.getName (),
                                    ((JavaClass) f.getDeclaringClass ()).getSimpleName ()
                                }
                            ));
                        }
                        p.setNext(problem);
                        problem = p;
                    }
                }
            } // if
        } // for
        return problem;
    }

    private boolean isAccessedByOriginalPackage(Resource r) {
        JavaClass[] sourceClasses  = (JavaClass[]) sourceClassesMap.get(r);
        JavaPackage sourcePackage = (JavaPackage) r.refImmediateComposite();

        for (int x = 0; x < sourceClasses.length; x++) {
            for (referencesIterator = sourceClasses[x].getReferences().iterator(); referencesIterator.hasNext();) {
                if (cancelRequest)
                    return false;
                Element elem = (Element) referencesIterator.next();
                Resource res = elem.getResource();
                JavaPackage pkg = (JavaPackage) res.refImmediateComposite();
                if (/*pkg == sourcePackage &&*/ !resources.contains(res))
                    return true;
            }
        }
        return false;
    }
    
    private String getFeatureKindName(Feature f) {
        String featureKind;
        if (f instanceof Method) {
            featureKind = getString ("LBL_Method"); // NOI18N
        } else if (f instanceof Field) {
            featureKind = getString ("LBL_Field"); // NOI18N
        } else if (f instanceof Constructor) {
            featureKind = getString ("LBL_Constructor"); // NOI18N
        } else if (f instanceof JavaClass) {
            featureKind = getString ("LBL_Class"); // NOI18N
        } else {
            featureKind = "???"; // NOI18N
        }
        return featureKind;
    }
    
    private void collectAllClasses (Collection classes, JavaClass jc) {
        classes.add (jc);
        Iterator iter = jc.getFeatures().iterator ();
        while (iter.hasNext ()) {
            Object obj = iter.next();
            if (obj instanceof JavaClass)
                collectAllClasses (classes, (JavaClass) obj);
        }
    }
    
    private FileObject target;
    
    private Problem setParameters (FileObject target, String newPackageName) {
        this.newPackageName = newPackageName;
        
        Problem p = null;
        this.target = target;
        fireProgressListenerStart(refactoring.PARAMETERS_CHECK, 1 + resources.size());
        try {

            for (Iterator i = resources.iterator(); i.hasNext();) {
                Resource r = (Resource) i.next(); 
                FileObject targetRoot = refactoring.getTargetClassPathRoot();
                String targetPackageName = refactoring.getTargetPackageName(r);
                FileObject targetF = targetRoot.getFileObject(targetPackageName.replace('.', '/'));
            
                String pkgName = null;
                if ((targetF!=null && !targetF.canWrite())) {
                    return new Problem(true, new MessageFormat(getString("ERR_PackageIsReadOnly")).format( // NOI18N
                    new Object[] {targetPackageName}
                    ));
                }
                
                this.movingToDefaultPackageMap.put(r, Boolean.valueOf(targetF!= null && targetF.equals(classPath.findOwnerRoot(targetF))));
                pkgName = targetPackageName;
            
                if (pkgName == null) {
                    pkgName = ""; // NOI18N
                } else if (pkgName.length() > 0) {
                    pkgName = pkgName + '.';
                }
                //targetPrefix = pkgName;
                fireProgressListenerStep();
                
                JavaClass[] sourceClasses = (JavaClass[]) sourceClassesMap.get(r);
                String[] names = new String [sourceClasses.length];
                for (int x = 0; x < names.length; x++) {
                    names [x] = sourceClasses [x].getName();
                }
                
                FileObject movedFile = JavaMetamodel.getManager().getDataObject(r).getPrimaryFile();
                String fileName = movedFile.getName();
                if (targetF!=null) {
                    FileObject[] children = targetF.getChildren();
                    for (int x = 0; x < children.length; x++) {
                        if (children[x].getName().equals(fileName) && "java".equals(children[x].getExt()) && !children[x].equals(movedFile) && !children[x].isVirtual()) { //NOI18N
                            return new Problem(true, new MessageFormat(
                                    getString("ERR_ClassToMoveClashes")).format(new Object[] {fileName} // NOI18N
                            ));
                        }
                    } // for
                }
                
                boolean accessedByOriginalPackage = ((Boolean) accessedByOriginalPackageMap.get(r)).booleanValue();
                boolean movingToDefaultPackage = ((Boolean) movingToDefaultPackageMap.get(r)).booleanValue();
                if (p==null && accessedByOriginalPackage && movingToDefaultPackage) {
                    p= new Problem(false, getString("ERR_MovingClassToDefaultPackage")); // NOI18N
                }
                
                if (((DataObject) sourceObjectsMap.get(r)).getFolder().getPrimaryFile().equals(targetF) && isPackageCorrect(r)) {
                    return new Problem(true, getString("ERR_CannotMoveIntoSamePackage"));
                }
            }
            fireProgressListenerStep();
        } finally {
            fireProgressListenerStop();
        }
        return p;
    }
    
    private boolean isPackageCorrect(Resource r) {
        String resourceName = r.getName();
        String estimatedPackageName;
        int index = resourceName.lastIndexOf('/');
        if (index == -1) {
            estimatedPackageName = "";
        } else {
            estimatedPackageName = resourceName.substring(0, index);
        }
        return estimatedPackageName.replace('/', '.').equals(r.getPackageName());
    }
    
    private boolean isPackageRename = false;
    private String newPackageName;
    
    private Problem checkParameters(FileObject targetClassPathRoot, String newName) {
        if (!isValidPackageName(newName)) {
            String s = getString("ERR_InvalidPackage");
            String msg = new MessageFormat(s).format(
                new Object[] {newName}
            );
            return new Problem(true, msg);
        }
        return null;
    }
    
    private static boolean isValidPackageName(String name) {
        if (name == null) {
            return false;
        }
        if (name.startsWith(".") || name.endsWith(".")) //NOI18N
            return false;
        StringTokenizer tokenizer = new StringTokenizer(name, "."); // NOI18N;
        while (tokenizer.hasMoreTokens()) {
            if (!Utilities.isJavaIdentifier(tokenizer.nextToken())) {
                return false;
            }
        }
        return true;
    }
    
    static final JavaPackage findJavaPackage (ClassPath cp, FileObject file) {
        FileObject root = cp.findOwnerRoot(file);
        JavaModelPackage jmp = JavaModel.getJavaExtent(root);
        JavaPackageClass proxy = jmp.getJavaPackage();
        return proxy.resolvePackage(getPackageName(cp, file));
    }
    
    static final String getString(String key) {
        return NbBundle.getMessage(MoveClassRefactoring.class, key);
    }
    
    // --------------------------------------------------------------------------
    private FileObject createFolder(FileObject folder, String name, boolean isUndo)
    throws IOException {
        String separators;
        if (File.separatorChar != '/')
            separators = "/" + File.separatorChar; // NOI18N
        else
            separators = "/";   // NOI18N
        
        StringTokenizer st = new StringTokenizer(name, separators);
        while (st.hasMoreElements()) {
            name = st.nextToken();
            if (name.length() > 0) {
                FileObject f = folder.getFileObject(name);
                if (f == null) {
                    try {
                        f = folder.createFolder(name);
                        if (isUndo) {
                            createdFolderNames.add(f.getURL());
                        } else {
                            createdFolderNamesDuringUndo.add(f.getURL());
                        }
                    } catch (SyncFailedException ex) {
                        // there might be unconsistency between the cache
                        // and the disk, that is why
                        folder.refresh();
                        // and try again
                        f = folder.getFileObject(name);
                        if (f == null) {
                            // if still not found than we have to report the
                            // exception
                            throw ex;
                        }
                    }
                }
                folder = f;
            }
        }
        return folder;
    }
    
    private void putSubfolders(FileObject fo) {
        if (".svn".equalsIgnoreCase(fo.getNameExt()) || "_svn".equalsIgnoreCase(fo.getNameExt())) { //NOI18N
            return;
        }
        try {
            createdFolderNames.add(fo.getURL());
        } catch (FileStateInvalidException fsi) {
            ErrorManager.getDefault().notify(fsi);
        }
        FileObject children[] = fo.getChildren();
        for (int i = 0; i<children.length; i++) {
            if (children[i].isFolder()) {
                putSubfolders(children[i]);
            }
        }
    }

    
    private ArrayList createdFolderNames = new ArrayList();
    private ArrayList createdFolderNamesDuringUndo = new ArrayList();
    
    
    // MoveClass ................................................................
    class MoveClass extends SimpleRefactoringElementImpl implements RefactoringElementImplementation, ExternalChange {
        
        private DataObject source;
        private FileObject target;
        private FileObject oldTarget = null;
        private JavaClass[] sourceClasses;
        private PositionBounds bounds = null;
        
        private DataObject copiedObject = null;
        
        private FileObject targetRoot = null;
        private String packageName = null;
        
        public MoveClass (DataObject source, FileObject target, JavaClass[] sourceClasses) {
            this.source = source;
            this.target = target;
            this.sourceClasses = sourceClasses;
        }
        
        public MoveClass (DataObject source, FileObject targetRoot, String packageName,  JavaClass[] sourceClasses) {
            this.source = source;
            this.targetRoot = targetRoot;
            this.sourceClasses = sourceClasses;
            this.packageName = packageName;
        }
        
        public String getText() {
            return getDisplayText ();
        }
    
        public String getDisplayText() {
            String name;
            if (target == null) {
                name = packageName;
            } else {
                name = getPackageName(classPath, target);
            }
            if ("".equals(name)) {
                name = getString ("LBL_DefaultPackage"); // NOI18N 
            }
            if (sourceClasses == null) {
                return new MessageFormat(MoveClassRefactoringPlugin.getString("LBL_MoveObject")).format( // NOI18N
                    new Object[] {source.getNodeDelegate().getDisplayName(), name}
                );
                
            } else {
                return new MessageFormat(MoveClassRefactoringPlugin.getString("LBL_MoveClass")).format( // NOI18N
                    new Object[] {name}
                );
            }
        }

        public Element getJavaElement() {
            if (sourceClasses == null || sourceClasses.length==0) 
                return null;
            return (Element) sourceClasses[0].refImmediateComposite();
        }

        public PositionBounds getPosition() {
            if (sourceClasses == null || sourceClasses.length==0) 
                return null;
            if (bounds == null) {
                if (sourceClasses[0].isValid()) {
                    Resource res = sourceClasses[0].getResource();
                    if (res!=null) {
                        bounds = JavaMetamodel.getManager().getElementPosition(res.getPackageIdentifier());
                    }
                }
            }
            return bounds;
        }

        public void performChange() {
            String name = target==null ? packageName : getPackageName(classPath, target);
            if (sourceClasses != null) 
                JavaModel.getResource(source.getPrimaryFile()).setPackageName(name);
            if (target==null || !target.equals(source.getPrimaryFile().getParent())) {
                JavaMetamodel.getManager().registerExtChange(this);
            }
        }
        
        private void doMove(final boolean isUndo) {    
            try {
                source.getPrimaryFile().getFileSystem().runAtomicAction(new FileSystem.AtomicAction() {
                    public void run() throws IOException {
                        if (target == null) {
                            target = createFolder(targetRoot, packageName.replace('.','/'), isUndo);
                        } else if (!target.isValid()) {
                            target = createFolder(target.getFileSystem().getRoot(), target.getPath(), isUndo);
                        }
                        
                        DataFolder folder = DataFolder.findFolder(target);
                        DataFolder sourceFolder = DataFolder.findFolder(source.getPrimaryFile().getParent());
                        oldTarget = source.getPrimaryFile().getParent();
                        source.move(folder);
                        ArrayList col = isUndo ? createdFolderNamesDuringUndo : createdFolderNames;
                        for (int i=col.size()-1; i>=0;i--) {
                            URL name = (URL) col.get(i);
                            FileObject fileToRemove = URLMapper.findFileObject(name);
                            if (fileToRemove!=null && fileToRemove.isValid() && fileToRemove.getChildren().length == 0) {
                                fileToRemove.delete();
                                col.remove(i);
                            }
                        }
                    }
                });
                JavaMetamodel.getManager().addModified(source.getPrimaryFile());
                //movedSource = source;
            } catch (IOException e) {
                System.out.println(e);
                e.printStackTrace ();
            }
        }
        
        private boolean canDelete(DataFolder df) {
            FileObject fo = df.getPrimaryFile();
            ClassPath cp = ClassPath.getClassPath(fo, ClassPath.SOURCE);
            if (cp == null)
                return false;
            return !fo.equals(cp.findOwnerRoot(fo));
        }
        
        public void performExternalChange() {
            doMove(false);
        }
        
        public void undoExternalChange() {
            if (oldTarget == null) return;
            FileObject temp = target;
            target = oldTarget;
            oldTarget = temp; 
            doMove(true);
            target = temp; 
        } 

        public FileObject getParentFile() {
            return source.getPrimaryFile();
        }
    } // MoveClass ..............................................................

    // InsertImport .............................................................
    static class InsertImport extends SimpleRefactoringElementImpl {
        
        protected ImportClass proxy;
        protected Resource source;
        protected String newName;
        protected boolean isNamed;
        private PositionBounds bounds = null;
        
        protected Import addedImport = null;
        
        public InsertImport (Resource source, String newName, boolean isNamed) {
            this.source = source;
            this.newName = newName;
            this.isNamed = isNamed;
            proxy = ((JavaModelPackage) source.refOutermostPackage ()).getImport ();
        }
        
        public String getText() {
            return getDisplayText ();
        }
    
        public String getDisplayText() {
            return new MessageFormat (MoveClassRefactoringPlugin.getString("LBL_InsertImport")).format ( // NOI18N
                new Object[] {newName}
            );
        }

        public void performChange() {
            addedImport = proxy.createImport (newName, null, false, !isNamed);
            source.addImport(addedImport);
        }

        public Element getJavaElement() {
            return source;
        }

        public PositionBounds getPosition() {
            if (bounds == null) {
                bounds = JavaMetamodel.getManager().getElementPosition(source);
            }
            return bounds;
        }

        public FileObject getParentFile() {
            return null;
        }
        
    } // InsertImport ...........................................................
    
    // RemoveImport .............................................................
    static class RemoveImport extends SimpleRefactoringElementImpl {
        
        private Import imp;
        private Resource res;
        private PositionBounds bounds = null;
        
        public RemoveImport (Import imp) {
            this.imp = imp;
            this.res = (Resource)imp.refImmediateComposite();
        }
                
        public String getText() {
            return getDisplayText ();
        }
    
        public String getDisplayText() {
            return new MessageFormat (MoveClassRefactoringPlugin.getString("LBL_RemoveImport")).format ( // NOI18N
                new Object[] {imp.getName()}
            );
        }

        public void performChange() {
            res.getImports().remove(imp);
        }

        public Element getJavaElement() {
            return res;
        }

        public PositionBounds getPosition() {
            if (bounds == null) {
                bounds = JavaMetamodel.getManager().getElementPosition (res);
            }
            return bounds;
        }

        public FileObject getParentFile() {
            return null;
        }

    } // RemoveImport ...........................................................
    
}    
