/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.refactoring.plugins;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import javax.swing.Action;
import org.netbeans.jmi.javamodel.ClassMember;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.LocalVariable;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.modules.refactoring.APIAccessor;
import org.netbeans.modules.refactoring.CheckUtils;
import org.netbeans.modules.refactoring.SafeDeleteRefactoringElement;
import org.netbeans.modules.refactoring.api.AbstractRefactoring;
import org.netbeans.modules.refactoring.api.Problem;
import org.netbeans.modules.refactoring.api.RefactoringElement;
import org.netbeans.modules.refactoring.api.RefactoringSession;
import org.netbeans.modules.refactoring.api.WhereUsedQuery;
import org.netbeans.modules.refactoring.api.SafeDeleteRefactoring;
import org.netbeans.modules.refactoring.spi.ProblemDetailsFactory;
import org.netbeans.modules.refactoring.spi.ProblemDetailsImplementation;
import org.netbeans.modules.refactoring.spi.RefactoringElementsBag;
import org.netbeans.modules.refactoring.spi.ui.RefactoringUI;
import org.netbeans.modules.refactoring.ui.RefactoringPanel;
import org.netbeans.modules.refactoring.ui.SafeDeleteUI;
import org.netbeans.modules.refactoring.ui.WhereUsedQueryUI;
import org.openide.util.Cancellable;
import org.openide.util.NbBundle;

/**
 * The plugin that carries out Safe Delete refactoring.
 * @author Bharath Ravikumar, Jan Becicka
 */
public class SafeDeleteRefactoringPlugin extends JavaRefactoringPlugin {
    private SafeDeleteRefactoring refactoring;
    private WhereUsedQuery[] whereUsedQueries;
    
    /**
     * Creates the a new instance of the Safe Delete refactoring
     * plugin.
     * @param refactoring The refactoring to be used by this plugin
     */
    public SafeDeleteRefactoringPlugin(SafeDeleteRefactoring refactoring) {
        this.refactoring = refactoring;
    }
    
    /**
     * For each element to be refactored, the corresponding
     * prepare method of the underlying WhereUsed query is
     * invoked to check for usages. If none is present, a
     * <CODE>SafeDeleteRefactoringElement</CODE> is created
     * with the corresponding element.
     * @param refactoringElements
     * @return
     */
    public Problem prepare(RefactoringElementsBag refactoringElements) {
        Collection refElementCollection = refactoringElements.getSession().getRefactoringElements();
        RefactoringSession inner = RefactoringSession.create("delete"); // NOI18N
        Set refactoredObjects = new HashSet();
        fireProgressListenerStart(AbstractRefactoring.PARAMETERS_CHECK, whereUsedQueries.length + 1);
        for(int i = 0;i < whereUsedQueries.length; ++i) {
            Object refactoredObject = whereUsedQueries[i].getRefactoredObject();
            refactoredObjects.add(refactoredObject);
//          Check whether there are any usages of a particular element
//          Doesn't make sense to check usages of a Resource.Ignore it.
            if(! (whereUsedQueries[i].getRefactoredObject() instanceof Resource) )
                whereUsedQueries[i].prepare(inner);
            
            SafeDeleteRefactoringElement element = new SafeDeleteRefactoringElement((Element) refactoredObject);
            refactoringElements.add(refactoring, element);
            fireProgressListenerStep();
        }
        for (Iterator iter = inner.getRefactoringElements().iterator(); iter.hasNext(); ) {
            Element elem = ((RefactoringElement) iter.next()).getJavaElement();
            Element comp = elem;
            boolean isOuterRef = true;
            while (!(comp instanceof Resource) && comp != null) {
                if (refactoredObjects.contains(comp)) {
                    isOuterRef = false;
                    break;
                }
                comp = (Element) comp.refImmediateComposite();
            }
            if (isOuterRef) {
                fireProgressListenerStop();
                return new Problem(false, getString("ERR_ReferencesFound"), ProblemDetailsFactory.createProblemDetails(new ProblemDetailsImplemen(new WhereUsedQueryUI(elem), inner)));
            }
        }
        fireProgressListenerStop();
        return null;
    }
    
    private class ProblemDetailsImplemen implements ProblemDetailsImplementation {
        
        private RefactoringUI ui;
        private RefactoringSession rs;
        
        public ProblemDetailsImplemen(RefactoringUI ui, RefactoringSession rs) {
            this.ui = ui;
            this.rs = rs;
        }
        
        public void showDetails(Action callback, Cancellable parent) {
            parent.cancel();
            new RefactoringPanel(ui, rs, callback).setVisible(true);
        }
        
        public String getDetailsHint() {
            return getString("LBL_ShowUsages");
        }
        
    }
    
    /**
     * Checks whether the element being refactored is a valid Method/Field/Class
     * @return Problem returns a generic problem message if the check fails
     */
    public Problem preCheck() {
        Element[] refElements = refactoring.getElementsToDelete();
        for(int i = 0;i < refElements.length; ++i) {
            Element refactoredObject = refElements[i];
            boolean validType = refactoredObject instanceof ClassMember
                    || refactoredObject instanceof LocalVariable
                    || refactoredObject instanceof Resource;
            if(!validType) {
                String errMsg = NbBundle.getMessage(SafeDeleteRefactoringPlugin.class,
                        "ERR_SafeDel_InvalidType"); // NOI18N
                return new Problem(true,errMsg);
            }
            
            if (!CheckUtils.isElementInOpenProject(refactoredObject)) {
                return new Problem(true, NbBundle.getMessage(JavaRefactoringPlugin.class, "ERR_ProjectNotOpened"));
            }
        }
        return null;
    }
    
    /**
     * A No-op for this particular refactoring.
     */
    public Problem fastCheckParameters() {
        //Nothing to be done for Safe Delete
        return null;
    }
    
    /**
     * Invokes the checkParameters of each of the underlying
     * WhereUsed refactorings and returns a Problem (if any)
     * returned by any of these queries.
     */
    public Problem checkParameters() {
        //This class expects too many details from SafeDeleteRefactoring
        //But there's no other go I guess.
        Element[] namedElements = refactoring.getElementsToDelete();
        whereUsedQueries = new WhereUsedQuery[namedElements.length];
        for(int i = 0;i <  whereUsedQueries.length; ++i) {
            whereUsedQueries[i] = APIAccessor.DEFAULT.createInternalWhereUsedQuery(namedElements[i], refactoring);
            whereUsedQueries[i].setSearchInComments(refactoring.isCheckInComments());
        }
        
        Problem problemFromUsage = null;
        for(int i = 0;i < whereUsedQueries.length; ++i) {
//          Fix for issue 63050. Doesn't make sense to check usages of a Resource.Ignore it.
            if(whereUsedQueries[i].getRefactoredObject() instanceof Resource)
                continue;
            if((problemFromUsage = whereUsedQueries[i].checkParameters()) != null)
                return problemFromUsage;
        }
        return null;
    }
    
    
    //Private Helper methods
    private static String getString(String key) {
        return NbBundle.getMessage(SafeDeleteRefactoringPlugin.class, key);
    }
    
    /**
     *Returns the default critical error message for this refactoring
     *
     */
    private Problem getProblemMessage(Object refactoredObject) {
        String errorMsg = NbBundle.getMessage(SafeDeleteUI.class,
                "DSC_SafeDelProblem", refactoredObject);// NOI18N
        return new Problem(true,errorMsg);
    }
    
}
