/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.j2ee.websphere6;

import java.io.*;
import java.util.*;
import java.lang.reflect.*;

import javax.enterprise.deploy.model.*;
import javax.enterprise.deploy.model.exceptions.*;
import javax.enterprise.deploy.shared.*;

import org.openide.*;
import org.openide.filesystems.*;
import org.netbeans.modules.j2ee.deployment.config.*;
import org.netbeans.modules.j2ee.deployment.devmodules.spi.*;

import org.netbeans.modules.j2ee.websphere6.util.WSDebug;

/**
 * Wrapper for j2eeserver's <code>DeployableObject</code> implementation.
 * This class serves as a wrapper for j2eeserver's <code>DeployableObject</code>
 * implementation that fixes its incompartibility with the JSR-88 specification.
 * The j2eeserver's implementation does not implement the 
 * <code>getEntry()</code> and <code>entries()</code> methods, while these are
 * used intesively by the servers' classes.
 * 
 * @author Kirill Sorokin
 */
public class WSDeployableObject implements DeployableObject {
    
    /**
     * The original j2eeserver's <code>DeployableObject</code> implementation,
     * all operations except <code>getEntry()</code> and <code>entries</code>
     * are delegated to it
     */
    DeployableObject deployableObject;
    
    /**
     * The module provider that we fetch from the deployable object in order to
     * implement the <code>getEntry()</code> and <code>entries</code> methods
     */
    J2eeModuleProvider provider;
    
    /**
     * Map of <code>DDBeanRoot</code>s associated with this deployable object
     */
    Map ddBeanRoots = new HashMap();
    
    /**
     * Creates a new instance of <code>WSDeployableObject</code>.
     * 
     * @param deployableObject the original j2eeserver's 
     *      <code>DeployableObject</code> implementation
     */
    public WSDeployableObject(DeployableObject deployableObject) {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "WSDeployableObject(" +         // NOI18N
                    deployableObject + ")");                           // NOI18N
        
        // save the supplied deployable object
        this.deployableObject = deployableObject;
        
        // fetch the J2eeModuleProvider handle
        try {
            Method method = deployableObject.getClass().
                    getMethod("getProvider", new Class[0]);            // NOI18N
            this.provider = (J2eeModuleProvider) method.
                    invoke(deployableObject, new Object[0]);
        } catch (IllegalAccessException e) {
            ErrorManager.getDefault().notify(ErrorManager.EXCEPTION, e);
        } catch (NoSuchMethodException e) {
            ErrorManager.getDefault().notify(ErrorManager.EXCEPTION, e);
        } catch (InvocationTargetException e) {
            ErrorManager.getDefault().notify(ErrorManager.EXCEPTION, e);
        }
    }
    
    /**
     * Finds a plugin's wrapper for the supplied j2eeserver's 
     * <code>DDBEanRoot</code> implemetation.
     * 
     * @param bean j2eeserver's implementation that is used as a key during 
     *      search
     */
    public DDBeanRoot findDDBeanRoot(DDBeanRoot bean) {
        // get all registered wrappers
        Collection values = ddBeanRoots.values();
        
        // iterate over the collection and check whether any of them is the
        // wrapper for the supplied key
        for (Iterator iterator = values.iterator(); iterator.hasNext();) {
            WSDDBeanRoot wsBean = (WSDDBeanRoot) iterator.next();
            
            // if the wrapper's origin is the key - return it
            if (wsBean.getOrigin().equals(bean)) {
                return wsBean;
            }
        }
        
        // if no wrappers found return null
        return null;
    }
    
    /**
     * Delegates the call to the j2eeserver's <code>DeployableObject</code> 
     * implementation object. Returns a wrapper for the result value.
     */
    public DDBeanRoot getDDBeanRoot(String str) throws FileNotFoundException, 
            DDBeanCreateException {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getDDBeanRoot(" + str + ")");  // NOI18N
        
        // if there is no such wrapper registered already - create a new one 
        // basing on the return value from the j2eeserver's deployable object
        if (ddBeanRoots.get(str) == null) {
            ddBeanRoots.put(str, new WSDDBeanRoot(deployableObject.
                    getDDBeanRoot(str), this));
        }
        
        // return the registered wrapper for this string
        return (DDBeanRoot) ddBeanRoots.get(str);
    }
    
    /**
     * Delegates the call to the j2eeserver's <code>DeployableObject</code> 
     * implementation object. Returns a wrapper for the result value.
     */
    public DDBeanRoot getDDBeanRoot() {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getDDBeanRoot()");             // NOI18N
        
        // if there is no wrapper for the blank string registered already - 
        // create a new one basing on the return value from the j2eeserver's 
        // deployable object
        if (ddBeanRoots.get("") == null) {                             // NOI18N
            ddBeanRoots.put("", new WSDDBeanRoot(deployableObject.     // NOI18N
                    getDDBeanRoot(), this));
        }
        
        // return the registered wrapper for the blank string
        return (DDBeanRoot) ddBeanRoots.get("");                       // NOI18N
    }

    /**
     * Delegates the call to the j2eeserver's <code>DeployableObject</code> 
     * implementation object.
     */
    public String[] getText(String str) {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getText(" + str + ")");        // NOI18N
        
        return deployableObject.getText(str);
    }
    
    /**
     * Delegates the call to the j2eeserver's <code>DeployableObject</code> 
     * implementation object.
     */
    public Class getClassFromScope(String str) {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getClassFromScope(" + str +    // NOI18N
                    ")");                                              // NOI18N
        
        return deployableObject.getClassFromScope(str);
    }
    
    /**
     * Delegates the call to the j2eeserver's <code>DeployableObject</code> 
     * implementation object.
     */
    public DDBean[] getChildBean(String str) {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getChildBean(" + str + ")");   // NOI18N
        
        return deployableObject.getChildBean(str);
    }
    
    /**
     * Delegates the call to the j2eeserver's <code>DeployableObject</code> 
     * implementation object.
     */
    public ModuleType getType() {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getType()");                   // NOI18N
        
        return deployableObject.getType();
    }
    
    /**
     * Delegates the call to the j2eeserver's <code>DeployableObject</code> 
     * implementation object.
     */
    public String getModuleDTDVersion() {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getModuleDTDVersion()");       // NOI18N
        
        return deployableObject.getDDBeanRoot().getDDBeanRootVersion();
    }

    /**
     * Returns an <code>Enumeration</code> that contains all the source files 
     * that are be included in the deployable object. While not being 
     * completely to the spec which requires the compiled files to be included
     * here, it solves the issue of the method not being implemented, as the 
     * servers are most interested in configuration files which are not affected 
     * by compilation.
     * 
     * @return an <code>Enumeartion</code> with all the source files of the 
     *      project
     */
    public Enumeration entries() {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "entries()");                   // NOI18N
        
        // construct a new WSEntries object (which handles all the entries 
        // issues) and request the Enumeration
        Enumeration entries = new WSEntries(provider).getEntries();
        
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "returning: " + entries);       // NOI18N
        
        // debug output of the entries that are to be returned
        if (WSDebug.isEnabled()) {
            while (entries.hasMoreElements()) {
                WSDebug.notify(getClass(), (String) entries.nextElement());
            }
        }
        
        // rebuild the WSEntries object and return the Enumeration it builds
        return new WSEntries(provider).getEntries();
    }
    
    /**
     * Returns an open <code>InputStream</code> to a named deployable object 
     * entry.
     * 
     * @param str the name of the entry
     * @return an <code>InputStream</code> to the named entry, null if the 
     *      entry cannot be found
     */
    public InputStream getEntry(String str) {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getEntry(" + str + ")");       // NOI18N
        
        // construct a new WSEntries object and request the entry
        InputStream in = new WSEntries(provider).getEntry(str);
        
        // return the input stream
        return in;
    }
    
    /**
     * Returns a File handle for the named deployment object entry.
     * 
     * @param str the name of the entry
     * @return a File handle for the named entry, null if the entry cannot be 
     *      found    
     */
    public File getEntryFile(String str) {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getEntryFile(" + str + ")");   // NOI18N
        
        // construct a new WSEntries object and request the entry
        File file = new WSEntries(provider).getEntryFile(str);
        
        // return the file
        return file;
    }
    
    /**
     * A class that is responsible for performing all operation on deployable 
     * object entries, i.e. returning an <code>Enumeration</code> with all the
     * files of the project and returning a named entry.
     * 
     * @author Kirill Sorokin
     */
    private static class WSEntries {
        
        /**
         * Handle for the <code>J2eeModuleProvider</code> of the parent
         * deployable object
         */
        private J2eeModuleProvider provider;
        
        /**
         * A map with all the entries for the project
         */
        private HashMap entries = new HashMap();
        
        /**
         * Creates a new instance of <code>WSEntries</code>.
         * 
         * @param provider the J2eeModuleProvider of the parent deployable
         *      object
         */
        public WSEntries (J2eeModuleProvider provider) {
            // save the provider
            this.provider = provider;
            
            // for each directory returned by the call to 
            // provider.getSourceRoots() - traverse it and store the files
            // in the map
            try {
//                if (provider.getJ2eeModule().getArchive() == null) {
                    FileObject[] roots = provider.getSourceRoots();
                    for (int i = 0; i < roots.length; i++) {
                        traverseDir(FileUtil.toFile(roots[i]), FileUtil.
                                toFile(roots[i]).getPath());
                    }
//                } else {
//                    // do nothing for now
//                }
            } catch (IOException e) {
                ErrorManager.getDefault().notify(ErrorManager.EXCEPTION, e);
            }
        }
        
        /**
         * Returns a File handle for the named entry
         * 
         * @param str the entry's name
         * @return a File handle for the entry
         */
        public File getEntryFile(String entry) {
            return new File((String) entries.get(entry));
        }
        
        /**
         * Returns an input stream to the named entry
         * 
         * @param str entry the entry name
         * @return an InputStream to the entry's file
         */
        public InputStream getEntry(String entry) {
            // get the entry path and return a new FileInputStream to it
            try {
                return new FileInputStream(new File(
                        (String) entries.get(entry)));
            } catch (FileNotFoundException e) {
                return null;
            }
        }
        
        /**
         * Returns an Enumeration with all the entries of the parent deployable 
         * object
         *
         * @return Enumeration with the entries
         */
        public Enumeration getEntries() {
            return new WSEntriesEnumeration();
        }
        
        /**
         * Traverses the supplied directory and adds all files to the map.
         * 
         * @param root the base directory path
         * @param prefix the prefix that should be stripped from the file path
         *      to form the map's key
         */
        private void traverseDir(File root, String prefix) throws IOException {
            // if the base directory is indeed a file - just add it, otherwise
            // iterate through the directory files making recursive calls
            if (!root.isDirectory()) {
                // strip the prefix from the path to form the key and use the 
                // path as the value, if the file's name does not start with
                // ".LCK" and does not end with "~" (if it does - it's a NB
                // temporary lock file)
                if (!root.getName().startsWith(".LCK") &&              // NOI18N
                        !root.getName().endsWith("~")) {               // NOI18N
                    entries.put(root.getPath().substring(prefix.length() + 1).
                            replace(File.separatorChar, '/'), root.getPath());
                }
            } else {
                File[] files = root.listFiles();
                for (int i = 0; i < files.length; i++) {
                    traverseDir(files[i], prefix);
                }
            }
        }
        
        /**
         * An implementation of the Enumeration interface that is returned by
         * WSEntries when it receives a call to getEntries() method
         */
        private class WSEntriesEnumeration implements Enumeration {
            /**
             * Internal enumeration counter, used to determine the current 
             * position
             */
            private int i = 0;
            
            /**
             * Array used as the internal representation of the Enumeration, it 
             * is build from the WSEntries' map's keys
             */
            private Object[] array = entries.keySet().toArray();
            
            /**
             * Returns the next element of the enumeration
             * 
             * @return the next element
             */
            public Object nextElement() {
                return array[i++];
            }
            
            /**
             * Tells whether the enumeration has more elements
             *
             * @return whether the enumeration has more elements
             */
            public boolean hasMoreElements() {
                return i < array.length;
            }
        }
    }
}