package org.netbeans.modules.tasklist.javadoc;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.List;
import org.openide.cookies.OpenCookie;
import org.openide.src.ClassElement;
import org.openide.src.ElementFormat;
import org.openide.src.JavaDoc;
import org.openide.src.JavaDocTag;
import org.openide.src.MemberElement;
import org.openide.src.SourceElement;
import org.openide.src.SourceException;
import org.openide.util.NbBundle;
import org.openide.util.WeakListener;
import org.netbeans.modules.tasklist.javadoc.ext.JavaTagNames;

/** 
 * Holds the element and the informations about comment errors 
 */
public abstract class AutoCommenterElement {
    protected ArrayList errorList;
    protected List resolutionList = null;

    MemberElement srcElement = null;
    int srcError = AutoCommenter.JDC_OK;

    public AutoCommenterElement( MemberElement srcElement ) {
        this.srcElement = srcElement;
        PropertyChangeListener pp = new PropertyChangeListener(){
            public void propertyChange(PropertyChangeEvent evt){
                //System.err.println(evt.getPropertyName());
                /* Instead check e.g. instanceof DataObject && getPrimaryFile().hasExt("java")
                if( evt.getSource() instanceof org.netbeans.modules.java.JavaDataObject ) //ignore
                    return;
                */
                //System.err.println("change 2");
                //AutoCommenter.this.fireAutocommentChangeEvent();            
            }
        };
        //System.err.println("adding listeners");
        srcElement.addPropertyChangeListener(WeakListener.propertyChange(pp, srcElement));
        checkError();
    }

    String getName() {
        return getNameFormat().format( srcElement );
    }

    MemberElement getSrcElement() {
        return srcElement;
    }

    int getModifiers() {
        return srcElement.getModifiers();
    }

    int getErrorNumber() {
        return srcError;
    }

    void viewSource() {
        OpenCookie oc = ((OpenCookie)srcElement.getCookie( OpenCookie.class ));
        oc.open();
    }

    ArrayList getErrorList() {
        return errorList;
    }

    List getResolutionList() {
        return resolutionList;
    }

    /** Returns the source for this particular element. Used for locking purposes.
    */
    SourceElement findSource() {
        ClassElement decl = srcElement.getDeclaringClass();
        if (decl == null && (srcElement instanceof ClassElement)) {
            decl = (ClassElement)srcElement;
        }
        if (decl == null) return null;
        return decl.getSource();
    }

    abstract String[] getNotPermittedTags();

    abstract boolean elementTagsOk();

    abstract void autoCorrect() throws SourceException;

    abstract JavaDoc getJavaDoc();

    abstract ElementFormat getNameFormat();

    abstract String typeToString();

    static boolean isPermittedTag( JavaDocTag tag, String[] notPermittedTags ) {
        String tagName = tag.name();

        for ( int i = 0; i < notPermittedTags.length; i++ ) {
            if ( tagName.equals( notPermittedTags[i] ) )
                return false;
        }

        return true;
    }

    void modifyJavaDoc(Runnable mutator) throws SourceException {
        SourceElement src = findSource();
        if (src == null) {
            mutator.run();
        } else {
            src.runAtomicAsUser(mutator);
        }
    }

    private static boolean isEmptyString( String string ) {
        return string == null || string.trim().length() <= 0;
    }

    /** Checks syntax of the tags
     */

    boolean isOkTag( JavaDocTag tag ) {
        if ( isEmptyString( tag.text() ) ) {
            errorList.add(NbBundle.getMessage(AutoCommenterElement.class, "ERR_EmptyTag", tag.name()));
            // Can't fix this one - can I?
            //resolutionList.add(NbBundle.getMessage(AutoCommenter.class, ...
            return false;
        }

        if ( tag instanceof JavaDocTag.See ) {
            int len;
            String text;
            JavaDocTag.See seetag = (JavaDocTag.See) tag;

            if ((seetag.referencedClassName() != null) || (seetag.referencedMemberName() != null)) 
                return true;
            text=tag.text();
            len = text.length();
            if (len >= 2) {
                char first=text.charAt(0);
                char last=text.charAt(len-1);

                if (first=='"' && last==first)
                    return true;
                if (first=='<' && last=='>')
                    return true;
            }
            errorList.add(NbBundle.getMessage(AutoCommenterElement.class, "ERR_InvalidTag", seetag)); // NOI18N
            resolutionList.add(NbBundle.getMessage(AutoCommenterElement.class, "FIX_InvalidTag", seetag)); // NOI18N
            return false;
        }
        else if ( tag instanceof JavaDocTag.Param ) {
            if ( isEmptyString( ((JavaDocTag.Param)tag).parameterName() ) ) {
                errorList.add( NbBundle.getMessage(AutoCommenterElement.class, "ERR_ParamNoName", tag.name()));
                return false;
            }
            if ( isEmptyString( ((JavaDocTag.Param)tag).parameterComment() ) ) {
                errorList.add( NbBundle.getMessage(AutoCommenterElement.class, "ERR_ParamNoDescr", tag.name(), ((JavaDocTag.Param)tag).parameterName()));
                return false;
            }
        }
        else if ( tag instanceof JavaDocTag.Throws ) {
            if ( isEmptyString( ((JavaDocTag.Throws)tag).exceptionName() ) ) {
                errorList.add( NbBundle.getMessage(AutoCommenterElement.class, "ERR_ThrowsNoName", tag.name()));
                return false;
            }
            if ( isEmptyString( ((JavaDocTag.Throws)tag).exceptionComment() ) ) {
                errorList.add(NbBundle.getMessage(AutoCommenterElement.class, "ERR_ThrowsNoDescr", tag.name(), ((JavaDocTag.Throws)tag).exceptionName()));
                return false;
            }
        }
        else if ( tag instanceof JavaDocTag.SerialField ) {
            if ( isEmptyString( ((JavaDocTag.SerialField)tag).fieldName() ) ) {
                errorList.add(NbBundle.getMessage(AutoCommenterElement.class, "ERR_SerialFieldNoName", tag.name()));
                return false;
            }
            if ( isEmptyString( ((JavaDocTag.SerialField)tag).fieldType() ) ) {
                errorList.add(NbBundle.getMessage(AutoCommenterElement.class, "ERR_SerialFieldNoType", tag.name(), ((JavaDocTag.SerialField)tag).fieldName()));
                return false;
            }

            if ( isEmptyString( ((JavaDocTag.SerialField)tag).description() ) ) {
                errorList.add(NbBundle.getMessage(AutoCommenter.class, "ERR_SerialFieldNoDescr", tag.name(), ((JavaDocTag.SerialField)tag).fieldName()));
                return false;
            }
        }
        return true;
    }

    boolean isMultipleTags(String tag) {
        // Check for multiple tags
        boolean error = false;
        JavaDocTag[] tags = getJavaDoc().getTags(tag);
        if ( tags.length > 1) {
            errorList.add( NbBundle.getMessage(AutoCommenterElement.class, "ERR_DuplicatedTag", tags[0].name()));
            resolutionList.add(NbBundle.getMessage(AutoCommenterElement.class, "FIX_DuplicatedTag", tags[0].name()));
            error = true;
        }
        return error;
    }

    void checkError() {

        errorList = new ArrayList();
        resolutionList = new ArrayList();

        JavaDoc jdoc = getJavaDoc();

        if ( jdoc.isEmpty() ) {
            srcError = AutoCommenter.JDC_MISSING;
            errorList.add( NbBundle.getMessage(AutoCommenterElement.class, "ERR_JavadocMissing")); //NOI18N
// XXX Can I add something here?
            return;
        }

        JavaDocTag[] tags = jdoc.getTags();
        boolean error = false;

        if ( jdoc.getText() == null || jdoc.getText().trim().length() <= 0 ) {
            errorList.add(NbBundle.getMessage(AutoCommenterElement.class, "ERR_EmptyText"));  //NOI18N
            error = true;
        }

        for ( int i = 0; i < tags.length; i ++ ) {
            if ( !AutoCommenterElement.isPermittedTag( tags[i], getNotPermittedTags() ) ) {
                errorList.add(NbBundle.getMessage(AutoCommenterElement.class, "ERR_BadTag", tags[i].name(), typeToString()));
                resolutionList.add(NbBundle.getMessage(AutoCommenterElement.class, "FIX_BadTag", tags[i].name(), typeToString()));

                error = true;
                continue;
            }

            if ( !isOkTag( tags[i] ) ) {
                error = true;
                continue;
            }
        }

        if (isMultipleTags(JavaTagNames.TAG_SINCE)) {
            error = true;
        }

        if (isMultipleTags(JavaTagNames.TAG_DEPRECATED)) {
            error = true;
        }

        if ( !elementTagsOk( ) ) {
            error = true;
        }

        if ( !error ) {
            errorList.add(NbBundle.getMessage(AutoCommenterElement.class, "ERR_JavadocOK"));  //NOI18N
        }

        srcError = error ? AutoCommenter.JDC_ERROR : AutoCommenter.JDC_OK;

    }

    boolean isCorrectable() {

        JavaDocTag[] tags = getJavaDoc().getTags();

        for ( int i = 0; i < tags.length; i ++ ) {
            if ( !AutoCommenterElement.isPermittedTag( tags[i], getNotPermittedTags() ) ) {
                return true;
            }
        }

        return false;
    }

    void autoCorrect( JavaDoc jdoc ) throws SourceException {
        JavaDocTag[] tags = jdoc.getTags();
        ArrayList correctedTags = new ArrayList( tags.length );
        String correctedText;

        correctedText = jdoc.getText();

        if ( correctedText == null ) {
            correctedText = ""; // NOI18N
        }

        for ( int i = 0; i < tags.length; i ++ ) {
            if ( !AutoCommenterElement.isPermittedTag( tags[i], getNotPermittedTags() ) ) {
                continue;
            }
            correctedTags.add( tags[i] );
        }

        //jdoc.setRawText( generateRawText( correctedText, (Collection)correctedTags ) );
        jdoc.changeTags( (JavaDocTag[])correctedTags.toArray( new JavaDocTag[ correctedTags.size() ] ), JavaDoc.SET  );
    }
}
