/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.tasklist.usertasks;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

import org.openide.awt.StatusDisplayer;
import org.openide.options.SystemOption;
import org.openide.util.MapFormat;
import org.openide.util.NbBundle;

/** 
 * Settings for the user tasks module.
 *
 * @author Tor Norbye
 * @author tl
 */
public class Settings extends SystemOption {
    /** serial uid */
    static final long serialVersionUID = 2L;

    // Option labels
    public static final String
        PROP_APPEND = "append", // NOI18N
        PROP_FILENAME = "filename", // NOI18N
        PROP_MAKE_BACKUPS = "backups", // NOI18N
        PROP_HOURS_PER_DAY = "hoursPerDay", // NOI18N
        PROP_DAYS_PER_WEEK = "daysPerWeek", // NOI18N
        PROP_COLLECT_WORK_PERIODS = "collectWorkPeriods", // NOI18N
        PROP_DETECT_INACTIVITY = "detectInactivity", // NOI18N
        PROP_AUTO_SWITCH_TO_COMPUTED = "autoSwitchToComputed"; // NOI18N
    
    /** Return the signleton cppSettings */
    public static Settings getDefault() {
	return (Settings) findObject(Settings.class, true);
    }

    /**
     * Default constructor.
     */
    public Settings() {
        setAppend(false);
        setFilename("{userdir}{/}tasklist.ics"); // NOI18N
        setBackups(false);
        setHoursPerDay(8);
        setDaysPerWeek(5);
        setCollectWorkPeriods(false);
        setDetectInactivity(false);
        setAutoSwitchToComputed(false);
    }
    
    /**
     * Get the display name.
     *
     *  @return value of OPTION_TASK_SETTINGS_NAME
     */
    public String displayName () {
	return NbBundle.getMessage(Settings.class,
            "OPTION_TASK_SETTINGS_NAME"); // NOI18N
    }

    /**
     * @return true iff the user wants backup files to be
     * created when the tasklist is saved.
     */
    public boolean getBackups() {
	// By default, true
        Boolean b = (Boolean)getProperty(PROP_MAKE_BACKUPS);
 	return (b != Boolean.FALSE);
    }

    /** 
     * Sets the makeBackups property
     * 
     * @param makeBackups True iff you want to make backups
     */
    public void setBackups(boolean makeBackups) {
	Boolean b = makeBackups ? Boolean.TRUE : Boolean.FALSE;
	putProperty(PROP_MAKE_BACKUPS, b, true);
    }

    /**
     * Getter for the autoSwitchToComputed property.
     *
     * @return true if each time a subtask (b) is added to a task (a)
     * (a) will automatically compute spent time, progress and effort.
     */
    public boolean getAutoSwitchToComputed() {
	// By default, false
        Boolean b = (Boolean)getProperty(PROP_AUTO_SWITCH_TO_COMPUTED);
 	return (b == Boolean.TRUE);
    }

    /** 
     * Sets the autoSwitchToComputed property
     * 
     * @param b true if each time a subtask (b) is added to a task (a)
     * (a) will automatically compute spent time, progress and effort.
     */
    public void setAutoSwitchToComputed(boolean b) {
	Boolean bool = b ? Boolean.TRUE : Boolean.FALSE;
	putProperty(PROP_AUTO_SWITCH_TO_COMPUTED, bool, true);
    }

    /**
     * Returns the collectWorkPeriods property.
     *
     * @return true if the work periods should be collected.
     */
    public boolean getCollectWorkPeriods() {
	// By default, true
        Boolean b = (Boolean)getProperty(PROP_COLLECT_WORK_PERIODS);
 	return b == Boolean.TRUE;
    }

    /** 
     * Sets the makeBackups property
     *
     * @param b true if the work periods should be collected
     */
    public void setCollectWorkPeriods(boolean b) {
	putProperty(PROP_COLLECT_WORK_PERIODS, Boolean.valueOf(b), true);
    }

    /**
     * Returns the detectInactivity property.
     *
     * @return true if the user inactivity should be detected
     */
    public boolean getDetectInactivity() {
        Boolean b = (Boolean) getProperty(PROP_DETECT_INACTIVITY);
 	return b == Boolean.TRUE;
    }

    /** 
     * Sets the detectInactivity property
     *
     * @param b true if the user inactivity should be detected.
     */
    public void setDetectInactivity(boolean b) {
	putProperty(PROP_DETECT_INACTIVITY, Boolean.valueOf(b), true);
    }

    /**
     * @return true iff the user wants to append items to the
     * tasklist instead of prepending.
     */
    public boolean getAppend() {
	// By default, false
        Boolean b = (Boolean)getProperty(PROP_APPEND);
 	return (b == Boolean.TRUE);
    }

    /** 
     * Indicate if the user wants to append items to the
     * tasklist instead of prepending.
     *
     * @param append True iff you want to append instead of prepend
     */
    public void setAppend(boolean append) {
	Boolean b = append ? Boolean.TRUE : Boolean.FALSE;
	putProperty(PROP_APPEND, b, true);
    }

    /** Sets the name of the file to read/write the tasklist in
    */
    public void setFilename(String fname) {
        String t = getFilename();
        if (t.equals(fname))
            return;

	if (fname.trim().length() == 0) {
	    // Use default
	    fname = NbBundle.getMessage(Settings.class,
					 "DefaultFilename"); // NOI18N
	}

        // Check that the file is valid? Should at least make sure
        // the parent dir exists
	// Try compiling the regular expression to make sure it's valid
        File f = new File(expand(fname));
        File p = f.getParentFile();
        if (!p.exists()) {
            // Print message in the message window?
            StatusDisplayer.getDefault().setStatusText(
                                 NbBundle.getMessage(Settings.class,
                                                     "NoFolder",  // NOI18N
                                                     p.getPath()));
            throw new IllegalArgumentException();
        }
        putProperty(PROP_FILENAME, fname, true);
    }

    /** 
     * Gets the name of the file to read/write the tasklist in
     */
    public String getFilename() {
        String fname = (String)getProperty(PROP_FILENAME);
        if (fname == null) {
            fname = NbBundle.getMessage(Settings.class,
                                        "DefaultFilename"); // NOI18N
        }
        return fname;
    }

    /** 
     * Gets the name of the file to read/write the tasklist in,
     * expanded such that {userdir} etc. is expanded to the real
     * filename.
     */
    public String getExpandedFilename() {
        String fname = getFilename();
        return expand(fname);
    }

    /**
     * Hours per day property
     *
     * @return value of the property
     */
    public int getHoursPerDay() {
        Integer b = (Integer) getProperty(PROP_HOURS_PER_DAY);
        if (b == null)
            return 8;
        else
            return b.intValue();
    }
    
    /**
     * Working days per week
     *
     * @return value of the property
     */
    public int getDaysPerWeek() {
        Integer b = (Integer) getProperty(PROP_DAYS_PER_WEEK);
        if (b == null)
            return 5;
        else
            return b.intValue();
    }
    
    /**
     * Sets the hours per day property
     *
     * @param h value of the property
     */
    public void setHoursPerDay(int h) {
        if (h > 0 && h <= 24)
            putProperty(PROP_HOURS_PER_DAY, new Integer(h), true);
    }
    
    /**
     * Sets the days per week property
     *
     * @param h value of the property
     */
    public void setDaysPerWeek(int h) {
        if (h > 0 && h <= 7)
            putProperty(PROP_DAYS_PER_WEEK, new Integer(h), true);
    }
    
    /** 
     * Expand a given filename using our map format 
     */
    private static String expand(String fname) {
        Map m = new HashMap(2);
        m.put("userdir", System.getProperty("netbeans.user")); // NOI18N
        m.put("/", File.separator); // NOI18N
        MapFormat f = new MapFormat(m);
        String result = f.format(fname);
        return result;
    }
}
