package org.netbeans.modules.tasklist.usertasks;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.net.URL;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;

import org.netbeans.modules.tasklist.usertasks.treetable.AdvancedTreeTableNode;
import org.netbeans.modules.tasklist.usertasks.treetable.FilterIntf;
import org.netbeans.modules.tasklist.usertasks.treetable.TreeTableNode;
import org.netbeans.modules.tasklist.core.util.ObjectList;
import org.netbeans.modules.tasklist.usertasks.model.UserTask;

/**
 * Task node
 */
public class UserTaskTreeTableNode extends AdvancedTreeTableNode {
    /**
     * Filters a task.
     *
     * @return true if the filter accepts <code>ut</code> or one of 
     * it's subtasks
     */
    public static boolean acceptsRecursively(UserTask ut, FilterIntf filter) {
        if (filter.accept(ut))
            return true;
        Iterator it = ut.getSubtasks().iterator();
        while (it.hasNext()) {
            UserTask child = (UserTask) it.next();
            if (acceptsRecursively(child, filter))
                return true;
        }
        return false;
    }
    
    protected boolean unmatched;
    protected ObjectList.Listener tl;
    protected PropertyChangeListener pcl;
    
    /** 
     * Creates a new instance of UserTaskTreeTableNode
     *  
     * @param filter filter to be used or null
     * @param ut a user task
     * @param m model for this node
     * @param comparator comparator to be used or null
     */
    public UserTaskTreeTableNode(
    TreeTableNode parent, FilterIntf filter, 
    UserTasksTreeTableModel m, UserTask ut, Comparator comparator) {
        super(m, parent, ut);
        
        this.filter = filter;
        this.comparator = comparator;
        pcl = new PropertyChangeListener() {
            public void propertyChange(PropertyChangeEvent evt) {
                fireObjectChanged();
            }
        };
        ut.addPropertyChangeListener(pcl);
        tl = new ObjectList.Listener() {
            public void listChanged(ObjectList.Event e) {
                switch (e.getType()) {
                    case ObjectList.Event.EVENT_ADDED: {
                        Object[] obj = e.getObjects();
                        for (int i = 0; i < obj.length; i++) {
                            fireChildObjectAdded(obj[i]);
                        }
                        break;
                    }
                    case ObjectList.Event.EVENT_REMOVED: {
                        Object[] obj = e.getObjects();
                        for (int i = 0; i < obj.length; i++) {
                            fireChildObjectRemoved(obj[i]);
                        }
                        break;
                    }
                    case ObjectList.Event.EVENT_REORDERED:
                        fireChildObjectsReordered();
                        break;
                    case ObjectList.Event.EVENT_STRUCTURE_CHANGED:
                        refreshChildren();
                        break;
                    default:
                        throw new InternalError("unexpected event type"); // NOI18N
                }
            }
        };
        ut.getSubtasks().addListener(tl);
    }
    
    /**
     * Returns true if the filter does not accept this node but accepts
     * some of the children
     *
     * @return true = unmatched
     */
    public boolean isUnmatched() {
        return unmatched;
    }
    
    /**
     * Sets the unmatched attribute
     *
     * @param unmatched true = the filter does not accept this node but 
     * some of it's subnodes
     */
    public void setUnmatched(boolean unmatched) {
        this.unmatched = unmatched;
    }
    
    /**
     * Returns user task associated with this node
     *
     * @return user task
     */
    public UserTask getUserTask() {
        return (UserTask) object;
    }
    
    public Object getValueAt(int column) {
        UserTask ut = getUserTask();
        switch (column) {
            case UserTasksTreeTableModel.SUMMARY:
                return ut.getSummary();
            case UserTasksTreeTableModel.PRIORITY:
                return new Integer(ut.getPriority());
            case UserTasksTreeTableModel.DONE:
                return Boolean.valueOf(ut.isDone());
            case UserTasksTreeTableModel.PERCENT_COMPLETE:
                return new Integer(ut.getPercentComplete());
            case UserTasksTreeTableModel.EFFORT:
                return ut;
            case UserTasksTreeTableModel.REMAINING_EFFORT:
                return new Integer(ut.getRemainingEffort());
            case UserTasksTreeTableModel.SPENT_TIME:
                return new Integer(ut.getSpentTime());
            case UserTasksTreeTableModel.DETAILS: 
                return ut.getDetails();
            case UserTasksTreeTableModel.FILE_BASE_NAME:
                URL url = ut.getUrl();
                if (url == null)
                    return ""; // NOI18N
                else
                    return url.toExternalForm();
            case UserTasksTreeTableModel.LINE_NUMBER:
                return new Integer(ut.getLineNumber() + 1);
            case UserTasksTreeTableModel.CATEGORY:
                return ut;
            case UserTasksTreeTableModel.CREATED:
                return new Long(ut.getCreatedDate());
            case UserTasksTreeTableModel.LAST_EDITED:
                return new Long(ut.getLastEditedDate());
            case UserTasksTreeTableModel.COMPLETED_DATE:
                return new Long(ut.getCompletedDate());
            case UserTasksTreeTableModel.DUE_DATE:
                if (ut.getDueDate() == null)
                    return null;
                else
                    return new Long(ut.getDueDate().getTime());
            case UserTasksTreeTableModel.OWNER:
                return ut;
            case UserTasksTreeTableModel.START: {
                long start = ut.getStart();
                if (start == -1)
                    return null;
                else
                    return new Date(start);
            }
            case UserTasksTreeTableModel.SPENT_TIME_TODAY:
                return new Integer(ut.getSpentTimeToday());
            default:
                return "<error>"; // NOI18N
        }
    }    
    
    public void setValueAt(Object aValue, int column) {
        UserTask ut = getUserTask();
        switch (column) {
            case UserTasksTreeTableModel.SUMMARY:
                ut.setSummary((String) aValue);
                break;
            case UserTasksTreeTableModel.DONE:
                ut.setDone(((Boolean) aValue).booleanValue());
                break;
            case UserTasksTreeTableModel.CATEGORY:
                ut.setCategory((String) aValue);
                break;
            case UserTasksTreeTableModel.PRIORITY:
                ut.setPriority(((Integer) aValue).intValue());
                break;
            case UserTasksTreeTableModel.PERCENT_COMPLETE:
                ut.setPercentComplete(((Integer) aValue).intValue());
                break;
            case UserTasksTreeTableModel.DETAILS: 
                ut.setDetails((String) aValue);
                break;
            case UserTasksTreeTableModel.OWNER:
                ut.setOwner((String) aValue);
                break;
            case UserTasksTreeTableModel.EFFORT:
                ut.setEffort(((Integer) aValue).intValue());
                break;
        }
    }

    public String toString() {
        return super.toString() + 
            "[userTask=" + getUserTask().toString() + "]"; // NOI18N
    }
    
    public void destroy() {
        super.destroy();
        getUserTask().getSubtasks().removeListener(tl);
        getUserTask().removePropertyChangeListener(pcl);
    }    
    
    public Iterator getChildrenObjectsIterator() {
        return getUserTask().getSubtasks().iterator();
    }
    
    public AdvancedTreeTableNode createChildNode(Object child) {
        UserTaskTreeTableNode n = new UserTaskTreeTableNode(
            this, filter, (UserTasksTreeTableModel) model, 
            (UserTask) child, comparator);
        if (getFilter() != null && !getFilter().accept(child))
            n.unmatched = true;
        return n;
    }    
    
    public boolean accept(Object child) {
        if (getFilter() == null)
            return true;
        
        return acceptsRecursively((UserTask) child, getFilter());
    }
    
    public boolean isCellEditable(int column) {
        if (column == UserTasksTreeTableModel.DONE)
            return !getUserTask().isProgressComputed();
        else if (column == UserTasksTreeTableModel.PERCENT_COMPLETE)
            return !getUserTask().isProgressComputed();
        else if (column == UserTasksTreeTableModel.EFFORT)
            return !getUserTask().isEffortComputed();
        else
            return column == UserTasksTreeTableModel.SUMMARY ||
                column == UserTasksTreeTableModel.CATEGORY ||
                column == UserTasksTreeTableModel.DETAILS ||
                column == UserTasksTreeTableModel.PRIORITY ||
                column == UserTasksTreeTableModel.OWNER;
    }
}
