/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.tasklist.usertasks.translators;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.SimpleTimeZone;
import java.util.TimeZone;

import javax.swing.filechooser.FileSystemView;

import org.netbeans.modules.tasklist.core.export.ExportImportFormat;
import org.netbeans.modules.tasklist.core.export.ExportImportProvider;
import org.netbeans.modules.tasklist.core.export.OpenFilePanel;
import org.netbeans.modules.tasklist.core.util.ExtensionFileFilter;
import org.netbeans.modules.tasklist.core.util.SimpleWizardPanel;
import org.netbeans.modules.tasklist.usertasks.UTUtils;
import org.netbeans.modules.tasklist.usertasks.model.UserTask;
import org.netbeans.modules.tasklist.usertasks.model.UserTaskList;
import org.netbeans.modules.tasklist.usertasks.UserTaskView;
import org.netbeans.modules.tasklist.usertasks.model.Dependency;
import org.openide.DialogDisplayer;
import org.openide.ErrorManager;
import org.openide.NotifyDescriptor;
import org.openide.WizardDescriptor;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.URLMapper;
import org.openide.text.Line;
import org.openide.util.NbBundle;

/**
 * This class provides import/export capabilities for the iCalendar calendar
 * format (used by for example KDE's Konqueror calendar/todoitem tool)
 * as specified in RFC 2445
 *
 * @author Tor Norbye
 * @author Trond Norbye
 * @author tl
 */
public class ICalReader {
    private Reader reader = null;
    private int lineno = 0;
    private int prevChar = -1;
    
    private StringBuffer nsb = new StringBuffer(400); // Name
    private StringBuffer psb = new StringBuffer(400); // Param
    private StringBuffer vsb = new StringBuffer(400); // Value
    
    /**
     * Constructor
     *
     * @param input iCal input
     */
    public ICalReader(InputStream input) throws UnsupportedEncodingException {
        this.reader = new InputStreamReader(input, "UTF-8"); // NOI18N
    }
    
    /** 
     * Read (doing all the ical unfolding) the next content line.
     * Side effects the reader object and the lineno.
     *
     * @return true if something was read. false for EOF.
     */
    private boolean processContentLine() throws IOException {
        // Reuse string buffers for improved efficiency
        nsb.setLength(0);
        psb.setLength(0);
        vsb.setLength(0);
        
        if (prevChar != -1) {
            nsb.append((char)prevChar);
        }
        
        // Read in characters, doing substitutions as necessary
        boolean escape = (prevChar == '\\');
        prevChar = -1;
        StringBuffer sb = nsb; // Processing name
        boolean processingName = true; // may not need these flags anymore, use sb
        boolean processingValue = false;
        
        while (true) {
            int ci = reader.read();
            if (ci == -1) {
                // End of stream
                return nsb.length() != 0;
            }
            char c = (char)ci;
            // See section 4.3.11 in rfc 2445
            if (escape) {
                escape = false;
                switch (c) {
                    case '\\':
                        sb.append('\\');
                        break;
                    case 'n':
                        sb.append('\n');
                        break;
                    case 'N':
                        sb.append('N');
                        break;
                    case ';':
                        sb.append(';');
                        break;
                    case ',':
                        sb.append(',');
                        break;
                    default:
                        // Error - illegal input. For now I guess
                        // we'll just pass the escape through...
                        sb.append('\\');
                        sb.append(c);
                }
            } else {
                switch (c) {
                    case '\\':
                        escape = true;
                        break;
                    case ' ':
                        if (processingName) {
                            processingName = false;
                            sb = psb;
                        } else if (processingValue) {
                            sb.append(c);
                        }
                        break;
                    case ';':
                        if (processingName) {
                            processingName = false;
                            sb = psb;
                        } else if (processingValue) {
                            sb.append(c);
                        }
                        break;
                    case ':':
                        if (processingValue) {
                            // Error in input - I've seen Korganizer do this;
                            // they're supposed to escape : but they didn't
                            sb.append(c);
                        } else {
                            sb = vsb;
                            processingValue = true;
                            processingName = false;
                        }
                        break;
                    case '\r':
                        // The spec calls for lines to be terminated with \r\n
                        // but internally we don't want \r's
                        break;
                    case '\n':
                        // New line
                        lineno++;
                        prevChar = reader.read();
                        while (prevChar == '\n') {
                            // Skip blank lines
                            prevChar = reader.read();
                            lineno++;
                        }
                        
                        // @TODO TROND: Si meg... dette stemmer vel ikke helt??
                        // jeg skal jo ogs? godta HTAB (ASCII 9!!!)
                        if (prevChar == ' ' || prevChar == '\t') {
                            // Aha! Line continuation -- we've just
                            // unfolded a line, keep processing
                            break;
                        } else { // includes case where prevChar==-1: EOF
                            // No, this is a new content line so
                            // consider ourselves done with this line
                            return true;
                        }
                    default:
                        sb.append(c);
                }
            }
        }
    }
    
    /**
     * Read an iCalendar stream, and store all of the VTODOs inside the tasklist.
     * Keep all unrecognized lines in otherItems...
     *
     * @param handler handler for iCal events
     * @throws IOException if a read error occurs
     * @throws UnknownFileFormatException if I somehow believes that this is no
     *         iCalendar format...
     * @return true if success
     */
    public void read(ICalHandler handler) throws IOException 
    {
        do {
            processContentLine();

            if (nsb.length() == 0) {
                break;
            } else if (nsb.length() == 1 && nsb.charAt(0) == '\r') { // NOI18N
                continue; // skip empty lines....
            }
            
            handler.contentLine(nsb, vsb, psb);
        } while (true);
    }
    
    public static void main(String[] params) throws Exception {
        new ICalReader(new FileInputStream("c:\\drmtasks.ics")).read(
            new ICalHandler() {
                public void contentLine(StringBuffer name, StringBuffer value, StringBuffer param) {
                    System.out.println("name=" + name + " " +
                        "value=" + value + " " + 
                        "param=" + param);
                }
            }
        );
    }
}
