/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * BuildLocalizedJars.java
 *
 * Created on October 7, 2004, 9:56 AM
 */

package org.netbeans.nbbuild;

import java.io.*;
import java.util.*;
import java.util.jar.*;
import java.util.zip.*;

import org.w3c.dom.*;
import org.xml.sax.*;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.taskdefs.*;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.types.FileSet;
import org.apache.tools.ant.DirectoryScanner;

import org.netbeans.nbbuild.LocalizedJar;

/**
 * Parses nbbuild/template/modules.xml file, walks trough list of jars and tries
 * to find localized resources in translatedfiles/src/${module.dir}/src and
 * create appropriate jars with localized resources in locale/${jarname}_${locale}.jar
 * @author  Rudolf Balada
 */
public class BuildLocalizedJars extends org.apache.tools.ant.Task {
    private File nbDir = null;
    private String topDir = "."; //NOI18N
    private String subDir = ""; //NOI18N
    private String targetDir = null;
    private File module_trackingxml = null;
    /** all module entries, indexed by jarfile */
    private Map/*<String,Entry>*/ entries;
    
    
    /**
     * Set nbdir value where NetBeans build resides to access information
     * in manifests of module jars
     */
    public void setNbDir(File nbd) throws BuildException {
        if ((! nbd.exists()) && (! nbd.isDirectory()))
            throw new BuildException("nbdir attribute value \"" + nbd + "\" points to non-existing directory \""+nbd.getAbsolutePath()+"\"", this.getLocation());
        module_trackingxml = new File(nbd,"module_tracking.xml");
        this.nbDir = nbd;
    }
    
    /**
     * Set targetdir value where to place jarfiles with localized resources
     */
    public void setTargetDir(String td) throws BuildException {
        File testTd1 = new File(td);
        File testTd2 = new File(this.getProject().getBaseDir(), td);
        if ((! testTd1.exists()) && (! testTd2.exists()))
            throw new BuildException("targetdir attribute value \"" + td + "\" points to non-existing directory", this.getLocation());
        if (testTd1.isDirectory()) {
            this.targetDir = testTd1.getAbsolutePath();
        } else if (testTd2.isDirectory()) {
            this.targetDir = testTd2.getAbsolutePath();
        } else {
            throw new BuildException("targetdir attribute value \"" + td + "\" is not a directory", this.getLocation());
        }
    }
    
    /**
     * Set topdir value where to start searching for localized resources
     * The search is done in ${topdir}/${module_dir}/${subdir}
     */
    public void setTopDir(String td) throws BuildException {
        File testTd1 = new File(td);
        File testTd2 = new File(this.getProject().getBaseDir(), td);
        if ((! testTd1.exists()) && (! testTd2.exists()))
            throw new BuildException("topdir attribute value \"" + td + "\" points to non-existing directory", this.getLocation());
        if (testTd1.isDirectory()) {
            this.topDir = testTd1.getAbsolutePath();
        } else if (testTd2.isDirectory()) {
            this.topDir = testTd2.getAbsolutePath();
        } else {
            throw new BuildException("topdir attribute value \"" + td + "\" is not a directory", this.getLocation());
        }
    }
    
    /**
     * Set subdir directory name where to look for localized resources
     * The search is done in ${topdir}/${module_dir}/${subdir}
     */
    public void setSubDir(String sd) {
        this.subDir = sd;
    }
    
    /**
     * Parse a modules.xml file.
     */
    public void parseModulesXml(File modulesXml) throws IOException, SAXException {
        Document mDoc = XMLUtil.parse(new InputSource(modulesXml.toURI().toString()),
                false, true, /*XXX*/null, null);
        this.entries = new HashMap();
        List/*<Element>*/ l = XMLUtil.findSubElements(mDoc.getDocumentElement());
        Iterator it = l.iterator();
        String modulesDir = "modules"; //NOI18N
        while (it.hasNext()) {
            Element el = (Element)it.next();
            String path = el.getAttribute("name"); //NOI18N
            String cn = el.getAttribute("codename"); //NOI18N
            String cnb = cn.lastIndexOf("/") > 0 ? cn.substring(0,cn.lastIndexOf("/")):cn; //NOI18N
            String cluster = el.getAttribute("path"); //NO18N
            Element jarEl = null;
            String jar = modulesDir + '/' + cnb.replace('.', '-') + ".jar"; //NOI18N
            this.entries.put(jar, new Entry(path, cnb, jar, cluster));
        }
    }
    
    
    /**
     * Find one entry by jar file name.
     * @param jn the desired jar file name
     * @return the matching entry or null
     */
    public Entry findByJarName(String jn) {
        return (Entry)entries.get(jn);
    }
    
    /**
     * Check inputs and do the job
     */
    public void execute() throws BuildException {
        
        if (nbDir == null) throw new BuildException("You have to set nbdir attribute to location of existing NetBeans build", this.getLocation());
        if (targetDir == null) throw new BuildException("You have to set targetdir attribute to location where to put generated localized data.", this.getLocation());
        
        
        // load module_tracking.xml entries
        try {
            parseModulesXml(module_trackingxml);
        } catch (IOException ioe) {
            throw new BuildException("I/O Error accessing file \"" + module_trackingxml.getAbsolutePath() + "\"", ioe, this.getLocation());
        } catch (SAXException se) {
            throw new BuildException("File \"" + module_trackingxml.getAbsolutePath() + "\" is not valid XML document", se, this.getLocation());
        }
        
        // add definition of locjar task if necessary
        this.getProject().addTaskDefinition("locjar", LocalizedJar.class); //NOI18N
        
        // walk through list of jar files
        Iterator jarFileIt = entries.keySet().iterator();
        while (jarFileIt.hasNext()) {
            String entryName = (String) jarFileIt.next();
            Entry jarEntry = (Entry) entries.get(entryName);
            // filename with module jar; we'll search it in nbdir
            String jarFileName = jarEntry.getJar();
            // module's path in CVS ( nb_all/${jarDir} as well as nb_all/translatedfiles/src/${jarDir} )
            String jarDir = jarEntry.getPath();
            String jarCluster = jarEntry.getCluster();
            
            // check whether directory with localized resource exist
            File resRootDir = new File(topDir + File.separator + jarDir, subDir);
            if (!resRootDir.exists()) {
                // there's no directory with localized data for actual jar file
                log("Directory "+resRootDir.getAbsolutePath()+" with localized resources for module jar file " +
                        jarFileName+" does not exist",Project.MSG_WARN);
                continue;
            }
            
            String modulejar = jarCluster + File.separator + jarFileName;
            String jarCnb = jarEntry.getCnb();
            // load OpenIDE-Module attribute from module jar's manifest and check
            // consistency of OpenIDE-Module tag with code name base
            java.util.jar.Attributes attr = null;
            if (!(new File(nbDir + File.separator + modulejar).exists())) {
                log("Module jar for module " + jarCnb + " from cluster " + jarCluster + " doesn't exist - skipping.", Project.MSG_WARN);
                continue; //with next module
            }
            try {
                JarFile modulejarfile = new JarFile(nbDir + File.separator + modulejar);
                try {
                    attr = modulejarfile.getManifest().getMainAttributes();
                    String oiModule = attr.getValue("OpenIDE-Module"); //NOI18N
                    if  (oiModule == null)
                        throw new BuildException("Jar file " + modulejar +
                                " doesn't have OpenIDE-Module tag in manifest an though is not regular NetBeans module jar.", this.getLocation());
                    if  (!(oiModule.startsWith(jarCnb)))
                        throw new BuildException("Module jar manifest's tag OpenIDE-Module value \"" + oiModule +
                                "\" does not start with code name base " + jarCnb +
                                ". There's probably something wrong with data in file " + module_trackingxml.getAbsolutePath() +
                                ". Check data consistency between module jar build and module_tracking.xml.", this.getLocation());
                } finally {
                    modulejarfile.close();
                }
            } catch (IOException ioe) {
                throw new BuildException("I/O exception while reading module jar file " + modulejar, ioe, this.getLocation());
            }
            
            
            // all checks done, let's call LocalizedJar
            LocalizedJar locjar;
            locjar = (org.netbeans.nbbuild.LocalizedJar) getProject().createTask("locjar"); //NOI18N
            locjar.setBasedir(new File(topDir, jarDir + File.separator + subDir ) );
            locjar.setJarfile(new File(targetDir,modulejar));
            locjar.setPreserveModuleJar(true);
            locjar.execute();
            
        }
    }
    
    
    /**
     * One entry in the file.
     */
    public final class Entry {
        
        private final String path;
        private final String cnb;
        private final String jar;
        private final String cluster;
        
        Entry(String path, String cnb, String jar, String cluster) {
            this.path = path;
            this.cnb = cnb;
            this.jar = jar;
            this.cluster = cluster;
        }
        
        public String getCluster() {
            return cluster;
        }
        
        /**
         * Get the path in nb_all, e.g. ant/grammar.
         */
        public String getPath() {
            return path;
        }
        
        /**
         * Get the code name base, e.g. org.netbeans.modules.ant.grammar.
         */
        public String getCnb() {
            return cnb;
        }
        
        /**
         * Get the JAR file path, e.g. modules/org-netbeans-modules-ant-grammar.jar.
         * If the jar file name and logical path was not specified in original entry,
         * then it tries to return jarfile with path from nb.modules.dir property followed
         * by jar filename built of code name base with dots replaced by dashes.
         * In case the property nb.modules.dir is not set default "modules/" path is used.
         */
        public String getJar() {
            if (jar != null) {
                return jar;
            } else {
                // Default uses just cnb.
                log(this.getClass().getName() + " - the process should never get here if the method parseModulesXml() worked correctly!", Project.MSG_WARN);
                String modulesDir = System.getProperty("nb.modules.dir"); //NOI18N
                if ((!(modulesDir == null)) && (!(modulesDir.equals("")))) { //NOI18N
                    return modulesDir + '/' + cnb.replace('.', '-') + ".jar";  //NOI18N
                } else {
                    // fallback solution
                    return "modules/" + cnb.replace('.', '-') + ".jar"; //NOI18N
                }
                
            }
        }
        
    }
    
}
