/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.search.project;

import java.awt.Component;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.lang.ref.Reference;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ProjectUtils;
import org.netbeans.api.project.SourceGroup;
import org.netbeans.api.project.Sources;
import org.netbeans.api.project.ui.OpenProjects;
import org.netbeans.modules.search.SearchPanel;
import org.netbeans.modules.search.SearchPerformer;
import org.openide.filesystems.FileObject;
import org.openide.nodes.AbstractNode;
import org.openide.nodes.Children;

import org.openide.nodes.Node;
import org.openide.util.Mutex;
import org.openide.util.actions.CallableSystemAction;
import org.openide.util.HelpCtx;
import org.openide.util.NbBundle;
import org.openide.util.SharedClassObject;
import org.openide.util.lookup.Lookups;
import org.openidex.search.FileObjectFilter;
import org.openidex.search.SearchInfo;
import org.openidex.search.SearchInfoFactory;


/**
 * Action which searches visible and sharable files of all open projects.
 * <p>
 * This action uses two different mechanisms of enabling/disabling,
 * depending on whether the action is available in the toolbar or not:
 * <ul>
 *     <li><u>if the action is in the toolbar:</u><br />
 *         The action is updated (enabled/disabled) continuously,
 *         i.e. whenever some project is open or closed.
 *         </li>
 *     <li><u>if the action is <em>not</em> in the toolbar</u><br />
 *         The action state is not updated but it is computed on demand,
 *         i.e. when method <code>isEnabled()</code> is called.
 *         </li>
 * </ul>
 * Moreover, the first call of method <code>isEnabled()</code> returns
 * <code>false</code>, no matter whether some projects are open or not.
 * This is made so based on the assumption that the first call of
 * <code>isEnabled()</code> is done during IDE startup as a part of menu
 * creation. It reduces startup time as it does not force projects
 * initialization.
 *
 * @author  Petr Kuzel
 * @author  Marian Petras
 */
public final class ProjectsSearchAction extends CallableSystemAction
                                        implements PropertyChangeListener {

    static final long serialVersionUID = 4554342565076372610L;
    
    /**
     * name of a shared variable - is this the first call of method
     * <code>isEnabled()</code>?
     * Value of this variable is non-<code>null</code> only until method
     * {@link #isEnabled()} is called for the first time.
     */
    private static final String VAR_FIRST_ISENABLED
                                = "first call of isEnabled()";          //NOI18N
    /**
     * name of a shared variable - reference to the toolbar presenter
     */
    private static final String VAR_TOOLBAR_COMP_REF
                                = "toolbar presenter ref";              //NOI18N
    /**
     * name of a shared variable - are we listening on the set of open projects?
     * It contains <code>Boolean.TRUE</code> if we are listening,
     * and <code>null</code> if we are not listening.
     */
    private static final String VAR_LISTENING
                                = "listening";                          //NOI18N
    
    /**
     */
    protected void initialize() {
        super.initialize();
        putProperty(VAR_FIRST_ISENABLED, Boolean.TRUE);
    }

    /**
     */
    public Component getToolbarPresenter() {
        synchronized (getLock()) {
            Component presenter = getStoredToolbarPresenter();
            if (putProperty(VAR_LISTENING, Boolean.TRUE) == null) {
                OpenProjects.getDefault().addPropertyChangeListener(this);
                putProperty(VAR_FIRST_ISENABLED, null);
                updateState();
            }
            return presenter;
        }
    }

    /**
     * Returns a toolbar presenter.
     * If the toolbar presenter already exists, returns the existing instance.
     * If it does not exist, creates a new toolbar presenter, stores
     * a reference to it to shared variable <code>VAR_TOOLBAR_BTN_REF</code>
     * and returns the presenter.
     *
     * @return  existing presenter; or a new presenter if it did not exist
     */
    private Component getStoredToolbarPresenter() {
        Object refObj = getProperty(VAR_TOOLBAR_COMP_REF);
        if (refObj != null) {
            Reference ref = (Reference) refObj;
            Object presenterObj = ref.get();
            if (presenterObj != null) {
                return (Component) presenterObj;
            }
        }
        
        Component presenter = super.getToolbarPresenter();
        putProperty(VAR_TOOLBAR_COMP_REF, new WeakReference(presenter));
        return presenter;
    }
    
    /**
     * Checks whether the stored toolbar presenter exists but does not create
     * one if it does not exist.
     *
     * @return  <code>true</code> if the reference to the toolbar presenter
     *          is not <code>null</code> and has not been cleared yet;
     *          <code>false</code> otherwise
     * @see  #getStoredToolbarPresenter
     */
    private boolean checkToolbarPresenterExists() {
        Object refObj = getProperty(VAR_TOOLBAR_COMP_REF);
        if (refObj == null) {
            return false;
        }
        return ((Reference) refObj).get() != null;
    }
    
    /**
     * This method is called if we are listening for changes on the set
     * of open projecst and some project(s) is opened/closed.
     */
    public void propertyChange(PropertyChangeEvent e) {
        synchronized (getLock()) {
            
            /*
             * Check whether listening on open projects is active.
             * This block of code may be called even if listening is off.
             * It can happen if this method's synchronized block contended
             * for the lock with another thread which just switched listening
             * off.
             */
            if (getProperty(VAR_LISTENING) == null) {
                return;
            }
            
            if (checkToolbarPresenterExists()) {
                updateState();
            } else {
                OpenProjects.getDefault().removePropertyChangeListener(this);
                putProperty(VAR_LISTENING, null);
                putProperty(VAR_TOOLBAR_COMP_REF, null);
            }
        }
        
    }

    /**
     */
    public boolean isEnabled() {
        synchronized (getLock()) {
            if (getProperty(VAR_LISTENING) != null) {
                return super.isEnabled();
            } else if (getProperty(VAR_FIRST_ISENABLED) == null) {
                return OpenProjects.getDefault().getOpenProjects().length != 0;
            } else {
                
                /* first call of this method */
                putProperty(VAR_FIRST_ISENABLED, null);
                return false;
            }
        }
    }
    
    /**
     */
    private synchronized void updateState() {
        
        /*
         * no extra synchronization needed - the method is called
         * only from synchronized blocks of the following methods:
         *    propertyChange(...)
         *    getToolbarPresenter()
         */
        
        final boolean enabled
                = OpenProjects.getDefault().getOpenProjects().length != 0;
        Mutex.EVENT.writeAccess(new Runnable() {
            public void run() {
                setEnabled(enabled);
            }
        });
    }

    protected String iconResource() {
        return "org/openide/resources/actions/find.gif";                //NOI18N
    }
    
    public String getName() {
        return NbBundle.getMessage(ProjectsSearchAction.class,
                                   "LBL_SearchProjects");               //NOI18N
    }

    public HelpCtx getHelpCtx() {
        return new HelpCtx(ProjectsSearchAction.class);
    }

    /** Where to search */
    Node[] getNodesToSearch() {
        Project[] openProjects = OpenProjects.getDefault().getOpenProjects();
        
        if (openProjects.length == 0) {
            
            /*
             * We cannot prevent this situation. The action may be invoked
             * between moment the last project had been removed and the removal
             * notice was distributed to the open projects listener (and this
             * action disabled). This may happen if the the last project
             * is being removed in another thread than this action was
             * invoked from.
             */
            return new Node[0];
        }
        
        List searchInfos = new ArrayList();
        List rootFolders = new ArrayList();
        for (int i = 0; i < openProjects.length; i++) {
            Object prjSearchInfo = openProjects[i].getLookup().lookup(SearchInfo.class);
            if (prjSearchInfo != null) {
                searchInfos.add((SearchInfo) prjSearchInfo);
                continue;
            }
            
            Sources sources = ProjectUtils.getSources(openProjects[i]);
            SourceGroup[] sourceGroups
                    = sources.getSourceGroups(Sources.TYPE_GENERIC);
            for (int j = 0; j < sourceGroups.length; j++) {
                rootFolders.add(sourceGroups[j].getRootFolder());
            }
        }

        int nodesCount = 0;
        if (!searchInfos.isEmpty()) {
            nodesCount += searchInfos.size();
        }
        if (!rootFolders.isEmpty()) {
            nodesCount++;
        }
        
        if (nodesCount == 0) {
            return new Node[0];
        }
        
        Node[] searchNodes = new Node[nodesCount];
        int nodeIndex = 0;
        if (!searchInfos.isEmpty()) {
            for (Iterator it = searchInfos.iterator(); it.hasNext(); ) {
                searchNodes[nodeIndex++] = new AbstractNode(
                                                Children.LEAF,
                                                Lookups.singleton((SearchInfo) it.next()));
            }
        }
        if (!rootFolders.isEmpty()) {
            SearchInfo searchInfo = SearchInfoFactory.createSearchInfo(
                    (FileObject[])
                            rootFolders.toArray(new FileObject[rootFolders.size()]),
                    true,           //recursive
                    new FileObjectFilter[] {SearchInfoFactory.VISIBILITY_FILTER,
                                            SearchInfoFactory.SHARABILITY_FILTER});
            searchNodes[nodeIndex++] = new AbstractNode(
                                                Children.LEAF,
                                                Lookups.singleton(searchInfo));
        }

        searchNodes[0].setValue(
                SearchPanel.PROP_DIALOG_TITLE,
                NbBundle.getMessage(ProjectsSearchAction.class,
                                    "LBL_Title_SearchProjects"));       //NOI18N
        return searchNodes;
    }

    /** Perform this action. */
    public void performAction() {
        SearchPerformer performer
                = ((SearchPerformer) SharedClassObject.findObject(
                        SearchPerformer.class, true));
        performer.performAction(getNodesToSearch());
    }
    
    /**
     */
    protected boolean asynchronous() {
        return false;
    }

}
