/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.search.project;

import java.awt.EventQueue;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.Iterator;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ProjectManager;
import org.netbeans.api.project.ui.OpenProjects;
import org.netbeans.junit.NbTestCase;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.nodes.Node;
import org.openide.util.actions.SystemAction;
import org.openidex.search.SearchInfo;

/**
 *
 * @author  Marian Petras
 * @author  Jaroslav Tulach
 */
public class ProjectsSearchActionTest extends NbTestCase {
    
    static final String PRJ_WITH_SEARCH_INFO = "prjWithSearchInfo";
    static final String PRJ_WITHOUT_SEARCH_INFO = "prjWithoutSearchInfo";
    
    private WorkDirCleaner workDirCleaner;
    private File prjRootDir;
    private FileObject prjRootFileObj;
    
    public ProjectsSearchActionTest(String testName) {
        super(testName);
    }
    
    protected boolean runInEQ() {
        return true;
    }
    
    /**
     * Mostly copied from <code>OpenProjectCLITest.java</code>.
     * 
     * @author  Jaroslav Tulach
     * @author  Marian Petras
     */
    protected void setUp() throws Exception {
        workDirCleaner = null;
        prjRootDir = null;
        prjRootFileObj = null;
        
        OpenProjects openProjectsImpl = OpenProjects.getDefault();
        final Project[] openProjects = openProjectsImpl.getOpenProjects();
        if (openProjects.length == 0) {
            clearWorkDir();
        } else {
            openProjectsImpl.close(openProjects);
            workDirCleaner = new WorkDirCleaner();
            EventQueue.invokeLater(workDirCleaner);
        }
    }
    
    final class WorkDirCleaner implements Runnable {
        private IOException ioExc;
        public void run() {
            try {
                clearWorkDir();
            } catch (IOException ex) {
                this.ioExc = ex;
            }
        }
    }
    
    /**
     * Tests that when action <em>Find in Projects</em> is invoked and there is
     * a project with {@code SearchInfo} in its {@code Lookup}, that
     * {@code SearchInfo} is used by the action.
     */
    public void testProjectWithSearchInfo() throws IOException {
        checkWorkDirCleaner();
        
        openProject(PRJ_WITH_SEARCH_INFO);
        
        Collection searchInfos = getSearchInfoToBeUsed();
        assertTrue("Single SearchInfo instance expected", searchInfos.size() == 1);
        assertTrue("Unexpected SearchInfo used by ProjectsSearchAction",
                   searchInfos.iterator().next().getClass() == MockSearchInfo.class);
    }
    
    /**
     * Tests that when action <em>Find in Projects</em> is invoked, there is
     * an project opened and the project does not contain {@code SearchInfo}
     * in its {@code Lookup}, a default {@code SearchInfo} is used.
     */
    public void testProjectWithoutSearchInfo() throws IOException {
        checkWorkDirCleaner();
        
        openProject(PRJ_WITHOUT_SEARCH_INFO);
        
        Collection searchInfos = getSearchInfoToBeUsed();
        assertFalse("No SearchInfo used by ProjectsSearchAction", searchInfos.isEmpty());
        
        for (Iterator it = searchInfos.iterator(); it.hasNext(); ) {
            String searchInfoClassName = it.next().getClass().getName();
            if (!searchInfoClassName.startsWith("org.openidex.search.")) {
                fail("Unexpected SearchInfo used by ProjectsSearchAction: " + searchInfoClassName);
            }
        }
    }
    
    /**
     */
    private void checkWorkDirCleaner() {
        if ((workDirCleaner != null) && (workDirCleaner.ioExc != null)) {
            fail("WorkDirCleaner thrown an exception: " + workDirCleaner.ioExc);
        }
    }
    
    /**
     */
    private Collection getSearchInfoToBeUsed() {
        ProjectsSearchAction prjSearchAction = (ProjectsSearchAction) SystemAction.get(ProjectsSearchAction.class);
        assertNotNull("ProjectsSearchAction not found", prjSearchAction);
        
        prjSearchAction.isEnabled();    //the first call of 'isEnabled()' always returns 'false'
        boolean isEnabled = prjSearchAction.isEnabled();
        assertTrue("ProjectsSearchAction should be enabled", isEnabled);
        
        Node[] nodesToSearch = prjSearchAction.getNodesToSearch();
        assertTrue("No nodes to search", (nodesToSearch != null) && (nodesToSearch.length != 0));
        
        Collection result = new ArrayList(nodesToSearch.length);
        for (int i = 0; i < nodesToSearch.length; i++) {
            Node node = nodesToSearch[i];
            Object searchInfo = node.getLookup().lookup(SearchInfo.class);
            if (searchInfo != null) {
                result.add(searchInfo);
            }
        }
        return result;
    }
    
    /**
     * 
     */
    private void openProject(String prjDirName) throws IOException {
        assert EventQueue.isDispatchThread();
        
        prjRootDir = new File(getWorkDir(), prjDirName);
        prjRootDir.mkdirs();
        
        new File(prjRootDir, "nbproject").mkdirs();
        
        prjRootFileObj = FileUtil.toFileObject(prjRootDir);
        assertTrue(prjRootDir.getParent() + " should be recognized as a project directory",
                   ProjectManager.getDefault().isProject(prjRootFileObj));
        
        Project project = null;
        try {
            project = ProjectManager.getDefault().findProject(prjRootFileObj);
        } catch (IOException ex) {
            fail("Could not load project " + prjRootFileObj);
        }
        assertNotNull("findProject(...) returned null", project);
        
        OpenProjects.getDefault().open(new Project[] { project }, false);
        Project[] openProjects = OpenProjects.getDefault().getOpenProjects();
        assertEquals("Exactly one open project should be open", 1, openProjects.length);
        assertEquals("Another project is open", openProjects[0].getProjectDirectory(), prjRootFileObj);
    }

    /**
     *
     * @author  Marian Petras
     */
    static final class MockSearchInfo implements SearchInfo {
        
        public boolean canSearch() {
            return true;
        }

        public Iterator objectsToSearch() {
            return Collections.emptyList().iterator();
        }
}
    
}