/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.search;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import org.openide.options.SystemOption;
import org.openide.util.NbBundle;
import org.openide.util.HelpCtx;
import org.openide.util.Lookup;
import org.openide.util.SharedClassObject;
import org.openidex.search.SearchType;

/**
 * <!-- PENDING -->
 *
 * @author Marian Petras
 */
public class SearchProjectSettings extends SystemOption {

    // static .....................................................................................

    private static final long serialVersionUID = 6955446757377175182L;

    /**
     * Name of property &quot;search criteria&quot;.
     *
     * @see  #getCriteria
     * @see  #setCriteria
     */
    public static final String PROP_CRITERIA = "search criteria";       //NOI18N
    
    /** [PENDING] */
    private static SearchCriterion[] searchCriteria;
    
    // SystemOption implementation ..................................................................

    /**
    * Returns name of these settings.
    */
    public String displayName() {
        return NbBundle.getBundle(SearchProjectSettings.class)
               .getString("TEXT_Search_settings");                     //NOI18N
    }

    public HelpCtx getHelpCtx () {
        return new HelpCtx(SearchProjectSettings.class);
    }

    // properties .................................................................................

    /** */
    private void importOldSettings() {
        
        /* import old search criteria: */
        List oldCriteria = new ArrayList();
        Collection instances = Lookup.getDefault()
                               .lookup(new Lookup.Template(SearchType.class))
                               .allInstances();
        for (java.util.Iterator i = instances.iterator(); i.hasNext(); ) {
            SearchType instance = (SearchType) i.next();
            if (instance.isValid()) {
                try {
                    oldCriteria.add(new SearchCriterion(instance));
                } catch (java.io.IOException ex) {
                    org.openide.ErrorManager.getDefault().notify(
                            org.openide.ErrorManager.EXCEPTION,
                            ex);
                }
            }
        }
        if (oldCriteria.isEmpty()) {
            searchCriteria = new SearchCriterion[0];
        } else {
            searchCriteria = new SearchCriterion[oldCriteria.size()];
            oldCriteria.toArray(searchCriteria);
        }
    }

    /**
     * <!-- PENDING -->
     *
     * @see  #setCriteria
     * @see  #PROP_CRITERIA
     */
    public SearchCriterion[] getSearchCriteria() {
        if (searchCriteria == null) {
            importOldSettings();
        }
        return searchCriteria;
    }

    /**
     * <!-- PENDING -->
     *
     * @see  #getCriteria
     * @see  #PROP_CRITERIA
     */
    public void setSearchCriteria(SearchCriterion[] criteria) {
        if (criteria == null) {
            criteria = new SearchCriterion[0];
        }
        SearchCriterion[] old = searchCriteria;
        searchCriteria = criteria;
        firePropertyChange(PROP_CRITERIA, old, searchCriteria);
    }
    
    /** */
    static final SearchProjectSettings getInstance() {
        return (SearchProjectSettings)
               SharedClassObject.findObject(SearchProjectSettings.class, true);
    }
    
    /**
     * Adds new saved search criterion.
     *
     * @param  c  search criterion to be added
     */
    void addSearchCriterion(SearchCriterion c) {
        if (searchCriteria == null) {
            searchCriteria = new SearchCriterion[1];
            searchCriteria[0] = c;
            firePropertyChange(PROP_CRITERIA, null, searchCriteria);
        } else {
            SearchCriterion[] old = searchCriteria;
            searchCriteria = new SearchCriterion[old.length + 1];
            System.arraycopy(old, 0, searchCriteria, 0, old.length);
            searchCriteria[old.length] = c;
            firePropertyChange(PROP_CRITERIA, old, searchCriteria);
        }
    }

    /**
     * Finds and replaces an existing search criterion with another one.
     * Search criterion to be replaced is found by fields
     * {@link #name} and {@link #searchTypeClassName}.
     * If no matching search criterion is found, no change is done.
     *
     * @param  name  name of a search criterion to search for
     * @param  className  class name of a search criterion to search for
     * @param  c  search criterion to replace the found criterion by
     * @return  <code>true</code> if a search criterion was found and replaced
     *          (including the case that the criterion passed as the argument
     *          was found);
     *          <code>false</code> otherwise
     */
    boolean replaceSearchCriterion(String name,
                                   String className,
                                   SearchCriterion c) {
        if (searchCriteria == null) {
            return false;
        }
        for (int i = 0; i < searchCriteria.length; i++) {
            SearchCriterion candidate = searchCriteria[i];
            if (candidate.name.equals(name)
                    && candidate.searchTypeClassName.equals(className)) {
                searchCriteria[i] = c;
                markSearchCriteriaChanged();
                return true;
            }
        }
        return false;
    }
    
    /** */
    void markSearchCriteriaChanged() {
        firePropertyChange(PROP_CRITERIA, null, null);
    }
    
}
