/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.search.types;


import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.loaders.DataObjectNotFoundException;
import org.openide.nodes.Node;
import org.openidex.search.SearchType;

import org.openide.ErrorManager;
import org.openide.util.Lookup;
import org.openidex.search.SearchInfo;


/**
 * Type used by SearchType which can be part of <code>DataObjectSearchGroup</code>.
 *
 * @author  Petr Kuzel
 * @see org.openidex.search.DataObjectSearchGroup
 */
public abstract class DataObjectType extends SearchType {

    private static final long serialVersionUID = 1L;
    //private static final long serialVersionUID = 4048358224805855612L;

    /***** LIVE UPDATE: ****** /
    / ** Property change listener. * /
    private transient PropertyChangeListener propListener;
    / *************************/
    
    
    /** Prepares search object for search. Listens on the underlying 
     * object and fires SearchType.PROP_OBJECT_CHANGED property change
     * in cases object has changed. */
    protected void prepareSearchObject(Object searchObject) {
        
        /***** LIVE UPDATE: ****** /
        DataObject dataObject = extractDataObject(searchObject);

        if (dataObject == null) {
            return;
        }
        dataObject.addPropertyChangeListener(
            WeakListeners.propertyChange(getDataObjectListener(), dataObject)
        );
        / *************************/
    }

    /***** LIVE UPDATE: ****** /
    / ** Gets property change listener which listens on changes on searched data object. * /
    private synchronized PropertyChangeListener getDataObjectListener() {
        if (propListener == null) {
            propListener = new PropertyChangeListener() {
                public void propertyChange(PropertyChangeEvent evt) {
                    if (DataObject.PROP_COOKIE.equals(evt.getPropertyName())) {
                        firePropertyChange(PROP_OBJECT_CHANGED, null, evt.getSource());
                    }
                }
            };
        }
        
        return propListener;
    }
    / *************************/
    
    /**
     */
    public boolean testObject(Object object) {
        DataObject dataObject = extractDataObject(object);
        return dataObject != null && testDataObject(dataObject);
    }

    /** Gets data object from search object. */
    private static DataObject extractDataObject(Object object) {
        DataObject dataObject = null;
        
        if (object instanceof DataObject) {
            dataObject = (DataObject) object;
        } else if (object instanceof FileObject) {
            try {
                dataObject = DataObject.find((FileObject) object);
            } catch (DataObjectNotFoundException dnfe) {
                ErrorManager.getDefault().notify(ErrorManager.EXCEPTION, dnfe);
            }
        }

        return dataObject;
    }

    /** Creates search types classes. */
    protected Class[] createSearchTypeClasses() {
        return new Class[] {DataObject.class};
    }
    
    /**
     * Checks whether a data object matches the criteria defined in this search
     * type.
     *
     * @param  dataObject  object to be tested
     * @return  <code>true</code> if the object matches the criteria,
     *          <code>false</code> it it does not
     */
    protected abstract boolean testDataObject(DataObject dataObject);

    /**
     * {@inheritDoc}
     *
     * @return  <code>true</code> if any of the specified nodes is a folder,
     *          <code>false</code> otherwise
     */
    public boolean enabled(Node[] nodes) {
        
        if (nodes == null || nodes.length == 0) {
            return false;
        }

        // NodeCookie test
        for (int i = 0; i < nodes.length; i++ ) {
            Lookup nodeLookup = nodes[i].getLookup();

            /*
             * All DataObject containers are searchable.
             * Even if they have SearchInfo and their canSearch() returns false.
             */
            if (nodeLookup.lookup(DataObject.Container.class) != null) {
                return true;
            }

            /*
             * We must have checked that canSearch() returns true.
             *
             * This method must ensure that if it returns true,
             * all the passed nodes are searchable. Only nodes whose
             * SearchInfo.canSearch() return true are guaranteed to be
             * searchable.
             */
            Object searchInfo = nodeLookup.lookup(SearchInfo.class);
            if (searchInfo != null && ((SearchInfo) searchInfo).canSearch()) {
                return true;
            }
        }

        return false;
    }

}
