/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */


package org.netbeans.modules.search.types;


import java.io.*;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;

import org.netbeans.modules.search.SearchDisplayer;

import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.nodes.AbstractNode;
import org.openide.nodes.Children;
import org.openide.nodes.Node;
import org.openide.util.actions.NodeAction;
import org.openide.util.actions.SystemAction;
import org.openide.util.HelpCtx;
import org.openide.util.NbBundle;
import org.openide.windows.OutputEvent;
import org.openide.windows.OutputListener;
import org.openide.xml.XMLUtil;
import org.openidex.search.SearchPattern;


/**
 * Test DataObject primaryFile for line full-text match.
 *
 * @author  Petr Kuzel
 * @author  Marian Petras
 */
public class FullTextType extends TextType {

    private static final long serialVersionUID = 1L;
    //private static final long serialVersionUID = -8671182156199506714L;
    
    /** array of searchable application/x-<em>suffix</em> MIME-type suffixes */
    private static final Collection searchableXMimeTypes;
    
    static {
        searchableXMimeTypes = new HashSet(17);
        searchableXMimeTypes.add("csh");                                //NOI18N
        searchableXMimeTypes.add("httpd-php");                          //NOI18N
        searchableXMimeTypes.add("httpd-php-source");                   //NOI18N
        searchableXMimeTypes.add("javascript");                         //NOI18N
        searchableXMimeTypes.add("latex");                              //NOI18N
        searchableXMimeTypes.add("php");                                //NOI18N
        searchableXMimeTypes.add("sh");                                 //NOI18N
        searchableXMimeTypes.add("tcl");                                //NOI18N
        searchableXMimeTypes.add("tex");                                //NOI18N
        searchableXMimeTypes.add("texinfo");                            //NOI18N
        searchableXMimeTypes.add("troff");                              //NOI18N
    }

    /**
     * Holds information about occurences of matching strings within individual
     * <code>DataObject</code>s.
     * <p>
     * Contains mappings
     * <blockquote>
     * (<code>DataObject</code>) -&gt; (list of occurences of matchings strings
     * withing the <code>DataObject</code>)
     * </blockquote>
     * </p>
     */
    private transient Map detailsMap;

    /**
     * Maximum number of matches reported for one line.
     * It eliminates huge memory consumption for single letter searches on long lines.
     */
    private static final int MAX_REPORTED_OCCURENCES_ON_LINE = 5;

    /**
     * Maximum number of matches reported for one file.
     * It eliminates huge memory consumption for single letter searches in long files.
     */
    private static final int MAX_REPORTED_OCCURENCES_IN_FILE = 200;


    /** */
    public Object clone() {
        FullTextType clone = (FullTextType) super.clone();
        clone.detailsMap = new HashMap(20);
        return clone;
    }
    
    /** */
    public void destroy() {
        if (detailsMap != null) {
            detailsMap.clear();
        }
    }
    
    /**
     */
    protected String displayName() {
        /*
         * For all non-default search types, display name is taken from
         * the search type's bean descriptor. But this search type has
         * no bean descriptor so we must override this method.
         */
        
        return NbBundle.getMessage(FullTextType.class,
                                   "TEXT_FULLTEXT_CRITERION");          //NOI18N
    }

    /** Gets details map. */
    private Map getDetailsMap() {
        if (detailsMap != null) {
            return detailsMap;
        }
        
        synchronized(this) {
            if (detailsMap == null) {
                detailsMap = new HashMap(20);
            }
        }
        
        return detailsMap;
    }
    
    /**
     */
    public boolean testDataObject(DataObject dobj) {
        InputStream is = null;
        LineNumberReader reader = null;
	SearchPattern searchPattern = createSearchPattern();
        try {
            String line = ""; //NOI18N

            // primary file of the DataObject
            FileObject fo = dobj.getPrimaryFile();
            if (fo == null) {
                return false;
            }
            // primary file content
            is = fo.getInputStream();
            reader = new LineNumberReader(new InputStreamReader(is));

            ArrayList txtDetails = new ArrayList(5);

            int fileCount = 0;
            while (fileCount < MAX_REPORTED_OCCURENCES_IN_FILE) {
                line = reader.readLine();
                if (line == null) {
                    break;
                }
                if (matchString != null) {
                    int lineNum = reader.getLineNumber(); // current line + 1
                    int markLen = matchString.length();

                    String stringToSearch = caseSensitive ? line
                                                          : line.toUpperCase();
                    int i = matchString(stringToSearch, 0);
                    int lineCount = 0;
                    while (i >= 0 && lineCount < MAX_REPORTED_OCCURENCES_ON_LINE) {
                        TextDetail det = new TextDetail(dobj, searchPattern);
                        det.setLine(lineNum);
                        det.setColumn(i + 1); // current column + 1
                        det.setLineText(line);
                        det.setMarkLength(markLen);
                        
                        txtDetails.add(det);

                        i = matchString(stringToSearch, i + 1);
                        lineCount ++;
                        fileCount++;
                    }
                } else if (matchRE(line)) {

                    // TODO detect all occurences as in above code 

                    TextDetail det = new TextDetail(dobj, searchPattern);
                    det.setLine(reader.getLineNumber());
                    det.setLineText(line);
                    Matcher matcher = getMatcher();
                    int start = matcher.start();
                    int len = matcher.end() - start;
                    det.setColumn(start +1);
                    det.setMarkLength(len);

                    txtDetails.add(det);
                    fileCount++;
                }
            }

            // TODO somehow notify user if MAX_REPORTED_OCCURENCES_IN_FILE or MAX_REPORTED_OCCURENCES_ON_LINE

            if (txtDetails.isEmpty()) {
                return false;
            }

            txtDetails.trimToSize();
            getDetailsMap().put(dobj, txtDetails);

            return true;
        } catch (FileNotFoundException fnfe) {
            return false;
        } catch (IOException ioe) {
            org.openide.ErrorManager.getDefault().notify(org.openide.ErrorManager.INFORMATIONAL, ioe);
            return false;
        } finally {
            if (reader != null) {
                try {
                    reader.close();
                    reader = null;
                    is = null;  //marks that the InputStream is closed
                } catch (IOException ex) {
                    org.openide.ErrorManager.getDefault().notify(
                            org.openide.ErrorManager.INFORMATIONAL, ex);
                    
                }
            }
            if (is != null) {
                try {
                    is.close();
                    is = null;
                } catch (IOException ex) {
                    org.openide.ErrorManager.getDefault().notify(
                            org.openide.ErrorManager.INFORMATIONAL, ex);
                }
            }
        }
    }
    
    /**
     * @return  <code>true</code> if the file's MIME-type denotes a searchable
     *                            file;
     *          <code>false</code> otherwise
     */
    protected boolean acceptSearchObject(Object searchObject) {
        DataObject dataObj = (DataObject) searchObject;
        FileObject fileObj = dataObj.getPrimaryFile();
        String mimeType = fileObj.getMIMEType();
        
        if (mimeType.equals("content/unknown")                          //NOI18N
                || mimeType.startsWith("text/")) {                      //NOI18N
            return true;
        }
        if (mimeType.startsWith("application/")) {                      //NOI18N
            final String subtype = mimeType.substring(12);
            return subtype.equals("rtf")                                //NOI18N
                   || subtype.equals("sgml")                            //NOI18N
                   || subtype.startsWith("xml-")                        //NOI18N
                   || subtype.endsWith("+xml")                          //NOI18N
                   || subtype.startsWith("x-")                          //NOI18N
                      && searchableXMimeTypes.contains(subtype.substring(2));
        }
        return false;
    }


    /**
     * @param  resultObject  <code>DataObject</code> to create the nodes for
     * @return  <code>DetailNode</code>s representing the matches,
     *          or <code>null</code> if no matching string is known for the
     *          specified object
     * @see  DetailNode
     */
    public Node[] getDetails(Object resultObject) {
        List details = (List)getDetailsMap().get(resultObject);
        
        if(details == null)
            return null;

        List detailNodes = new ArrayList(details.size());
        
        for(Iterator it = details.iterator(); it.hasNext();) {
            
            TextDetail txtDetail = (TextDetail)it.next();
        
            Node detailNode = new DetailNode(txtDetail);
            detailNodes.add(detailNode);        
        }

        return (Node[])detailNodes.toArray(new Node[detailNodes.size()]);
    }

    /**
     * @param  node representing a <code>DataObject</code> with matches
     * @return  <code>DetailNode</code>s representing the matches,
     *          or <code>null</code> if the specified node does not represent
     *          a <code>DataObject</code> or if no matching string is known for
     *          the specified object
     */
    public Node[] getDetails(Node node) {
        DataObject dataObject = (DataObject)node.getCookie(DataObject.class);
        
        if(dataObject == null) 
            return null;
        
        return getDetails(dataObject);
    }
    
    /**
     */
    public int getDetailsCount(Object resultObject) {
        List details = (List) getDetailsMap().get(resultObject);
        return (details != null) ? details.size() : 0;
    }

    public SearchPattern createSearchPattern() {
        return SearchPattern.create(
                matchString!=null?matchString:reString, 
                wholeWords, 
                caseSensitive, 
                matchString==null);
    }
    
    /**
     * Node that represents information about one occurence of a matching
     * string.
     *
     * @see  TextDetail
     */
    private static class DetailNode extends AbstractNode
                                    implements OutputListener {
        
        /** Detail to represent. */
        private TextDetail txtDetail;
        

        
        /**
         * Constructs a node representing the specified information about
         * a matching string.
         *
         * @param txtDetail  information to be represented by this node
         */
        public DetailNode(TextDetail txtDetail) {
            super(Children.LEAF);
            
            this.txtDetail = txtDetail;
            
            setDefaultAction(SystemAction.get(GotoDetailAction.class));
            setShortDescription(DetailNode.getShortDesc(txtDetail));
            setValue(SearchDisplayer.ATTR_OUTPUT_LINE,
                     DetailNode.getFullDesc(txtDetail));
        }

        
        /** */
        protected SystemAction[] createActions() {
            return new SystemAction[] {
                SystemAction.get(GotoDetailAction.class),
            };
        }

        /** */
        public String getName() {
            return txtDetail.getLineText() + "      [" + DetailNode.getName(txtDetail) + "]";  // NOI18N
        }

        public String getHtmlDisplayName() {
            String colored;
            if (txtDetail.getMarkLength() > 0 && txtDetail.getColumn() > 0) {
                try {
                    StringBuffer bold = new StringBuffer();
                    String plain = txtDetail.getLineText();
                    int col0 =   txtDetail.getColumn() -1;  // base 0

                    bold.append(XMLUtil.toElementContent(plain.substring(0, col0)));  // NOI18N
                    bold.append("<b>");  // NOi18N
                    int end = col0 + txtDetail.getMarkLength();
                    bold.append(XMLUtil.toElementContent(plain.substring(col0, end)));
                    bold.append("</b>"); // NOi18N
                    if (txtDetail.getLineText().length() > end) {
                        bold.append(XMLUtil.toElementContent(plain.substring(end)));
                    }
                    colored = bold.toString();
                } catch (CharConversionException ex) {
                    return null;
                }
            } else {
                try {
                    colored = XMLUtil.toElementContent( txtDetail.getLineText());
                } catch (CharConversionException e) {
                    return null;
                }
            }

            try {
                return colored + "      <font color='!controlShadow'>[" + XMLUtil.toElementContent(DetailNode.getName(txtDetail)) + "]";  // NOI18N
            } catch (CharConversionException e) {
                return null;
            }
        }
      
        /** Displays the matching string in a text editor. */
        private void gotoDetail() {
            txtDetail.showDetail(TextDetail.DH_GOTO);
        }

        /** Show the text occurence. */
        private void showDetail() {
            txtDetail.showDetail(TextDetail.DH_SHOW);
        }

        /** Implements <code>OutputListener</code> interface method. */
        public void outputLineSelected (OutputEvent evt) {
            txtDetail.showDetail(TextDetail.DH_SHOW);
        }

        /** Implements <code>OutputListener</code> interface method. */        
        public void outputLineAction (OutputEvent evt) {
            txtDetail.showDetail(TextDetail.DH_GOTO);
        }

        /** Implements <code>OutputListener</code> interface method. */
        public void outputLineCleared (OutputEvent evt) {
            txtDetail.showDetail(TextDetail.DH_HIDE);
        }

        /**
         * Returns name of a node representing a <code>TextDetail</code>.
         *
         * @param  det  detailed information about location of a matching string
         * @return  name for the node
         */
        private static String getName(TextDetail det) {
            int line = det.getLine();
            int col = det.getColumn();
            
            if (col > 0) {
                
                /* position <line>:<col> */
                return NbBundle.getMessage(FullTextType.class, "TEXT_DETAIL_FMT_NAME1",      //NOI18N
                        Integer.toString(line),
                        Integer.toString(col));
            } else {
                
                /* position <line> */
                return NbBundle.getMessage(FullTextType.class, "TEXT_DETAIL_FMT_NAME2",      //NOI18N
                        Integer.toString(line));
            }
        }

        /**
         * Returns short description of a visual representation of
         * a <code>TextDetail</code>. The description may be used e.g.
         * for a tooltip text of a node.
         *
         * @param  det  detailed information about location of a matching string
         * @return  short description of a visual representation
         */
        private static String getShortDesc(TextDetail det) {
            int line = det.getLine();
            int col = det.getColumn();
            
            if (col > 0) {
                
                /* line <line>, column <col> */
                return NbBundle.getMessage(FullTextType.class, "TEXT_DETAIL_FMT_SHORT1",   //NOI18N
                        new Object[] {Integer.toString(line),
                                      Integer.toString(col)});
            } else {
                
                /* line <line> */
                return NbBundle.getMessage(FullTextType.class, "TEXT_DETAIL_FMT_SHORT2",   //NOI18N
                        Integer.toString(line));
            }
        }

        /**
         * Returns full description of a visual representation of
         * a <code>TextDetail</code>. The description may be printed e.g. to
         * an OutputWindow.
         *
         * @param  det  detailed information about location of a matching string
         * @return  full description of a visual representation
         */
        private static String getFullDesc(TextDetail det) {
            String filename = det.getDataObject().getPrimaryFile().getNameExt();
            String lineText = det.getLineText();
            int line = det.getLine();
            int col = det.getColumn();

            if (col > 0) {

                /* [<filename> at line <line>, column <col>] <text> */
                return NbBundle.getMessage(FullTextType.class, "TEXT_DETAIL_FMT_FULL1",    //NOI18N
                        new Object[] {lineText,
                                      filename,
                                      Integer.toString(line),
                                      Integer.toString(col)});
            } else {

                /* [<filename> line <line>] <text> */
                return NbBundle.getMessage(FullTextType.class, "TEXT_DETAIL_FMT_FULL2",    //NOI18N
                        new Object[] {lineText,
                                      filename,
                                      Integer.toString(line)});
            }
        }
        
    } // End of DetailNode class.

    
    /**
     * This action displays the matching string in a text editor.
     * This action is to be used in the window/dialog displaying a list of
     * found occurences of strings matching a search pattern.
     */
    private static class GotoDetailAction extends NodeAction {
        
        /** */
        public String getName() {
            return NbBundle.getBundle(FullTextType.class).getString("LBL_GotoDetailAction");
        }
        
        /** */
        public HelpCtx getHelpCtx() {
            return new HelpCtx(GotoDetailAction.class);
        }

        /**
         * @return  <code>true</code> if at least one node is activated and
         *          the first node is an instance of <code>DetailNode</code>
         *          (or its subclass), <code>false</code> otherwise
         */
        protected boolean enable(Node[] activatedNodes) {
            return activatedNodes != null && activatedNodes.length != 0
                   && activatedNodes[0] instanceof DetailNode;
        }

        /**
         * Displays the matching string in a text editor.
         * Works only if condition specified in method {@link #enable} is met,
         * otherwise does nothing.
         */
        protected void performAction(Node[] activatedNodes) {
            if (enable(activatedNodes)) {
                ((DetailNode)activatedNodes[0]).gotoDetail();
            }
        }
        
        /**
         */
        protected boolean asynchronous() {
            return false;
        }
        
        
    } // End of GotoDetailAction class.

    /** Gets help context for this search type.
     * Implements superclass abstract method. */
    public HelpCtx getHelpCtx() {
        return new HelpCtx(FullTextType.class);
    }
        
}
