/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */


package org.netbeans.modules.search.types;


import java.awt.event.HierarchyEvent;
import java.awt.event.HierarchyListener;
import java.beans.Customizer;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.TreeSet;
import java.util.Enumeration;
import java.util.List;
import java.util.ResourceBundle;
import javax.swing.border.CompoundBorder;
import javax.swing.border.TitledBorder;
import javax.swing.JPanel;

import org.openide.ErrorManager;
import org.openide.awt.Mnemonics;
import org.openide.loaders.DataLoader;
import org.openide.loaders.DataLoaderPool;
import org.openide.util.Lookup;
import org.openide.util.NbBundle;


/**
 * Customizer of <code>ObjectTypeType</code> bean.
 *
 * @author  Petr Kuzel
 * @author  Marian Petras
 */
public class ObjectTypeCustomizer extends JPanel implements Customizer,
                                                            HierarchyListener {
    
    /**
     * representation class name of the folder loader.
     * This allows to eliminate object type &quot;Folders&quot;
     * from the list of displayed object types.
     */
    private static final String FOLDER_LOADER_NAME
            = "org.openide.loaders.DataFolder";                         //NOI18N

    /** Object to customize. */
    private ObjectTypeType peer;
    
    /** Flag indicating the are setting of object in run. */
    private boolean setting = false;
    /**
     * has this customizer been discovered since the Find dialog had been
     * displayed?
     */
    private boolean discovered = false;
    
    /**
     * cached collection of all loaders from the <code>DataLoaderPool</code>
     *
     * @see  org.openide.loaders.DataLoaderPool#allLoaders()
     */
    private transient Collection loaders;

    
    /** Creates new form <code>ObjectTextCustomizer</code>. */
    public ObjectTypeCustomizer() {
        initComponents();
        initAccessibility();
        addHierarchyListener(this);
        TitledBorder tb = new TitledBorder(NbBundle.getBundle(ObjectTypeCustomizer.class).getString("TEXT_LABEL_OBJECT_TYPE"));
        
        tb.setBorder(new CompoundBorder());
        setBorder (tb);
    }

    private void initAccessibility() {
        ResourceBundle bundle = NbBundle.getBundle(ObjectTypeCustomizer.class);
        this.getAccessibleContext().setAccessibleDescription(bundle.getString("TEXT_LABEL_OBJECT_TYPE"));
        typeList.getAccessibleContext().setAccessibleName(bundle.getString("ACSN_TypeList"));
        typeList.getAccessibleContext().setAccessibleDescription(bundle.getString("ACSD_TypeList"));
        jScrollPane1.getHorizontalScrollBar().getAccessibleContext()
                .setAccessibleName(bundle.getString("ACSN_HScrollBar"));//NOI18N
        jScrollPane1.getVerticalScrollBar().getAccessibleContext()
                .setAccessibleName(bundle.getString("ACSN_VScrollBar"));//NOI18N
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the FormEditor.
     */
    private void initComponents() {//GEN-BEGIN:initComponents
        lblType = new javax.swing.JLabel();
        jScrollPane1 = new javax.swing.JScrollPane();
        typeList = new javax.swing.JList();

        setLayout(new java.awt.BorderLayout(0, 6));

        lblType.setLabelFor(typeList);
        Mnemonics.setLocalizedText(lblType, NbBundle.getMessage(ObjectTypeCustomizer.class, "LBL_Type"));
        add(lblType, java.awt.BorderLayout.NORTH);

        typeList.addListSelectionListener(new javax.swing.event.ListSelectionListener() {
            public void valueChanged(javax.swing.event.ListSelectionEvent evt) {
                typeListValueChanged(evt);
            }
        });

        jScrollPane1.setViewportView(typeList);

        add(jScrollPane1, java.awt.BorderLayout.CENTER);

    }//GEN-END:initComponents

    private void typeListValueChanged (javax.swing.event.ListSelectionEvent evt) {//GEN-FIRST:event_typeListValueChanged
        if (setting) {
            // ignore calls caused by setObject() implementation
            return;
        }
        if (loaders == null) {
            peer.setMask(new Class[0]);
            return;
        }

        List toret = new ArrayList();
        Object[] sel = typeList.getSelectedValues();

        for (Iterator i = loaders.iterator(); i.hasNext(); ) {
            DataLoader nextLoader = (DataLoader) i.next();
            String id = nextLoader.getDisplayName();

            if (id == null) {
                continue; //may be null :-(
            }
            for (int j = 0; j < sel.length; j++) {
                if (sel[j] == null) {
                    continue; //may be null :-(
                }
                if (id.equals((String) sel[j])) {
                    toret.add(nextLoader);
                    break;
                }
            }
        }

        Class[] ret =  new Class[toret.size()];

        Iterator it = toret.iterator();
        int k = 0;
        while (it.hasNext()) {
            ret[k++] = it.next().getClass();
        }

        peer.setMask(ret);
    }//GEN-LAST:event_typeListValueChanged


    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JLabel lblType;
    private javax.swing.JList typeList;
    // End of variables declaration//GEN-END:variables

    // implements java.beans.Customizer
    public void setObject(Object obj) {
        peer = (ObjectTypeType) obj;
    }
    
    /**
     * Loads names of registered object types into the <code>JList</code>.
     */
    private void loadObjectTypes() {
        assert loaders == null;
        
        setting = true;
        try {

            Enumeration en = ((DataLoaderPool) Lookup.getDefault().lookup(DataLoaderPool.class))
                             .allLoaders();

            // Model data for list.
            TreeSet types = new TreeSet();

            ArrayList selected = new ArrayList();

            boolean folderLoaderFound = false;
            
            Class[] mask = peer.getMask();
            if ((mask != null) && (mask.length == 0)) {
                mask = null;
            }
            int loadersToFind = (mask != null) ? mask.length : 0;
            
            loaders = new ArrayList();
            while (en.hasMoreElements()) {
                DataLoader nextLoader = (DataLoader) en.nextElement();
                loaders.add(nextLoader);
                
                /*
                 * Skip the folder object type - we are not able to search
                 * for folders:
                 */
                if (!folderLoaderFound) {
                    if (FOLDER_LOADER_NAME
                            .equals(nextLoader.getRepresentationClassName())) {
                        folderLoaderFound = true;
                        continue;
                    }
                }

                String displayName = nextLoader.getDisplayName();
                if (displayName == null) {
                    continue;
                }
                types.add(displayName);
                
                // Create indices of values to select.
                if (loadersToFind > 0) {
                    Class repreClass = nextLoader.getRepresentationClass();
                    if (repreClass != null) {
                        for (int j = 0; j < mask.length; j++) {
                            if (mask[j] == null) {
                                continue;
                            }
                            DataLoader loader = DataLoader.getLoader(mask[j]);
                            if (loader != null && repreClass.equals(loader.getRepresentationClass())) {
                                selected.add(displayName);
                                loadersToFind--;
                                mask[j] = null;
                                break; 
                            }
                        }
                    }
                }
            }
            ((ArrayList) loaders).trimToSize();

            // select saved values
            typeList.setListData((String[]) types.toArray(new String[types.size()]));

            if (selected.size() > 0) {
                int[] indexes = new int[selected.size()];

                ArrayList typesList = new ArrayList(types);
                for (int i = 0; i < indexes.length; i++) {
                    indexes[i] = typesList.indexOf(selected.get(i));
                }
                typeList.setSelectedIndices(indexes);
            }
        } catch (Exception e) {
            ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, e);
        } finally {
            setting = false;
        }

    }

    /** Dummy implementation of <code>Customizer</code> interface method. */
    public void addPropertyChangeListener(PropertyChangeListener p1) {}

    /** Dummy implementation of <code>Customizer</code> interface method. */
    public void removePropertyChangeListener(PropertyChangeListener p1) {}

    /**
     * Initiates loading of object types the first time this customizer
     * is displayed.
     */
    public void hierarchyChanged(HierarchyEvent e) {
        if (discovered) {
            return;
        }
        
        if (((e.getChangeFlags() & HierarchyEvent.SHOWING_CHANGED) != 0)
                && isShowing()) {
            discovered = true;
            loadObjectTypes();
        }
    }
    
    /** Overrides superclass method. Requests ffocus si delegated to typeList component. */
    public void requestFocus() {
        typeList.requestFocus();
    }

    
}
