/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */


package org.netbeans.modules.search.types;


import java.awt.Color;
import java.awt.Component;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.beans.Customizer;
import java.beans.PropertyChangeListener;
import java.util.HashSet;
import java.util.Iterator;
import java.util.ResourceBundle;
import java.util.List;
import java.util.Set;
import java.util.Vector;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JRootPane;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.border.CompoundBorder;
import javax.swing.border.TitledBorder;
import javax.swing.UIManager;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import org.netbeans.modules.search.DialogLifetime;
import org.netbeans.modules.search.FindDialogMemory;

import org.openide.awt.Mnemonics;
import org.openide.util.NbBundle;
import org.openidex.search.SearchPattern;


/**
 * Customizer of TextType beans.
 *
 * @author  Petr Kuzel
 * @author  Marian Petras
 */
public abstract class TextCustomizer extends JPanel 
        implements Customizer, DialogLifetime, ItemListener,
                   KeyListener
{

    private TextType peer;

    /** Creates new form FullTextCustomizer */
    public TextCustomizer() {
        initComponents ();
        initAccessibility ();
        initTextFieldListeners();
        TitledBorder tb = new TitledBorder(getBorderLabel());
        tb.setBorder(new CompoundBorder());
        setBorder (tb);              

        initHistory();
    }

   
    /** Allow derived customizers.
    */
    protected String getBorderLabel() {
        return null;
    }

    private void initAccessibility(){                
        ResourceBundle bundle = NbBundle.getBundle(TextCustomizer.class);
        substringComboBox.getAccessibleContext().setAccessibleName(bundle.getString("ACSN_PROP_SUBSTRING"));
        substringComboBox.getAccessibleContext().setAccessibleDescription(bundle.getString("ACSD_PROP_SUBSTRING"));
        caseSensitiveCheckBox.getAccessibleContext().setAccessibleDescription(bundle.getString("ACS_TEXT_LABEL_CASE_SENSITIVE"));         
        wholeWordsCheckBox.getAccessibleContext().setAccessibleDescription(bundle.getString("ACS_TEXT_LABEL_WHOLE_WORDS"));         
        regexpCheckBox.getAccessibleContext().setAccessibleDescription(bundle.getString("ACS_TEXT_LABEL_RE"));        
    }
    
    /**
     */
    private void initTextFieldListeners() {
        class TextChangeListener implements DocumentListener {
            private JTextField textField;
            TextChangeListener(JTextField textField) {
                this.textField = textField;
            }
            public void changedUpdate(DocumentEvent e) {
                documentChanged();
            }
            public void insertUpdate(DocumentEvent e) {
                documentChanged();
            }
            public void removeUpdate(DocumentEvent e) {
                documentChanged();
            }
            private void documentChanged() {
                stringChanged();
            }
        }
        ((JTextField)substringComboBox.getEditor().getEditorComponent()).getDocument().addDocumentListener(
                new TextChangeListener((JTextField)substringComboBox.getEditor().getEditorComponent()));
    }
      
    /**
     * Initializes the search pattern combo-box's popup with the last
     * entered patterns. The combo-box's text-field remains empty.
     */
    private void initHistory() {                
        final List/*<SearchPattern>*/ patterns = getSearchPatterns();
        if (!patterns.isEmpty()) {
            final Set inserted = new HashSet(patterns.size());
            final Vector itemsList = new Vector(patterns.size());
            
            final Iterator it = patterns.iterator();
            while (it.hasNext()) {
                String str = ((SearchPattern)it.next()).getSearchExpression();
                if (inserted.add(str)) {
                    itemsList.add(str);
                }
            }
            substringComboBox.setModel(new DefaultComboBoxModel(itemsList));
            substringComboBox.setSelectedIndex(-1);
        }
    }
    
    /**
     * Initializes the text-field with the last value from the history.
     */
    public void initFromHistory() {
        Object topmostItem = getSearchPatterns().get(0);
        if (topmostItem != null) {
            putCurrentSearchPattern((SearchPattern) topmostItem);
        }
    }
    
    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the FormEditor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        substringComboBox = new javax.swing.JComboBox();
        jPanel1 = new javax.swing.JPanel();

        setLayout(new java.awt.GridBagLayout());

        substringComboBox.setEditable(true);
        substringComboBox.setMaximumSize(new java.awt.Dimension(2147483647, 2147483647));
        substringComboBox.setMinimumSize(null);
        substringComboBox.setOpaque(false);
        substringComboBox.setPreferredSize(null);
        substringComboBox.addItemListener(this);
        activateEnterKeyBypass();
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 5, 0, 11);
        add(substringComboBox, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weighty = 1.0;
        add(jPanel1, gridBagConstraints);

        Mnemonics.setLocalizedText(caseSensitiveCheckBox, NbBundle.getMessage(TextCustomizer.class, "TEXT_LABEL_CASE_SENSITIVE"));  //NOI18N
        caseSensitiveCheckBox.setBorder(null);
        caseSensitiveCheckBox.addItemListener(this);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(2, 12, 0, 11);
        add(caseSensitiveCheckBox, gridBagConstraints);

        Mnemonics.setLocalizedText(wholeWordsCheckBox, NbBundle.getMessage(TextCustomizer.class, "TEXT_LABEL_WHOLE_WORDS"));  //NOI18N
        wholeWordsCheckBox.setBorder(null);
        wholeWordsCheckBox.addItemListener(this);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(5, 12, 0, 11);
        add(wholeWordsCheckBox, gridBagConstraints);

        Mnemonics.setLocalizedText(regexpCheckBox, NbBundle.getMessage(TextCustomizer.class, "TEXT_LABEL_RE"));  //NOI18N
        regexpCheckBox.setBorder(null);
        regexpCheckBox.addItemListener(this);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = new java.awt.Insets(2, 12, 0, 11);
        add(regexpCheckBox, gridBagConstraints);

    }
    // </editor-fold>//GEN-END:initComponents

    /**
     * Creates a bypass around the default JComboBox's and JTextField's
     * mechanism for handling key events.
     * The bypass ensures that the Enter key always activates
     * the default button (if enabled). It was made as a quick fix of bug #54279
     * ("Need to press Enter twice to start searching").
     */
    private void activateEnterKeyBypass() {
        Component editor = substringComboBox.getEditor().getEditorComponent();
        if (!(editor instanceof JTextField)) {
            assert false;
            return;
        }
        ((JTextField) editor).addKeyListener(this);
    }
    
    /**
     * If the pressed key was Enter, activates the default button (if enabled).
     * It also consumes the event so that the default
     * <code>JTextField</code>'s handling mechanism is bypassed.
     */
    public void keyPressed(KeyEvent e) {
        if ((e.getKeyCode() == KeyEvent.VK_ENTER) && (e.getModifiersEx() == 0)){
            
            JRootPane rootPane = SwingUtilities.getRootPane(this);
            if (rootPane != null) {
                JButton button = rootPane.getDefaultButton();
                if ((button != null) && button.isEnabled()) {
                    e.consume();
                    button.doClick();
                }
            }
        }
    }
    
    /**
     */
    public void keyReleased(KeyEvent e) {
        //ignore
    }
    
    /**
     */
    public void keyTyped(KeyEvent e) {
        //ignore
    }
    
    private String getComboText() {
        return ((JTextField)substringComboBox.getEditor().getEditorComponent()).getText();
    }
    
    /**
     * Called when the Regular Expression check-box is selected or deselected.
     */
    private void regexpChkBoxChanged() {
        enableUI();
        
        if (peer == null) {
            return;
        }
        
        final String text = getComboText();
        if ((text == null) || (text.length() == 0)) {
            
            /*
             * The checkbox was (de)selected at the moment the combo-box
             * editor was empty - so this cannot change the validity
             * and there is no need to update the peer.
             */
            return;
        }
        
        if (!regexpCheckBox.isSelected()) {
            peer.setMatchString(text);
            substringComboBox.getEditor().getEditorComponent().setForeground(getForegroundColor());
        } else {
            try {
                peer.setRe(text);
                substringComboBox.getEditor().getEditorComponent().setForeground(getForegroundColor());
            } catch (IllegalArgumentException ex) {
                substringComboBox.getEditor().getEditorComponent().setForeground(getErrorForegroundColor());
            }
        }
    }
    
    /**
     * Called whenever contents of the textfield changes.
     */
    private void stringChanged() {
        if (peer == null) {
            return;
        }
        
        final String text = getComboText();
        if (!regexpCheckBox.isSelected()) {
            peer.setMatchString(text);
            substringComboBox.getEditor().getEditorComponent().setForeground(getForegroundColor());
        } else {
            try {
                peer.setRe(text);
                substringComboBox.getEditor().getEditorComponent().setForeground(getForegroundColor());
            } catch (IllegalArgumentException ex) {
                substringComboBox.getEditor().getEditorComponent().setForeground(getErrorForegroundColor());
            }
        }
    }    
   
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private final javax.swing.JCheckBox caseSensitiveCheckBox = new javax.swing.JCheckBox();
    private javax.swing.JPanel jPanel1;
    private final javax.swing.JCheckBox regexpCheckBox = new javax.swing.JCheckBox();
    private javax.swing.JComboBox substringComboBox;
    private final javax.swing.JCheckBox wholeWordsCheckBox = new javax.swing.JCheckBox();
    // End of variables declaration//GEN-END:variables

    /** Initialize customizer with proper values.
    */
    public void setObject(final Object obj) {

        peer = (TextType) obj;

        // set default coloring
        substringComboBox.setForeground(getForegroundColor());

        String text;
        Boolean isRegexp = null;
        
        if ((text = peer.getRe()).length() != 0) {
            isRegexp = Boolean.TRUE;
        } else if ((text = peer.getMatchString()).length() != 0) {
            isRegexp = Boolean.FALSE;
        } else {
            text = getComboText();
        }
        
        if (isRegexp != null) {
            
            /* Let's initialize the form according to the peer: */
            substringComboBox.setSelectedItem(text);
            
            regexpCheckBox.setSelected(isRegexp.booleanValue());
            caseSensitiveCheckBox.setSelected(peer.isCaseSensitive());
            wholeWordsCheckBox.setSelected(peer.getWholeWords());
            
        } else if ((text != null) && (text.length() != 0)) {
            
            /* Let's initialize the peer according to the form content: */
            stringChanged();
        }
        
        enableUI();
    }
    
    public void addPropertyChangeListener(final PropertyChangeListener p1) {
    }

    public void removePropertyChangeListener(final PropertyChangeListener p1) {
    }

    public void requestFocus() {
        JTextField tf = (JTextField)substringComboBox.getEditor().getEditorComponent();
        int n = tf.getText().length();
        if (n > 0) {
            tf.setCaretPosition(0);
            tf.moveCaretPosition(n);
        }
        tf.requestFocus();
    }


    // colors

    private Color findColor (String key, Color defCol) {
        Color color = UIManager.getDefaults().getColor (key);
        if ( color != null ) {
            return color;
        }
        return defCol;
    }

    private Color getForegroundColor () {
        return findColor ("TextField.foreground", Color.black);
    }

    private Color getErrorForegroundColor () {
        return findColor ("TextField.errorForeground", Color.red);
    }

    /**
     * Enables or disables UI components (the checkboxes) based on the state
     * of the regexpCheckBox.
     */
    private void enableUI() {
        boolean r = regexpCheckBox.isSelected();
        regexpCheckBox.setEnabled(true);
        caseSensitiveCheckBox.setEnabled(!r);
        wholeWordsCheckBox.setEnabled(!r);          
    }
    
    /**
     * ItemListener implementation
     */
    public void itemStateChanged(ItemEvent e) {
        Object source = e.getSource();
        
        if (source == substringComboBox) {
            stringChanged();
        
        } else if (source == wholeWordsCheckBox) {
            peer.setWholeWords(wholeWordsCheckBox.isSelected());
            
        } else if (source == caseSensitiveCheckBox) {
            peer.setCaseSensitive(caseSensitiveCheckBox.isSelected());
            
        } else if (source == regexpCheckBox) {
            regexpChkBoxChanged();
            
        } else {
            /* This should never happen. */
            assert false;
        }
    }
    
    /**
     * Read current dialog contents as a SearchPattern. 
     * @return SearchPattern for the contents of the current dialog. Null if the
     *         search string is empty, meaning that the dialog is empty.
     */
    private SearchPattern getCurrentSearchPattern() {
        return SearchPattern.
                create(
                    getComboText(),
                    wholeWordsCheckBox.isSelected(),
                    caseSensitiveCheckBox.isSelected(),
                    regexpCheckBox.isSelected());
    }
    
    /**
     * Init dialog according to the pattern.
     * @param pat SearchPattern to fill into the dialog. Use null to empty the 
     *            dialog.
     */
    private void putCurrentSearchPattern(final SearchPattern pat) {
        assert pat != null;

        substringComboBox.setSelectedItem(pat.getSearchExpression());
        wholeWordsCheckBox.setSelected(pat.isWholeWords());
        caseSensitiveCheckBox.setSelected(pat.isMatchCase());
        regexpCheckBox.setSelected(pat.isRegExp());
        
        enableUI();
    }
    
    private void initCheckBoxes(SearchPattern pat) {
        wholeWordsCheckBox.setSelected(pat.isWholeWords());
        caseSensitiveCheckBox.setSelected(pat.isMatchCase());
        regexpCheckBox.setSelected(pat.isRegExp());
        enableUI();
    }
    
    /*****************************/
    /** 
     * Hooks to search history 
     * By default, the history is empty.
     **/
   
    /** 
     * Returns an unmodifiable List of SearchPatterns. 
     * By default, history is stored locally and not serialized between sessions.
     *
     * @return unmodifiable List of SearchPatterns 
     */
    abstract protected List/*<SearchPattern>*/ getSearchPatterns();
    
    /** Adds SearchPattern to SearchHistory 
     *  @param pattern the SearchPattern to add
     */
    abstract protected void addSearchPattern(SearchPattern pattern);


    public void onOk() {
        final SearchPattern searchPattern = getCurrentSearchPattern();
        
        if (searchPattern != null) {
            addSearchPattern(searchPattern);
            
            String expr = searchPattern.getSearchExpression();
            if ((expr != null) && (expr.length() != 0)) {
                FindDialogMemory.getDefault()
                        .setSearchTypeUsed(peer.getClass().getName(), true);
            }
        }
    }
    
    public void onCancel() {
    }

}
