/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */


package org.netbeans.modules.search.types;


import javax.swing.JEditorPane;
import javax.swing.SwingUtilities;
import javax.swing.text.Caret;

import org.openide.cookies.EditorCookie;
import org.openide.cookies.LineCookie;
import org.openide.loaders.DataObject;
import org.openide.text.Line;

import java.awt.*;
import org.openidex.search.SearchHistory;
import org.openidex.search.SearchPattern;


/** 
 * Holds details about one search hit in the text document.
 *
 * @author Tomas Pavek
 * @author Marian Petras
 */
public class TextDetail extends Object {

    /** Property name which indicates this detail to show. */
    public static final int DH_SHOW = 1;
    /** Property name which indicates this detail to go to. */
    public static final int DH_GOTO = 2;
    /** Property name which indicates this detail to hide. */
    public static final int DH_HIDE = 3;
    
    /** Data object. */
    private DataObject dobj;
    /** Line number where search result occures.*/
    private int line;
    /** Text of the line. */ 
    private String lineText;
    /** Column where search result starts. */
    private int column;
    /** Length of search result which to mark. */
    private int markLength;
    /** Line. */
    private Line lineObj;
    /** SearchPattern used to create the hit of this DetailNode */
    private SearchPattern searchPattern;

    
    
    /** Constructor using data object. 
     * @param pattern  SearchPattern used to create the hit of this DetailNode 
     */
    public TextDetail(DataObject dobj, SearchPattern pattern) {
        this.dobj = dobj;
        this.searchPattern = pattern;
    }

    /**
     * Shows the search detail on the DataObject.
     * The document is opened in the editor, the caret is positioned on the right line and column 
     * and searched string is marked.
     *
     * @param how indicates how to show detail. 
     * @see #DH_GOTO 
     * @see #DH_SHOW 
     * @see #DH_HIDE */
    public void showDetail(int how) {
        if (dobj == null) {
            Toolkit.getDefaultToolkit().beep();
            return;
        }
        if (lineObj == null) { // try to get Line from DataObject
            LineCookie lineCookie = (LineCookie) dobj.getCookie(LineCookie.class);
            if (lineCookie != null) {
                Line.Set lineSet = lineCookie.getLineSet();
                try {
                    lineObj = lineSet.getOriginal(line - 1);
                } catch (IndexOutOfBoundsException ioobex) {
                    // The line doesn't exist - go to the last line
                    lineObj = lineSet.getOriginal(findMaxLine(lineSet));
                    column = markLength = 0;
                }
            }
            if (lineObj == null) {
                Toolkit.getDefaultToolkit().beep();
                return;
            }
        }

        if (how == DH_HIDE) {
            return;
        }
        EditorCookie edCookie = (EditorCookie) dobj.getCookie(EditorCookie.class);
        if (edCookie != null)
            edCookie.open();
        if (how == DH_SHOW) {
            lineObj.show(Line.SHOW_TRY_SHOW, column - 1);
        }
        else if (how == DH_GOTO) {
            lineObj.show(Line.SHOW_GOTO, column - 1);
        }
        if (markLength > 0 && edCookie != null) {
            final JEditorPane[] panes = edCookie.getOpenedPanes();
            if (panes != null && panes.length > 0) {
                // Necessary since above lineObj.show leads to invoke later as well.
                SwingUtilities.invokeLater(new Runnable() {
                    public void run() {
                        Caret caret = panes[0].getCaret(); // http://www.netbeans.org/issues/show_bug.cgi?id=23626
                        caret.moveDot(caret.getDot() + markLength);
                    }
                });
            }
        }
        SearchHistory.getDefault().setLastSelected(searchPattern);
    }

    /** Getter for <code>lineText</code> property. */
    public String getLineText() {
        return lineText;
    }
    
    /** Setter for <code>lineText</code> property. */
    public void setLineText(String text) {
        lineText = text;
    }
    

    /**
     * Gets the <code>DataObject</code> where the searched text was found. 
     *
     * @return data object or <code>null</code> if no data object is available
     */
    public DataObject getDataObject() {
        return dobj;
    }

    /** Gets the line position of the text. */
    public int getLine() {
        return line;
    }

    /** Sets the line position of the text. */
    public void setLine(int line) {
        this.line = line;
    }

    /** Gets the column position of the text or 0 (1 based). */
    public int getColumn() {
        return column;
    }

    /** Sets the column position of the text. */
    public void setColumn(int col) {
        column = col;
    }

    /** Gets the length of the text that should be marked when the detail is shown. */
    public void setMarkLength(int len) {
        markLength = len;
    }

    /** @return length or 0 */
    public int getMarkLength() {
        return markLength;
    }
    
    /**
     * Returns the maximum line in the <code>set</code>.
     * Used to display the end of file when the corresponding
     * line no longer exists. (Copied from org.openide.text)
     *
     * @param set the set we want to search.
     * @return maximum line in the <code>set</code>.
     */
    private static int findMaxLine(Line.Set set) {
        int from = 0;
        int to = 32000;
        
        for (;;) {
            try {
                set.getOriginal(to);
                // if the line exists, double the max number, but keep
                // for reference that it exists
                from = to;
                to *= 2;
            } catch (IndexOutOfBoundsException ex) {
                break;
            }
        }
        
        while (from < to) {
            int middle = (from + to + 1) / 2;
            
            try {
                set.getOriginal(middle);
                // line exists
                from = middle;
            } catch (IndexOutOfBoundsException ex) {
                // line does not exists, we have to search lower
                to = middle - 1;
            }
        }
        
        return from;
    }

}
