/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.web.core.syntax;

import java.lang.reflect.Modifier;
import java.util.*;
import javax.swing.text.BadLocationException;
import org.netbeans.editor.TokenItem;

import org.netbeans.editor.ext.java.*;
import org.netbeans.editor.BaseDocument;
import org.netbeans.jmi.javamodel.Feature;
import org.netbeans.jmi.javamodel.Import;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.jmi.javamodel.Type;
import org.netbeans.modules.editor.NbEditorUtilities;
import org.netbeans.modules.editor.java.NbJavaJMISyntaxSupport;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.web.jsps.parserapi.JspParserAPI;
import org.netbeans.modules.web.jsps.parserapi.JspParserAPI.ParseResult;
import org.netbeans.modules.web.jsps.parserapi.PageInfo;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.netbeans.api.project.*;

/**
 * This class generates a "fake" JMI JavaClass containing "default JSP objects",
 * used JavaBeans also handle the JSP page imports.
 * The fake java class is then used by Java code completion to query the completion data.
 *
 * @author  Marek Fukala
 */
public class JspJavaSyntaxSupport extends NbJavaJMISyntaxSupport {
    
    JspSyntaxSupport jspSup;
    
    private static final String[][] IMPLICIT_OBJECTS =
    {{"request","javax.servlet.http.HttpServletRequest"},
     {"response","javax.servlet.http.HttpServletResponse"},
     {"session","javax.servlet.http.HttpSession"},
     {"application","javax.servlet.ServletContext"},
     {"out","javax.servlet.jsp.JspWriter"},
     {"config","javax.servlet.ServletConfig"}};
     
     private static final String[][] IMPLICIT_TAG_OBJECTS =
     {{"jspContext","javax.servlet.jsp.JspContext"}};
     
     private static final String[][] IMPLICIT_JSP_OBJECTS =
     {{"page","java.lang.Object"},
      {"pageContext","javax.servlet.jsp.PageContext"}};
      
      /** Stores JSP implicit objects, javaBeans and other potential classes from external sources (not impl. yet).*/
      private HashMap additionalJspObjectsMap = null;
      
      private JavaClass javaClass = null; //JMI java class for the JSP page
      private Resource res = null; //fake resource for the java class
      
      private FileObject sourceRoot = null; //source root
      
      /** Creates new JspJavaSyntaxSupport */
      public JspJavaSyntaxSupport(BaseDocument doc, JspSyntaxSupport jspSup) {
          super(doc);
          this.jspSup = jspSup;
          
          //find project according to the fileobject
          if(jspSup.getFileObject() != null) {
              Project project = FileOwnerQuery.getOwner(jspSup.getFileObject());
              if(project != null) {
                  Sources sources = ProjectUtils.getSources(project);
                  SourceGroup[] sg = sources.getSourceGroups("doc_root");
                  if(sg != null && sg.length > 0) sourceRoot = sg[0].getRootFolder();
              }
          }
      }
      
      public boolean isStaticBlock(int pos) {
          return false;
      }
      
      public JavaClass getJavaClass(int pos) {
          if(javaClass == null || !javaClass.isValid()) {
              initJavaClass();
          }
          return javaClass;
      }
      
      public Resource getResource() {
          if(res == null || !res.isValid()) {
              initJavaClass();
          }
          return res;
      }
      
      public Feature getFeatureAtPos(int pos, boolean includingJavadoc) {
          return null;
      }
      
      protected int getMethodStartPosition(int pos) {
          return 0;
      }
      
      public Collection getLocalVariableNames(String namePrefix, int pos, boolean exactMatch) {
          ArrayList al = new ArrayList();
          //add local variables given from java parser
          Collection col = super.getLocalVariableNames(namePrefix, pos, exactMatch);
          al.addAll(col);
          
          //check CC context - we have to distinguish between jsp scriptlet and declaration
          //in the second case do not add any jsp objects
          try {
              TokenItem token = jspSup.getItemAtOrBefore(pos);
              if(!token.getTokenContextPath().contains(JspJavaFakeTokenContext.JavaDeclarationTokenContext.contextPath)) {
                  //not directive => add jsp objects
                  Collection filteredJSPObjects = getJMIUtils().filterNames(getJspObjects().keySet(), namePrefix, exactMatch);
                  al.addAll(filteredJSPObjects);
              }
          }catch(BadLocationException e) {
              e.printStackTrace();
          }
          return al;
      }
      
      public Object findType(String varName, int varPos) {
          //test if the variable is from the JSP objects set
          Map jspObjs = getJspObjects();
          if(jspObjs.containsKey(varName) && jspObjs.get(varName) != null)
              return (Type)jspObjs.get(varName);
          else
              return super.findType(varName, varPos);
      }
      
      /** Creates a JavClass JMI Java model instance representing the JSP page. */
      private void initJavaClass() {
          if(sourceRoot != null) {
              JavaModel.getJavaRepository().beginTrans(true);
              try {
                  JavaModelPackage jmp = JavaModel.getJavaExtent(sourceRoot);
                  
                  //not sure why, but the first attempt to get JavaModelPackage
                  //returns null?!?
                  if(jmp == null) return ;
                  
                  //create the fake Java class
                  javaClass = jmp.getJavaClass().createJavaClass(
                          "JspFakeJMIClass",     // name
                          null,                  // annnotations
                          Modifier.PUBLIC,       // modifiers
                          null,                  // javadoc
                          null,                  // javaDoc text
                          null,                  // empty features
                          jmp.getMultipartId().createMultipartId("javax.servlet.http.HttpServlet", null, null),                 // superclass
                          null,                  // interfaces
                          null
                          );
                  
                  res = jmp.getResource().createResource();
                  
                  res.setName("JspFakeJMIClass.java");
                  res.getClassifiers().add(javaClass);
                  res.setPackageName("");
                  
              } finally {
                  JavaModel.getJavaRepository().endTrans();
              }
          }
      }
      
      private Map getJspObjects() {
          HashMap all = new HashMap(getImplicitObjects());
          //cache additional objects - the cache is cleared by refreshClasses() method
          if(additionalJspObjectsMap == null) {
              additionalJspObjectsMap = new HashMap(getJavaBeans());
              additionalJspObjectsMap.putAll(getAdditionalObjects());
          }
          
          all.putAll(additionalJspObjectsMap);
          return all;
      }
      
      /** an external source may override this method to add its
       * objects into the java CC - e.g. JSF objects. */
      protected Map getAdditionalObjects() {
          return Collections.EMPTY_MAP;
      }
      
      private Map getImplicitObjects() {
          HashMap defs =  new HashMap(getJMIClassMap(IMPLICIT_OBJECTS));
          
          if (NbEditorUtilities.getMimeType(jspSup.getDocument()).equals(JspUtils.TAG_MIME_TYPE))
              defs.putAll(getJMIClassMap(IMPLICIT_TAG_OBJECTS));
          else
              defs.putAll(getJMIClassMap(IMPLICIT_JSP_OBJECTS));
          
          return defs;
      }
      
      private Map getJMIClassMap(String[][] varsDef) {
          if(sourceRoot == null) return Collections.EMPTY_MAP;
          HashMap imos = new HashMap();
          JavaModel.getJavaRepository().beginTrans(true);
          try {
              JavaModelPackage jmp = JavaModel.getJavaExtent(sourceRoot);
              if(jmp == null) return Collections.EMPTY_MAP;
              
              for(int i = 0; i < varsDef.length; i ++) {
                  String fieldName = varsDef[i][0];
                  String fieldClassName = varsDef[i][1];
                  imos.put(fieldName, jmp.getJavaClass().resolve(fieldClassName));
              }
              return imos;
          } finally {
              JavaModel.getJavaRepository().endTrans();
          }
      }
      
      private Map getJavaBeans() {
          //add used javaBeans
          HashMap al = new HashMap();
          PageInfo.BeanData[] beanData = jspSup.getBeanData(); //may return null for JSPs outside of a webmodule
          if(beanData != null) {
              for (int i = 0; i < beanData.length; i++) {
                  al.putAll(getJMIClassMap(new String[][]{{beanData[i].getId(), beanData[i].getClassName()}}));
              }
              return al;
          } else return Collections.EMPTY_MAP;
      }
      
      public void refreshClassInfo() {
          //clear cache of additional objects so the next call will recreate the list
          additionalJspObjectsMap = null;
          getResource(); //init resource and javaclass
          updateImportClasses();
      }
      
      /** Are there any changes in the imports?
       */
      private void updateImportClasses(){
          JavaModel.getJavaRepository().beginTrans(true);
          try {
              JspParserAPI.ParseResult pre = jspSup.getParseResult();
              Resource res = getResource(); //get the resource, if invalid then reinitialize
              if (res != null && res.isValid() && pre != null){
                  PageInfo pi = pre.getPageInfo();
                  if(pi == null) {
                      //report error but do not break the entire CC 
                      ErrorManager.getDefault().notify(ErrorManager.WARNING, new NullPointerException("PageInfo obtained from JspParserAPI.ParseResult is null!"));
                      return ;
                  }
                  List imports = pi.getImports();
                  
                  //scan for unused imporst (registered into resource but not gotten from jasper)
                  Iterator existingImportsItr = res.getImports().iterator();
                  ArrayList existingImportsNames = new ArrayList();
                  ArrayList unusedImports = new ArrayList();
                  while(existingImportsItr.hasNext()) {
                      Import imp = (Import)existingImportsItr.next();
                      String impExp = imp.getName() + (imp.isOnDemand() ? ".*" : "");
                      existingImportsNames.add(impExp); //needed by imports adding code
                      if(!imports.contains(impExp)) unusedImports.add(imp);
                  }
                  //remove the unused imports
                  Iterator unusedImportsItr = unusedImports.iterator();
                  while(unusedImportsItr.hasNext()) {
                      Import imp = (Import)unusedImportsItr.next();
                      //System.out.println("--- REMOVING " + imp.getName());
                      res.getImports().remove(imp);
                  }
                  //add new imports
                  for (int i = 0; i < imports.size(); i ++){
                      String impExp = (String)imports.get(i);
                      if(!existingImportsNames.contains(impExp)) addImport(impExp);
                  }
              }
          } finally {
              JavaModel.getJavaRepository().endTrans();
          }
      }
      
      private void addImport(String impExp) {
          if(sourceRoot != null) {
              JavaModelPackage jmp = JavaModel.getJavaExtent(sourceRoot);
              Import imp = null;
              if(impExp.endsWith(".*")) {
                  String _impExp = impExp.substring(0, impExp.length() - 2); //cut of '.*'
                  //System.out.println("+++ ADDING " + _impExp);
                  imp = jmp.getImport().createImport(_impExp, null, false, true);
              } else {
                  //System.out.println("+++ ADDING " + impExp);
                  imp = jmp.getImport().createImport(impExp, null, false, false);
              }
              res.addImport(imp);
          }
      }
      
}

