/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.test.web;

import java.io.BufferedReader;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.net.URLConnection;
import org.netbeans.jemmy.JemmyException;
import org.openide.awt.HtmlBrowser;
import org.openide.util.Lookup;

/**
 * Designed to replace default URLDisplayer to better serve automated tests
 * requirements.
 * <p>
 * Usage:<br>
 * <pre>
 *      TestURLDisplayer displayer = TestURLDisplayer.getInstance();
 *      displayer.invalidateURL();
 *      // e.g. run a .jsp
 *      displayer.waitURL();
 *      String page = displayer.readURL();
 *      ...
 * </pre>
 *
 * @author Martin.Schovanek@sun.com
 */
public final class TestURLDisplayer extends HtmlBrowser.URLDisplayer {
    private static TestURLDisplayer instance;
    private boolean isURLValid = false;
    private URL url = null;
    private URLConnection con = null;
    
    public static synchronized TestURLDisplayer getInstance() {
        if (instance==null) {
            instance=registerTestURLDisplayer();
        }
        return instance;
    }
    
    public synchronized void showURL(URL u) {
        url=u;
        try {
            con = url.openConnection();
        } catch (IOException ex) {
            System.err.println("Cannot open URL: "+url);
            ex.printStackTrace();
        }
        // force to send request
        final URLConnection fc = con;
        new Thread () {
            public void run() {
                try {
                    fc.getInputStream();
                } catch (IOException ex) {
                    System.err.println("Cannot read URL: "+url);
                    ex.printStackTrace();
                }
            }
        }.start();

        notifyAll();
        isURLValid=true;
    }
    
    public synchronized void invalidateURL() {
        url = null;
        con = null;
        isURLValid = false;
    }
    
    public synchronized URL waitURL() throws InterruptedException {
        while (!isURLValid) {
            wait(60000);
            if (!isURLValid) {
                throw new IllegalStateException("Timeout expired.");
            }
        }
        return url;
    }
    
    public String readURL() {
        if (!isURLValid || url == null) {
            throw new IllegalStateException("URL is not valid.");
        }
        StringBuffer sb = new StringBuffer();
        InputStream is = null;
        try{
            is = con.getInputStream();
            BufferedReader reader = new BufferedReader(new InputStreamReader(is));
            String line = null;
            while((line=reader.readLine()) != null){
                sb.append(line+"\n");
            }
        }catch(Exception ex){
            ex.printStackTrace();
        }finally{
            try{
                is.close();
            }catch(Exception ex){
                // nothing to do
            }
        }
        return sb.toString();
    }
    
    public URL getURL() {
        return url;
    }
    
    private static TestURLDisplayer registerTestURLDisplayer() {
        try {
            File file = new File(System.getProperty("xtest.workdir")+"/sys/tests" +
                    "/META-INF/services/org.openide.awt.HtmlBrowser$URLDisplayer");
            DataOutputStream dout = new DataOutputStream(new FileOutputStream(file));
            dout.writeBytes(
                    "#-org.netbeans.core.NbTopManager$NbURLDisplayer\n"+
                    "org.netbeans.test.web.TestURLDisplayer\n");
            dout.close();
        } catch (IOException ex) {
            throw new JemmyException("Cannot register URL displayer.", ex);
        }
        Object displayer = Lookup.getDefault().lookup(HtmlBrowser.URLDisplayer.class);
        if (!displayer.getClass().equals(TestURLDisplayer.class)) {
            throw new JemmyException("URL displayer registration failed");
        }
        return (TestURLDisplayer) displayer;
    }
}
