/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.websvc.editor.completion;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.swing.text.BadLocationException;
import javax.swing.text.JTextComponent;
import org.netbeans.editor.TokenItem;
import org.netbeans.editor.ext.CompletionQuery;
import org.netbeans.editor.ext.Completion;
import org.netbeans.editor.ext.ExtEditorUI;
import org.netbeans.editor.BaseDocument;
import org.netbeans.editor.Utilities;
import org.netbeans.editor.ext.java.JCExpression;
import org.netbeans.editor.ext.java.JCFinder;
import org.netbeans.editor.ext.java.JavaCompletionQuery;
import org.netbeans.editor.ext.java.JavaSyntaxSupport;
import org.netbeans.editor.ext.java.JavaTokenContext;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.editor.NbEditorUtilities;
import org.netbeans.modules.editor.java.JCFinderFactory;
import org.netbeans.modules.editor.java.NbJavaJMISyntaxSupport;
import org.netbeans.modules.javacore.JMManager;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;

/**
 * Completion query for JSR-181 (Webservice Metadata) aNNotations.
 *
 * @author Marek Fukala
 * @author Dusan Balek
 */
public class WSCompletionQuery extends JavaCompletionQuery {
    
    private static final String WEBSERVICE_NN = "WebService";//NOI18N
    private static final String HANDLER_CHAIN_NN = "HandlerChain";//NOI18N
    
    private List resolvers;
    
    public WSCompletionQuery(boolean isJava15) {
        setJava15(isJava15);
    }
    
    protected JCFinder getFinder() {
        FileObject fo = getFileObject();
        return JCFinderFactory.getDefault().getFinder(fo);
    }
    
    protected CompletionQuery.Result getResult(JTextComponent component, JavaSyntaxSupport sup, boolean openingSource, int offset, JCExpression exp) {
        Completion completion = ((ExtEditorUI)Utilities.getEditorUI(component)).getCompletion();
        boolean autoPopup = completion != null ? completion.provokedByAutoPopup : false;
        JMIUtils utils = JMIUtils.get(getBaseDocument());
        utils.beginTrans(false);
        try {
            ((JMManager) JMManager.getManager()).setSafeTrans(true);
            Context ctx = new Context(component, (NbJavaJMISyntaxSupport)sup.get(NbJavaJMISyntaxSupport.class), openingSource, offset, utils, autoPopup);
            boolean ok = ctx.resolveExp(exp);
            return ok ? ctx.result : null;
        } finally {
            utils.endTrans(false);
        }
    }
    
    private FileObject getFileObject() {
        BaseDocument bDoc = getBaseDocument();
        DataObject dobj = NbEditorUtilities.getDataObject(bDoc);
        return dobj.getPrimaryFile();
    }
    
    private final class Context {
        
        /** Text component */
        private JTextComponent component;
        
        /** Syntax support for the given document */
        private NbJavaJMISyntaxSupport sup;
        
        /** Whether the query is performed to open the source file. It has slightly
         * different handling in some situations.
         */
        private boolean openingSource;
        
        /** End position of the scanning - usually the caret position */
        private int endOffset;
        
        /** If set to true true - find the type of the result expression.
         * It's stored in the lastType variable or lastPkg if it's a package.
         * The result variable is not populated.
         * False means that the code completion output should be collected.
         */
        private boolean findType;
        
        /** Whether currently scanning either the package or the class name
         * so the results should limit the search to the static fields and methods.
         */
        private boolean staticOnly = true;
        
        /** Last package found when scanning dot expression */
        private JavaPackage lastPkg;
        
        /** Last type found when scanning dot expression */
        private Type lastType;
        
        /** Result list when code completion output is generated */
        private DefaultResult result;
        
        /** Helper flag for recognizing constructors */
        private boolean isConstructor;
        
        private boolean isImport;
        private boolean isStaticImport;
        
        private boolean isGeneric;
        private Collection typeBounds;
        
        private boolean isAnnotation;
        private boolean isAnnotationOpen;
        
        private JavaClass curCls;
        
        /** True when code completion is invoked by auto popup. In such case, code completion returns no result
         * after "new ". To get a result, code completion has to be invoked manually (using Ctrl-Space). */ // NOI18N
        private boolean autoPopup;
        
        /** Finder associated with this Context. */
        private JMIUtils jmiUtils = null;
        
        public Context(JTextComponent component, NbJavaJMISyntaxSupport sup, boolean openingSource, int endOffset, JMIUtils utils, boolean autoPopup) {
            
            this.component = component;
            this.sup = sup;
            this.openingSource = openingSource;
            this.endOffset = endOffset;
            this.jmiUtils = utils;
            this.curCls = sup.getJavaClass(endOffset);
            if (this.curCls == null) {
                this.curCls = sup.getTopJavaClass();
            }
            this.autoPopup = autoPopup;
        }
        
        boolean resolveExp(JCExpression exp) {
            //System.out.println(exp);
            switch (exp.getExpID()) {
                case JCExpression.ANNOTATION_OPEN:
                    String annotationName = getAnnotationTypeName(exp);
                    resolveCompletionContext(annotationName, getCompletedMemberName(exp), "");
                    break;
                case JCExpression.CONSTANT:
                    //JCTokenProcessor hack - we cannot easily get an information about the current context
                    resolveConstantExpression(exp);
                    break;
            }
            return true;
        }
        
        private void resolveConstantExpression(JCExpression exp) {
            NNParser parser = new NNParser(getBaseDocument());
            NNParser.NN nn = parser.parseAnnotation(endOffset);
            //XXX I should improve the NNParser to provide the offset data!
            //temp - parse backward and look for the attribute name
            if(nn != null) {
                try {
                    TokenItem ti = sup.getTokenChain(endOffset -1, endOffset);
                    if(ti.getTokenID() == JavaTokenContext.STRING_LITERAL) {
                        ti = ti.getPrevious();
                        if(ti.getTokenID() == JavaTokenContext.WHITESPACE) ti = ti.getPrevious();
                        if(ti.getTokenID() == JavaTokenContext.EQ) {
                            ti = ti.getPrevious();
                            if(ti.getTokenID() == JavaTokenContext.WHITESPACE) ti = ti.getPrevious();
                            if(ti.getTokenID() == JavaTokenContext.IDENTIFIER) {
                                //probably the attribute name
                                String attrName = ti.getImage();
                                String value = exp.getTokenText(0);
                                resolveCompletionContext(nn.getName(), attrName, value);
                            }
                        }
                    }
                    
                }catch(BadLocationException ble) {
                    ble.printStackTrace();
                }
            }
            
            
        }
        
        private void resolveCompletionContext(String annotationName, String attrName, String attrValue) {
            if (WEBSERVICE_NN.equals(annotationName)) { // NOI18N
                completeWebService(attrName, attrValue);
            } else if (HANDLER_CHAIN_NN.equals(annotationName)) {
                completeHandlerChain(attrName, attrValue);
            }
        }
        
        
        private String getCompletedMemberName(JCExpression exp) {
            int parCnt = exp.getParameterCount();
            if (parCnt <= 0) {
                return null;
            }
            
            JCExpression lastMember = exp.getParameter(parCnt - 1);
            if (lastMember.getParameterCount() < 2) {
                return null;
            }
            JCExpression lastMemberParm1 = lastMember.getParameter(1);
            // trying to ensure that lastMember is something like name= and not something like name=foo
            if (lastMemberParm1.getExpID() != JCExpression.VARIABLE || lastMemberParm1.getTokenText(0).length() > 0) {
                return null;
            }
            
            String completedMember = lastMember.getParameter(0).getTokenText(0);
            
            return completedMember;
        }
        
        private void completeWebService(String attributeName, String completedAttributeValue) {
            if ("wsdlLocation".equals(attributeName)) { // NOI18N
                List results = results = new ArrayList();
                results.addAll(FileAttributeSupport.completionResults(endOffset, sup, getBaseDocument(), completedAttributeValue, true));
                result = new CompletionQuery.DefaultResult(component, "*", results, /*exp.getTokenOffset(0)*/0, 0); // NOI18N
            }
        }
        
        private void completeHandlerChain(String attributeName, String completedAttributeValue) {
            if ("file".equals(attributeName)) { // NOI18N
                List results = results = new ArrayList();
                results.addAll(FileAttributeSupport.completionResults(endOffset, sup, getBaseDocument(), completedAttributeValue, false));
                result = new CompletionQuery.DefaultResult(component, "*", results, /*exp.getTokenOffset(0)*/0, 0); // NOI18N
            }
        }
        
        private String getAnnotationTypeName(JCExpression exp) {
            assert exp != null;
            
            String result = null;
            
            if (exp.getParameterCount() < 1) {
                return result;
            }
            JCExpression variable = exp.getParameter(0);
            if (variable.getExpID() != JCExpression.VARIABLE) {
                return result;
            }
            // XXX this does not count with an annotation type written like "javax.persistence.Table"
            // should try to resolve the annotation type
            result = variable.getTokenText(0);
            return result;
        }
        
        
    }
    
}
