/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.websvc.wsdl;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.lang.ref.WeakReference;
import org.netbeans.api.xml.cookies.ValidateXMLCookie;
import org.openide.DialogDisplayer;
import org.openide.NotifyDescriptor;
import org.openide.nodes.Node;
import org.openide.loaders.MultiFileLoader;
import org.openide.loaders.DataObjectExistsException;
import org.openide.filesystems.FileObject;
import org.openide.util.NbBundle;

import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;

import org.netbeans.modules.j2ee.dd.api.web.DDProvider;
import org.netbeans.modules.j2ee.dd.api.common.ServiceRef;
import org.netbeans.modules.j2ee.dd.api.web.WebApp;
import org.netbeans.modules.j2ee.dd.api.common.RootInterface;
import org.netbeans.modules.j2ee.dd.api.common.CommonDDBean;
import org.netbeans.modules.j2ee.dd.api.common.VersionNotSupportedException;

import org.netbeans.modules.j2ee.deployment.devmodules.spi.J2eeModuleProvider;
import org.netbeans.modules.websvc.api.client.WebServicesClientSupport;
import org.netbeans.modules.websvc.core.ServiceInformation;
import org.netbeans.modules.websvc.wsdl.validator.ValidateXMLCookieImpl;

import org.xml.sax.InputSource;
import org.netbeans.modules.websvc.wsdl.xmlutils.XMLJ2eeDataObject;
import org.netbeans.modules.websvc.wsdl.xmlutils.SAXParseError;

/** Loader for Wsdl DataObjects.
 *
 * @author Peter Williams
 */
public class WsdlDataObject extends XMLJ2eeDataObject {
    private static final long serialVersionUID = 1871655333534008660L;
    public static final String WSDL_EXTENSION = "wsdl";    // NOI18N
    public static final String WSCOMPILE_CONFIG_EXTENSION = "xml";    // NOI18N
    
    private WeakReference portInformationHandlerRef = null;
    
    // If isClientWsdl is true, the WSDL file is in the WSDL folder of a web service
    // client enabled module and thus will have operations and UI exposed that affect
    // the service as it exists within the project.  E.g. deleting such a file will
    // actually remove the service from the project, not just delete the file on disk.
    private boolean clientResolved, isClientWsdl;
    
    /** Typical data object constructor.
     */
    public WsdlDataObject(FileObject pf, MultiFileLoader loader) throws DataObjectExistsException {
        super(pf, loader);
        init();
    }
    
    private void init() {
        ValidateXMLCookie validateCookie = new ValidateXMLCookieImpl(this);
        getCookieSet().add(validateCookie);
    }
    
    public boolean isClientWsdl() {
        if (!clientResolved) {
            initClientWsdl();
            clientResolved=true;
        }
        return isClientWsdl;
    }
    
    private void initClientWsdl() {
        isClientWsdl = false;
        FileObject wsdlFO = getPrimaryFile();
        
        // Check to make sure it has a non-null parent (can't be in WSDL folder if it does).
        FileObject parentFO = wsdlFO.getParent();
        if(parentFO != null) {
            // Does this module support web service clients?
            WebServicesClientSupport clientSupport = WebServicesClientSupport.getWebServicesClientSupport(wsdlFO);
            if(clientSupport != null) {
                // Is this file object in the WSDL folder of the client?
                FileObject wsdlFolder = clientSupport.getWsdlFolder();
                if(wsdlFolder != null && wsdlFolder.equals(parentFO)) {
                    // If we get here, the following conditions should be true:
                    //   The WSDL file is in a code module that supports webservice clients.
                    //   The WSDL file is in the proper WSDL folder of that module.
                    isClientWsdl = true;
                    
                    // With the addition of "Create service from wsdl" feature,
                    // wsdl files in this folder could also be services (if the project
                    // in question supports service at least) so we need additional
                    // heuristics for this case.
//                    WebServicesSupport serviceSupport = WebServicesSupport.getWebServicesSupport(wsdlFO);
//                    if(serviceSupport != null) {
//                        List serviceList = serviceSupport.getServices();
//                    }
                    
                    // for now, just check and see if there is a mapping file in the web-inf/meta-inf folder (parent)
                    FileObject ddFolder = wsdlFolder.getParent();
                    if(ddFolder != null) {
                        FileObject mappingFile = ddFolder.getFileObject(getName() + "-mapping", "xml"); // NOI18N
                        if(mappingFile != null) {
                            isClientWsdl = false;
                        }
                    }
                }
            }
        }
    }
    
    
    public boolean isRenameAllowed() {
        return !isClientWsdl();
    }
    
    public boolean isDeleteAllowed() {
        return true;
    }
    
    
    /** Create a node to represent the WSDL file. Overrides superclass method.
     * @return node delegate */
    protected Node createNodeDelegate() {
        return new WsdlDataNode(this);
    }
    
    /** Gets the Icon Base for node delegate when parser accepts the xml document as valid
     *
     * PENDING: move into node
     * @return Icon Base for node delegate
     */
    protected String getIconBaseForInvalidDocument() {
        return "org/netbeans/modules/websvc/wsdl/resources/wsdl_broken"; // NOI18N
    }
    
    /** Gets the Icon Base for node delegate when parser finds error(s) in xml document
     * @return Icon Base for node delegate
     *
     * PENDING: move into node
     */
    protected String getIconBaseForValidDocument() {
        return "org/netbeans/modules/websvc/wsdl/resources/wsdl16"; // NOI18N
    }
    
    protected SAXParseError updateNode(InputSource is) throws IOException {
        PortInformationHandler handler = new PortInformationHandler();
        
        // Invoke SAX parser on the WSDL to extract list of port bindings
        try {
            javax.xml.parsers.SAXParserFactory factory = javax.xml.parsers.SAXParserFactory.newInstance();
            factory.setNamespaceAware(true);
            javax.xml.parsers.SAXParser saxParser = factory.newSAXParser();
            saxParser.parse(is, handler);
            
            // Extract service names and porttypes
            synchronized (this) {
                portInformationHandlerRef = new WeakReference(handler);
            }
        } catch(javax.xml.parsers.ParserConfigurationException ex) {
            handler = null;
            return new SAXParseError(new org.xml.sax.SAXParseException(ex.getMessage(), null, ex));
        } catch(org.xml.sax.SAXParseException ex) {
            handler = null;
            return new SAXParseError(ex);
        } catch(org.xml.sax.SAXException ex) {
            handler = null;
            return new SAXParseError(new org.xml.sax.SAXParseException(ex.getMessage(), null, ex));
        } catch(FileNotFoundException ex) {
            // !PW Should never happen.
            handler = null;
            return new SAXParseError(new org.xml.sax.SAXParseException(ex.getMessage(), null, ex));
        } catch(IOException ex) {
            handler = null;
            throw ex;
        }
        
        return null;
    }
    
    protected void handleDelete() throws IOException {
        if(isClientWsdl()) {
            // If this is a "client WSDL", then remove the aspects of the service
            // that affect this project.  E.g. build script changes, service-ref, etc.
            //
            WebServicesClientSupport clientSupport = null;
            NotifyDescriptor descriptor;
            String message;
            
            // Find WebServicesClientSupport for this service
            clientSupport = WebServicesClientSupport.getWebServicesClientSupport(getPrimaryFile());
            if(clientSupport == null) {
                message = NbBundle.getMessage(WsdlDataObject.class, "ERR_CannotFindServiceSupport", // NOI18N
                        getName());
                NotifyDescriptor desc = new NotifyDescriptor.Message(message, NotifyDescriptor.Message.ERROR_MESSAGE);
                DialogDisplayer.getDefault().notify(desc);
                return;
            }
            
            // Delete action on a service:
            // 0. Confirmation Dialog.
            message = NbBundle.getMessage(WsdlDataObject.class, "MSG_QueryRemoveServiceClient", // NOI18N
                    getName());
            descriptor = new NotifyDescriptor.Confirmation(message, NotifyDescriptor.YES_NO_OPTION, NotifyDescriptor.QUESTION_MESSAGE);
            if(!DialogDisplayer.getDefault().notify(descriptor).equals(NotifyDescriptor.YES_OPTION)) {
                return;
            }
            
            // Get service names
            ServiceInformationImpl wsdlModel = new ServiceInformationImpl(this);
            String [] serviceNames = wsdlModel.getServiceNames();
            if(serviceNames == null || serviceNames.length == 0) {
                message = NbBundle.getMessage(WsdlDataObject.class, "ERR_CannotFindServiceName"); // NOI18N
                descriptor = new NotifyDescriptor.Message(message, NotifyDescriptor.ERROR_MESSAGE);
                DialogDisplayer.getDefault().notify(descriptor);
                return;
            }
            
            // 1. Remove service-ref entry(s) from web.xml
            
            // Make sure server specific support is available.
            FileObject wsdlFileObject = getPrimaryFile();
            Project project = FileOwnerQuery.getOwner(wsdlFileObject);
            J2eeModuleProvider j2eeMP = (J2eeModuleProvider) project.getLookup().lookup(J2eeModuleProvider.class);
            if(j2eeMP != null) {
                j2eeMP.getConfigSupport().ensureConfigurationReady();
                
                try {
                    // !PW FIXME DDProvider here is webapp specific -- will need updating
                    // for any other J2EE project type (or better, make DDProvider generic).
                    boolean ddNeedsSave = false;
                    
                    FileObject ddFO = clientSupport.getDeploymentDescriptor();
                    RootInterface rootDD = DDProvider.getDefault().getDDRoot(ddFO);
                    if(rootDD instanceof WebApp) {
                        for(int i = 0; i < serviceNames.length; i++) {
                            CommonDDBean bean = rootDD.findBeanByName("ServiceRef", "ServiceRefName", "service/" + serviceNames[i]); // NOI18N
                            if(bean instanceof ServiceRef) {
                                WebApp webApp = (WebApp) rootDD;
                                webApp.removeServiceRef((ServiceRef) bean);
                                ddNeedsSave = true;
                            }
                        }
                    }
                    
                    if(ddNeedsSave) {
                        // This also saves server specific configuration, if necessary.
                        rootDD.write(ddFO);
                    }
                } catch(VersionNotSupportedException ex) {
                    // This should not happen, but just in case.
                    message = NbBundle.getMessage(WsdlDataObject.class, "ERR_WrongJ2EEVersion"); // NOI18N
                    descriptor = new NotifyDescriptor.Message(message, NotifyDescriptor.ERROR_MESSAGE);
                    DialogDisplayer.getDefault().notify(descriptor);
                    return;
                } catch(IOException ex) {
                    // no file to edit, ignore this exception.
                }
            }
            
            // 2. Remove service from project.xml
            //    Side effect: Regenerate build-impl.xsl
            //    Optional - if last service, remove properties we generated.
            // !PW FIXME API only supports one service name.
            clientSupport.removeServiceClient(wsdlFileObject.getName());
            
            // 3. Remove -config file.  (For EA1, .wsc file is handled in step 4)
            FileObject parentFO = wsdlFileObject.getParent();
            if(parentFO != null && parentFO.isFolder()) {
                FileObject configFO = parentFO.getFileObject(wsdlFileObject.getName() + WsCompileConfigDataObject.WSCOMPILE_CONFIG_FILENAME_SUFFIX, WSCOMPILE_CONFIG_EXTENSION);
                if(configFO != null) {
                    // If IOException is thrown here, WSDL file will not be deleted.
                    // This is ok since clearly there is some sort of problem.
                    configFO.delete();
                }
            }
            
            // Final Note: Leave service registered w/ Registry because we do not
            // know what other projects might use it.
        }
        
        // 4. (Or step one if this is not a "client WSDL")
        //    Default handleDelete removes the WSDL file and any secondary files
        //    handled by the loader (config file in EA1 release).
        super.handleDelete();
    }
    
    public Node.Cookie getCookie(Class cls) {
        if(cls == ServiceInformation.class){
            return new ServiceInformationImpl(this);
        }
        
        return super.getCookie(cls);
    }
}
