/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.websvc.wsitconf.refactoring;

import java.io.IOException;
import java.text.MessageFormat;
import java.util.Collection;
import javax.jmi.reflect.RefObject;
import javax.xml.namespace.QName;
import org.netbeans.jmi.javamodel.Annotation;
import org.netbeans.jmi.javamodel.AttributeValue;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.Method;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.refactoring.api.AbstractRefactoring;
import org.netbeans.modules.refactoring.api.Problem;
import org.netbeans.modules.refactoring.spi.RefactoringElementsBag;
import org.netbeans.modules.refactoring.spi.RefactoringElementImplementation;
import org.netbeans.modules.websvc.jaxws.api.JAXWSSupport;
import org.netbeans.modules.websvc.wsitconf.util.JMIUtils;
import org.netbeans.modules.websvc.wsitconf.wsdlmodelext.WSITModelSupport;
import static org.netbeans.modules.websvc.wsitconf.wsdlmodelext.WSITModelSupport.CONFIG_WSDL_EXTENSION;
import static org.netbeans.modules.websvc.wsitconf.wsdlmodelext.WSITModelSupport.CONFIG_WSDL_SERVICE_PREFIX;
import org.netbeans.modules.xml.wsdl.model.Binding;
import org.netbeans.modules.xml.wsdl.model.BindingOperation;
import org.netbeans.modules.xml.wsdl.model.Definitions;
import org.netbeans.modules.xml.wsdl.model.Input;
import org.netbeans.modules.xml.wsdl.model.Message;
import org.netbeans.modules.xml.wsdl.model.Operation;
import org.netbeans.modules.xml.wsdl.model.Output;
import org.netbeans.modules.xml.wsdl.model.PortType;
import org.netbeans.modules.xml.wsdl.model.WSDLModel;
import org.openide.ErrorManager;
import org.openide.filesystems.FileLock;
import org.openide.filesystems.FileObject;
import org.openide.util.NbBundle;

/**
 *
 * @author Martin Grebac
 */
public final class WSITXmlRenameRefactoring {
    
    /**
     * Name of an annotation that represents web service.
     */
    protected static final String WS_ANNOTATION = "WebService";
    /**
     * Name of an annotation element that represents wsdl location.
     */
    protected static final String WSDL_LOCATION_ELEMENT = "wsdlLocation";

    private static final ErrorManager err = ErrorManager.getDefault().getInstance("org.netbeans.modules.websvc.wsitconf.refactoring");   // NOI18N
    
    public WSITXmlRenameRefactoring() { }
    
    public Problem precheck() {
        return null;
    }
    
    /** Find usages in wsit deployment descriptors: wsit-*.xml
     */
    public Problem prepare(AbstractRefactoring refactoring, RefObject refObject, String newName, RefactoringElementsBag refactoringElements) {
                
        if (refObject instanceof JavaClass) {
            JavaClass javaClass = (JavaClass) refObject;
            if (isWebSvcFromWsdl(javaClass)) {
                return null;
            }
            Resource r = javaClass.getResource();
            FileObject fo = JavaModel.getFileObject(r);
            JAXWSSupport supp = JAXWSSupport.getJAXWSSupport(fo);
            WSDLModel model = null;
            try {
                model = WSITModelSupport.getModelForServiceFromJava(javaClass, supp, false, null);
            } catch (IOException ex) {
                ex.printStackTrace();
            } catch (Exception ex) {
                ex.printStackTrace();
            }
            if (model == null){
                return null;
            }

            RefactoringElementImplementation refactoringElem =
                    new WSITXmlClassRenameRefactoringElement(javaClass.getName(), newName, model);
            refactoringElements.add(refactoring, refactoringElem);
            
        } else if (refObject instanceof Method) {
            Method mtd = (Method)refObject;
            JavaClass javaClass = JMIUtils.getDeclaringClass(mtd);
            if (isWebSvcFromWsdl(javaClass)) {
                return null;
            }
            Resource r = mtd.getResource();
            FileObject fo = JavaModel.getFileObject(r);
            JAXWSSupport supp = JAXWSSupport.getJAXWSSupport(fo);
            WSDLModel model = null;
            try {
                model = WSITModelSupport.getModelForServiceFromJava(javaClass, supp, false, null);
            } catch (IOException ex) {
                ex.printStackTrace();
            } catch (Exception ex) {
                ex.printStackTrace();
            }
            if (model == null){
                return null;
            }

            RefactoringElementImplementation refactoringElem =
                    new WSITXmlMethodRenameRefactoringElement(mtd.getName(), newName, model);
            refactoringElements.add(refactoring, refactoringElem);
        }
        
        return null;
    }

    Problem preCheck(RefObject refO) {
        return null;
    }

   /**
     * Rename refactoring element for wsit-*.xml
     */
    private static class WSITXmlClassRenameRefactoringElement extends AbstractRenameRefactoringElement {
               
        private String oldConfigName, newConfigName;
                
        /**
         * Creates a new instance of WSITXmlClassRenameRefactoringElement
         * 
         * @param oldName the fully qualified old name of the implementation class
         * @param newName the fully qualified new name of the implementation class
         */
        public WSITXmlClassRenameRefactoringElement(String oldName, String newName, WSDLModel model) {
            this.oldName = oldName;
            this.newName = newName;
            this.oldConfigName = CONFIG_WSDL_SERVICE_PREFIX + oldName;
            String pkg = oldName.substring(0, oldName.lastIndexOf('.')+1);
            this.newConfigName = CONFIG_WSDL_SERVICE_PREFIX + pkg + newName;
            this.model = model;
            this.parentFile = (FileObject) model.getModelSource().getLookup().lookup(FileObject.class);
        }
        
        public void performExternalChange() {
            FileLock lock;
            try {
                lock = parentFile.lock();
                parentFile.rename(lock, newConfigName, CONFIG_WSDL_EXTENSION);
                lock.releaseLock();
            } catch (IOException ex) {
                ex.printStackTrace();
            }
        }

        public void undoExternalChange() {
            FileLock lock;
            try {
                lock = parentFile.lock();
                parentFile.rename(lock, oldConfigName, CONFIG_WSDL_EXTENSION);
                lock.releaseLock();
            } catch (IOException ex) {
                ex.printStackTrace();
            }
        }
        
        public void performChange() {
            JavaMetamodel.getManager().registerExtChange(this);
        }
        
        /**
         * Returns text describing the refactoring formatted for display (using HTML tags).
         * @return Formatted text.
         */
        public String getDisplayText() {
            Object[] args = new Object [] {oldConfigName, newConfigName};
            return MessageFormat.format(NbBundle.getMessage(WSITXmlClassRenameRefactoringElement.class, "TXT_WsitXmlClassRename"), args);
        }
    }
    
   /**
     * Rename refactoring element for wsit-*.xml
     */
    private static class WSITXmlMethodRenameRefactoringElement extends AbstractRenameRefactoringElement {

        /**
         * Creates a new instance of WSITXmlMethodRenameRefactoringElement
         * 
         * @param oldName old name of operation
         * @param newName new name of operation
         */
        public WSITXmlMethodRenameRefactoringElement(String oldName, String newName, WSDLModel model) {
            this.oldName = oldName;
            this.newName = newName;
            this.model = model;
            this.parentFile = (FileObject) model.getModelSource().getLookup().lookup(FileObject.class);
        }
        
        public void performExternalChange() {
            Definitions d = model.getDefinitions();
            Binding b = (Binding) d.getBindings().toArray()[0];
            Collection<BindingOperation> bOperations = b.getBindingOperations();
            PortType portType = (PortType) d.getPortTypes().toArray()[0];
            Collection<Operation> operations = portType.getOperations();
            model.startTransaction();

            for (BindingOperation bOperation : bOperations) {
                if (oldName.equals(bOperation.getName())) {
                    bOperation.setName(newName);
                }
            }
            
            for (Operation o : operations) {
                if (oldName.equals(o.getName())) {
                    o.setName(newName);
                    Input i = o.getInput();
                    if (i != null) {
                        QName qname = i.getMessage().getQName();
                        Message msg = model.findComponentByName(qname, Message.class);
                        String oMsgName = msg.getName();
                        if (oMsgName != null) {
                            String nMsgName = oMsgName.replaceAll(oldName, newName);
                            msg.setName(nMsgName);
                        }
                        i.setMessage(i.createReferenceTo(msg, Message.class));
                    }
                    Output out = o.getOutput();
                    if (out != null) {
                        QName qname = out.getMessage().getQName();
                        Message msg = model.findComponentByName(qname, Message.class);
                        String oMsgName = msg.getName();
                        if (oMsgName != null) {
                            String nMsgName = oMsgName.replaceAll(oldName, newName);
                            msg.setName(nMsgName);
                        }
                        out.setMessage(out.createReferenceTo(msg, Message.class));
                    }
                }
            }
            model.endTransaction();
        }

        public void undoExternalChange() {
            Definitions d = model.getDefinitions();
            Binding b = (Binding) d.getBindings().toArray()[0];
            Collection<BindingOperation> bOperations = b.getBindingOperations();
            PortType portType = (PortType) d.getPortTypes().toArray()[0];
            Collection<Operation> operations = portType.getOperations();
            model.startTransaction();
            for (BindingOperation bOperation : bOperations) {
                if (newName.equals(bOperation.getName())) {
                    bOperation.setName(oldName);
                }
            }
            for (Operation o : operations) {
                if (newName.equals(o.getName())) {
                    o.setName(oldName);
                    Input i = o.getInput();
                    if (i != null) {
                        QName qname = i.getMessage().getQName();
                        Message msg = model.findComponentByName(qname, Message.class);
                        String oMsgName = msg.getName();
                        if (oMsgName != null) {
                            String nMsgName = oMsgName.replaceAll(newName, oldName);
                            msg.setName(nMsgName);
                        }
                        i.setMessage(i.createReferenceTo(msg, Message.class));
                    }
                    Output out = o.getOutput();
                    if (out != null) {
                        QName qname = out.getMessage().getQName();
                        Message msg = model.findComponentByName(qname, Message.class);
                        String oMsgName = msg.getName();
                        if (oMsgName != null) {
                            String nMsgName = oMsgName.replaceAll(newName, oldName);
                            msg.setName(nMsgName);
                        }
                        out.setMessage(out.createReferenceTo(msg, Message.class));
                    }
                }
            }
            model.endTransaction();
        }
        
        public void performChange() {
            JavaMetamodel.getManager().registerExtChange(this);
        }
        
        /**
         * Returns text describing the refactoring formatted for display (using HTML tags).
         * @return Formatted text.
         */
        public String getDisplayText() {
            Object[] args = new Object [] {oldName, newName};
            return MessageFormat.format(NbBundle.getMessage(WSITXmlMethodRenameRefactoringElement.class, "TXT_WsitXmlMethodRename"), args);
        }
    }
    
    /**
     * @return true if the given javaClass represents a web service from wsdl.
     */
    protected boolean isWebSvcFromWsdl(JavaClass javaClass){
        for (Object elem : javaClass.getAnnotations()) {
            Annotation ann = (Annotation) elem;
            if (ann.getTypeName() == null){
                continue;
            }
            if (ann.getTypeName().getName().equals(WS_ANNOTATION)){
                for (Object elem2 : ann.getAttributeValues()) {
                    AttributeValue value = (AttributeValue) elem2;
                    if (value.getName().equals(WSDL_LOCATION_ELEMENT)){
                        return true;
                    }
                }
            }
        }
        return false;
    }    
}
