/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.websvc.wsitconf.util;
import java.io.IOException;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import javax.xml.namespace.QName;

import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.netbeans.modules.websvc.jaxwsruntimemodel.JavaWsdlMapper;
import org.netbeans.modules.xml.wsdl.model.Binding;
import org.netbeans.modules.xml.wsdl.model.BindingOperation;
import org.netbeans.modules.xml.wsdl.model.Definitions;
import org.netbeans.modules.xml.wsdl.model.PortType;
import org.netbeans.modules.xml.wsdl.model.WSDLComponentFactory;
import org.netbeans.modules.xml.wsdl.model.WSDLModel;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileSystem;
import org.openide.filesystems.Repository;
import org.openide.loaders.DataFolder;
import org.openide.loaders.DataObject;
import org.openide.nodes.Node;

/**
 * Helper for JMI-related code operations
 *
 * @author Martin Adamek
 * @author Pavel Fiala
 */
public class JMIUtils {

    private static final String CLASS_TEMPLATE = "Templates/Classes/Class.java"; // NOI18N
    
    private JMIUtils() {}

    public static Method getMethod(Node node, String methodName) {
        if (methodName == null) {
            return null;
        }
        JavaClass jc = getJavaClassFromNode(node);
        Method[] methods = getMethods(jc);
        for (Method method : methods) {
            if (methodName.equals(method.getName())) {
                return method;
            }
        }
        return null;
    }
    
    public static Collection<BindingOperation> refreshOperations(Binding binding, JavaClass jc) {
        
        Collection<BindingOperation> operations = binding.getBindingOperations();
        if ((binding == null) || (jc == null)) {
            return operations;
        }
        
        WSDLModel model = binding.getModel();                
        boolean isTransaction = model.isIntransaction();
        if (!isTransaction) {
            model.startTransaction();
        }
        
        WSDLComponentFactory wcf = model.getFactory();
        Definitions d = (Definitions) binding.getParent();
                
        QName portTypeQName = binding.getType().getQName();
        PortType portType = null;
        
        Collection<PortType> portTypes = d.getPortTypes();
        Iterator<PortType> i = portTypes.iterator();
        while (i.hasNext()) {
            PortType pt = i.next();
            if (pt != null) {
                if (portTypeQName.getLocalPart().equals(pt.getName())) {
                    portType = pt;
                    break;
                }
            }
        }
        // create operations and add them to the binding element
        List<String> bindingOperationNames = JavaWsdlMapper.getOperationNames(jc);
        for (String name : bindingOperationNames) {
            if (!isOperationInList(name, operations)) {
                org.netbeans.modules.xml.wsdl.model.BindingOperation bindingOperation = wcf.createBindingOperation();
                bindingOperation.setName(name);
                binding.addBindingOperation(bindingOperation);

                // add input/output messages
                org.netbeans.modules.xml.wsdl.model.Message inputMsg = wcf.createMessage();
                inputMsg.setName(name);
                d.addMessage(inputMsg);

                org.netbeans.modules.xml.wsdl.model.Message outMsg = wcf.createMessage();
                outMsg.setName(name + "Response");                  //NOI18N
                d.addMessage(outMsg);

                org.netbeans.modules.xml.wsdl.model.RequestResponseOperation oper = wcf.createRequestResponseOperation();
                oper.setName(name);
                portType.addOperation(oper);

                org.netbeans.modules.xml.wsdl.model.Input input = wcf.createInput();
                oper.setInput(input);
                input.setMessage(input.createReferenceTo(inputMsg, org.netbeans.modules.xml.wsdl.model.Message.class));

                org.netbeans.modules.xml.wsdl.model.Output out = wcf.createOutput();
                oper.setOutput(out);
                out.setMessage(out.createReferenceTo(outMsg, org.netbeans.modules.xml.wsdl.model.Message.class));

                org.netbeans.modules.xml.wsdl.model.BindingOutput bindingOutput = wcf.createBindingOutput();
                bindingOperation.setBindingOutput(bindingOutput);
                org.netbeans.modules.xml.wsdl.model.BindingInput bindingInput = wcf.createBindingInput();
                bindingOperation.setBindingInput(bindingInput);

                //add faults
                List<String> operationFaults = JavaWsdlMapper.getOperationFaults(jc, name);
                for (String fault : operationFaults) {
                    org.netbeans.modules.xml.wsdl.model.BindingFault bindingFault = wcf.createBindingFault();
                    bindingOperation.addBindingFault(bindingFault);
                }
            }
        }
        
        if (!isTransaction) {
            model.endTransaction();
        }
        
        return binding.getBindingOperations();
    }

    private static boolean isOperationInList(String operName, Collection<BindingOperation> operations) {
        Iterator<BindingOperation> i = operations.iterator();
        while (i.hasNext()) {
            BindingOperation bo = i.next();
            if ((bo != null) && (operName.equals(bo.getName()))) {
                return true;
            }
        }
        return false;
    }
    
    public static JavaClass getDeclaringClass(Feature feature) {
        if (feature == null) {
            return null;
        }
        return feature instanceof JavaClass ? (JavaClass) feature : (JavaClass) feature.getDeclaringClass();
    }

    public static JavaClass getJavaClassFromNode(Node node) {
        beginJmiTransaction();
        JavaClass jc = null;
        try {
            DataObject dobj = (DataObject) node.getCookie(DataObject.class);
            if (dobj != null) {
                Resource res = JavaModel.getResource(dobj.getPrimaryFile());
                if (res != null) {
                    JavaModel.setClassPath(res);
                    List/*<JavaClass>*/ classes = res.getClassifiers();
                    if (classes.size() == 1) {
                        jc = (JavaClass)classes.get(0);
                    }
                }
            } else {
                Feature feature = (Feature)node.getLookup().lookup(Feature.class);
                if (feature != null) {
                    jc = feature instanceof JavaClass ? (JavaClass)feature : (JavaClass) feature.getDeclaringClass();
                }
            }
            return jc;
        } finally {
            endJmiTransaction();
        }
    }

    public static void beginJmiTransaction(boolean writeAccess) {
        JavaModel.getJavaRepository().beginTrans(writeAccess);
    }

    public static void beginJmiTransaction() {
        beginJmiTransaction(false);
    }

    public static void endJmiTransaction(boolean rollback) {
        JavaModel.getJavaRepository().endTrans(rollback);
    }

    public static void endJmiTransaction() {
        endJmiTransaction(false);
    }

    public static JavaClass findClass(String className) {
        JavaClass result = (JavaClass) resolveType(className);
        return result instanceof UnresolvedClass ? null : result;
    }

    public static Type resolveType(String typeName) {
        Type type = JavaModel.getDefaultExtent().getType().resolve(typeName);
        if (type instanceof UnresolvedClass) {
            Type basicType = JavaModel.getDefaultExtent().getType().resolve("java.lang." + typeName);  // NOI18N;
            if (!(basicType instanceof UnresolvedClass)) {
                return basicType;
            }
        }
        return type;
    }

    public static Method[] getMethods(JavaClass jc) {
        List/*<Method>*/ result = new LinkedList();
        if (jc != null) {
            List features = jc.getFeatures();
            for (Iterator it = features.iterator(); it.hasNext();) {
                Object o =  it.next();
                if (o instanceof Method) {
                    result.add(o);
                }
            }
        }
        return (Method[]) result.toArray(new Method[result.size()]);
    }

    private static boolean isEmptyOrNull(Collection collection){
        return collection == null ? true : collection.isEmpty();
    }   

    /**
     * Creates a new ordinary Java class. Uses the
     * same file template as a regular Java class (see {@link #CLASS_TEMPLATE}). 
     * The class is not created under an MDR transaction, a transaction is only 
     * entered to retrieve a {@link JavaClass} instance for the new class.
     *
     * <p>This method does not need to be called in a MDR transaction, but if
     * it is, that transaction needs to be enclosed in an 
     * {@link org.openide.filesystems.FileSystem#runAtomicAction atomic action}, 
     * otherwise a deadlock can occur, like in issue 77500.</p>
     */
    public static JavaClass createClass(FileObject targetFolder, String targetName) throws IOException{
        DataObject dobj = createClassFromTemplate(CLASS_TEMPLATE, targetFolder, targetName);
        return getJavaClassFromDataObject(dobj);
    }

    private static DataObject createClassFromTemplate(String template, FileObject targetFolder, String targetName) throws IOException {
        if (null == targetFolder || null == targetName || "".equals(targetName.trim())) { // NOI18N
            throw new IllegalArgumentException("Target folder and target name must be given."); // NOI18N
        }        
        FileSystem dfs = Repository.getDefault().getDefaultFileSystem();
        FileObject fo = dfs.findResource(template);
        DataObject dob = DataObject.find(fo);
        DataFolder dataFolder = DataFolder.findFolder(targetFolder);
        return dob.createFromTemplate(dataFolder, targetName);
    }
    
    /**
     * Gets JavaClass from given DataObject.
     * @param dobj must not be null.
     * @return JavaClass or null if couldn't get JavaClass
     *  from given dobj.
     */
    public static JavaClass getJavaClassFromDataObject(DataObject dobj) {
        if (null == dobj){
            throw new NullPointerException("DataObject must be given.");
        }
        JavaModel.getJavaRepository().beginTrans(false);
        try {
            Resource res = JavaModel.getResource(dobj.getPrimaryFile());
            if (res != null) {
                JavaModel.setClassPath(res);
                List/*<JavaClass>*/ classes = res.getClassifiers();
                if (classes.size() == 1) {
                    return (JavaClass)classes.get(0);
                }
            }
        } finally {
            JavaModel.getJavaRepository().endTrans(false);
        }
        return null;
    }
    
    public static void addInterface(JavaClass javaClass, String interfaceName) {
        List interfaceNames = javaClass.getInterfaceNames();
        for (Iterator it = interfaceNames.iterator(); it.hasNext();) {
            if (interfaceName.equals(((MultipartId) it.next()).getElement().getName())) {
                return;
            }
        }
        interfaceNames.add(createMultipartId(javaClass, interfaceName));
    }

    public static MultipartId createMultipartId(Feature feature, String name) {
        return getJavaModelPackage(feature).getMultipartId().createMultipartId(name, null, null);
    }

    private static JavaModelPackage getJavaModelPackage(Feature feature) {
        return feature != null ? (JavaModelPackage) feature.refImmediatePackage() : JavaModel.getDefaultExtent();
    }


    private static JavaModelPackage getJavaModelPackage(Element element) {
        return element != null ? (JavaModelPackage) element.refImmediatePackage() : JavaModel.getDefaultExtent();
    }
    
    public static Annotation createAnnotation(Element element, String annotationClassName, List attributeValues) {
        JavaClass annotationClass = (JavaClass) JavaModel.getDefaultExtent().getType().resolve(annotationClassName);
        return getJavaModelPackage(element).getAnnotation().createAnnotation(
                JavaModelUtil.resolveImportsForClass(element, annotationClass),
                attributeValues
                );
    }

    /**
     * Always use this method for adding exceptions to methods.
     * There is a bug in JMI, so method.getExceptions().add() is not working.
     * You have to use method.getExceptionNames().add()
     *
     * @param method method to add exception to
     * @param exceptionName name of exception to add
     */
    public static void addException(Method method, String exceptionName) {
        method.getExceptionNames().add(createMultipartId(method, exceptionName));
    }

    /**
     * Add more exceptions to method.
     *
     * @param method method to add exceptions to
     * @param exceptionNames List of String values - exceptions names
     */
    public static void addExceptions(Method method, List/*<String>*/ exceptionNames) {
        for (Iterator it = exceptionNames.iterator(); it.hasNext();) {
            addException(method, (String) it.next());
        }
    }

    public static Method createMethod(Element element, String name, int modifiers, String type) {
        return getJavaModelPackage(element).getMethod().createMethod(
                name, null, modifiers, null, null, null, null, null, null, null, getTypeReference(element, type), 0
                );
    }
    
    private static TypeReference getTypeReference(Element element, String name) {
        Type resolvedType = JavaModel.getDefaultExtent().getType().resolve(name);
        if (resolvedType instanceof PrimitiveType) {
            return getJavaModelPackage(element).getMultipartId().createMultipartId(name, null, null);
        } else {
            return JavaModelUtil.resolveImportsForType(element, resolvedType);
        }
    }

    public static Parameter createParameter(Element element, String name, String type) {
        return getJavaModelPackage(element).getParameter().createParameter(
                name, null, false, getTypeReference(element, type), 0, false
                );
    }
    
    public static AttributeValue createAttributeValue(Element element, String name, String value) {
        return getJavaModelPackage(element).getAttributeValue().createAttributeValue(
                name, getJavaModelPackage(element).getStringLiteral().createStringLiteral(value)
                );
    }
    
    public static AttributeValue createAttributeValue(Element element, String name, boolean value) {
        return getJavaModelPackage(element).getAttributeValue().createAttributeValue(
                name, getJavaModelPackage(element).getBooleanLiteral().createBooleanLiteral(value)
                );
    }
    
    public static AttributeValue createAttributeValue(Element element, String name, List arrayItems) {
        return getJavaModelPackage(element).getAttributeValue().createAttributeValue(
                name, getJavaModelPackage(element).getArrayInitialization().createArrayInitialization(arrayItems)
                );
    }
    
    public static AttributeValue createAttributeValue(Element element, String name, String type, String variable) {
        JavaClass typeClass = (JavaClass) JavaModel.getDefaultExtent().getType().resolve(type);
        String typeSimpleName = JavaModelUtil.resolveImportsForType(element, typeClass).getName();
        return getJavaModelPackage(element).getAttributeValue().createAttributeValue(
                name,
                getJavaModelPackage(element).getVariableAccess().createVariableAccess(typeSimpleName + (variable == null ? null : "." + variable), null, false)
                );
    }

}
