/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.xml.refactoring;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.netbeans.modules.xml.refactoring.impl.RefactoringUtil;
import org.netbeans.modules.xml.refactoring.spi.ChangeExecutor;
import org.netbeans.modules.xml.xam.Component;
import org.netbeans.modules.xml.xam.EmbeddableRoot;
import org.netbeans.modules.xml.xam.Model;
import org.netbeans.modules.xml.xam.Referenceable;
import org.netbeans.modules.xml.xam.dom.DocumentComponent;
import org.netbeans.modules.xml.xam.dom.DocumentModel;
import org.openide.filesystems.FileObject;

/**
 *
 * @author Nam Nguyen
 */
public abstract class RefactorRequest {
    private Set<Component> scope;
    private ChangeExecutor changeExecutor;
    private UsageSet usages;
    private boolean autosave = true;
    private List<ErrorItem> errors;

    public RefactorRequest() {
        this(null);
    }
    
    /**
     * @param scope list of search root constitute the refactoring scope.
     */
    public RefactorRequest(Set<Component> scope) {
        this.scope = scope;
        errors = new ArrayList<ErrorItem>();
    }
    
    public Set<Component> getScope() {
        return scope;
    }
    
    public void setScope(Set<Component> searchRoots) {
        scope = searchRoots;
    }

    public void setScopeLocal() {
        if (getTarget() instanceof DocumentModel) {
            scope = new HashSet<Component>();
            scope.add(((DocumentModel)getTarget()).getRootComponent());
        } else if (getTarget() instanceof Component) {
            scope = Collections.singleton(getRootOf((Component)getTarget()));
        }
        setAutosave(false);
    }
    
    public boolean isScopeLocal() {
        if (getScope() == null || getScope().size() > 1) {
            return false;
        }
        return getScope().iterator().next() == getRootOf(getTargetComponent());
    }
    
    public String getDescription() {
        return RefactoringUtil.getDescription(this);
    }
    
    /**
     * Returns target component, i.e., change component, of this refactoring request.
     */
    public abstract Referenceable getTarget();

    public abstract String getTargetName();
    
    public Model getTargetModel() {
        return getModel(getTarget());
    }
    
    public Component getTargetComponent() {
        return getComponent(getTarget());
    }
    
    public static Model getModel(Referenceable ref) {
        if (ref instanceof Model) {
            return (Model) ref;
        } else if (ref instanceof Component) {
            return ((Component)ref).getModel();
        } else {
            return null;
        }
    }
    
    public static Component getComponent(Referenceable ref) {
        if (ref instanceof Component) {
            return (Component) ref;
        } else if (ref instanceof DocumentModel) {
            return ((DocumentModel)ref).getRootComponent();
        } else {
            throw new IllegalArgumentException("Invalid target type "+ref.getClass().getName());
        }
    }
    
    public FileObject getFileObject() {
        FileObject fo = (FileObject) getTargetModel().getModelSource().getLookup().lookup(FileObject.class);
        assert fo != null : "Failed to lookup for file object in model source"; //NOI18N
        return fo;
    }
    
    /**
     * Returns the executor applying the change or null if none found or 
     * RefactoringManager#precheck or RefactoringManager#process has not been called
     * on this request.
     */
    public ChangeExecutor getChangeExecutor() {
        if (changeExecutor == null) {
            for (ChangeExecutor ce : RefactoringManager.getInstance().getExecutors()) {
                if (ce.canChange(getType(), getTarget())) {
                    changeExecutor = ce;
                    break;
                }
            }
        }
        return changeExecutor;
    }
    
    void setChangeExecutor(ChangeExecutor executor) {
        changeExecutor = executor;
    }
    
    /**
     * Returns the usages included in this refactoring request.
     */
    public UsageSet getUsages() {
        return usages;
    }
    
    public void setUsages(UsageSet usageSet) {
        usages = usageSet;
    }
    
    /**
     * Whether the refactoring changes should be automatically persisted.
     */
    public boolean getAutosave() {
        return autosave;
    }
    
    /**
     * Set whether the refactoring changes should be automatically persist.
     */
    public void setAutosave(boolean v) {
        autosave = v;
    }
    
    /**
     * Returns type of this refactoring request.
     */
    public abstract <T extends RefactorRequest> Class<T> getType();
    
    /**
     * Check if requested change has been performed on the target.
     */
    public abstract boolean confirmChangePerformed();
    
    /**
     * Returns errors on the change itself.  This does not include error from 
     * usages.
     */
    public List<ErrorItem> getChangeErrors() {
        return errors;
    }
    
    /**
     * Returns all the collected errors.
     */
    public List<ErrorItem> getErrors() {
        List ret = new ArrayList<ErrorItem>();
        ret.addAll(errors);
        if (usages != null) {
            for (UsageGroup u : usages.getUsages()) {
                if (u.getErrors() != null) {
                    ret.addAll(u.getErrors());
                }
            } 
        }
        return ret;
    }
    
    /**
     * Report an error happen during pre-refactoring check or during refactoring
     * processing.
     */
    public void addError(ErrorItem error) {
        errors.add(error);
    }

    public void addError(String message) {
        errors.add(new ErrorItem(getTarget(), message));
    }
    
    public void precheckChange() {
        clear();
        RefactoringUtil.precheckTarget(this);
    }
    
    public void precheckUsages() {
        RefactoringUtil.precheckUsageModels(this);
    }
    
    public boolean hasFatalErrors() {
        for (ErrorItem e : getErrors()) {
            if (e.getLevel() == ErrorItem.Level.FATAL) {
                return true;
            }
        }
        return false;
    }
    
    protected void clear() {
        errors = new ArrayList<ErrorItem>();
        usages = null;
    }

    public static Component getEffectiveParent(Component component) {
        if (component instanceof  EmbeddableRoot) {
            return ((EmbeddableRoot) component).getForeignParent();
        } else {
            return component.getParent();
        }
    }

    public static Component getRootOf(Referenceable referenceable) {
        return getRootOf(referenceable);
    }
    
    public static Component getRootOf(Component component) {
        Component root = (Component) component;
        while (root != null) {
            Component parent = getEffectiveParent(root);
            if (parent == null) {
                break;
            }
            root = parent;
        }
        return root;
    }

    public Set<Model> getDirtyModels() {
        HashSet<Model> dirties = new HashSet<Model>();
        if (RefactoringUtil.isDirty(getTargetModel())) {
            dirties.add(getTargetModel());
        }
        if (getUsages() != null) {
            for (Model model : getUsages().getModels()) {
                if (RefactoringUtil.isDirty(model)) {
                    dirties.add(model);
                }
            }
        }
        return dirties;
    }        
    
}
